package Application::Model::Page::Mobile;

=encoding UTF-8

=cut

=head1 DESCRIPTION

Базовый класс для всех мобильных пейджей.

=cut

use qbit;

use base qw(Application::Model::Page);

use Exception::Validator::Fields;

use PiConstants qw($CONTEXT_TARGET_TYPE);

consume qw(
  Application::Model::Role::Has::UnmoderatedRtbAuction
  Application::Model::Role::Has::Filters
  Application::Model::Role::Has::Tier
  Application::Model::Role::Page::Has::Tags
  Application::Model::Role::Page::Has::SingleState
  Application::Model::Role::Has::Page::Patch
  );

__PACKAGE__->abstract_methods(qw(get_block_model_names get_product_name _add _trim_opts));

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        bk_language  => 'Application::Model::BKLanguage',
        quality_coef => 'Application::Model::QualityCoef',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        map {$self->get_description_right($_)}
          qw(
          view_field__bk_languages
          edit_field__bk_languages

          view_field__block_title
          edit_field__block_title

          view_field__false_click
          edit_field__false_click

          view_field__fast_context
          edit_field__fast_context

          view_field__only_picture
          edit_field__only_picture

          view_field__performance_tgo_disable
          edit_field__performance_tgo_disable
          )
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},

        behavioral_targeting => {
            db         => TRUE,
            type       => 'boolean',
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
            adjust_type => 'str',
        },
        bk_languages => {
            depends_on   => [qw(page_id)],
            label        => d_gettext('Banner language ids'),
            check_rights => $self->get_right('view_field__bk_languages'),
            get          => sub {
                my $bk_langs = $_[0]->{'bk_languages'}->{$_[1]->{'page_id'}};
                [sort keys %$bk_langs];
            },
            need_check => {
                type     => 'array',
                optional => TRUE,
                all      => {type => 'scalar'},
            },
            type     => 'array',
            sub_type => 'number',
            api      => 1
        },
        bk_languages_name => {
            depends_on   => [qw(page_id)],
            label        => d_gettext('Banner languages'),
            check_rights => $self->get_right('view_field__bk_languages'),
            get          => sub {
                my $bk_langs = $_[0]->{'bk_languages'}->{$_[1]->{'page_id'}};
                [map {$bk_langs->{$_}} sort keys %$bk_langs];
            },
            type     => 'array',
            sub_type => 'string',
        },
        block_title => {
            db           => TRUE,
            check_rights => $self->get_right('view_field__block_title'),
            type         => 'string',
            need_check   => {
                len_max  => 255,
                optional => TRUE,
            },
            need_trim => TRUE,
        },
        block_types => {
            depends_on => ['page_id'],
            get        => sub {
                return $_[0]->model->get_block_types(%{$_[1]});
            },
            type     => 'array',
            sub_type => 'complex',
        },
        caption => {
            db         => TRUE,
            label      => d_gettext('Block caption'),
            type       => 'string',
            need_check => {len_min => 1, len_max => 255,},
            need_trim  => TRUE,
            api        => 1,
        },
        false_click => {
            db           => TRUE,
            type         => 'boolean',
            check_rights => $self->get_right('view_field__false_click'),
            need_check   => {
                type     => 'boolean',
                optional => TRUE,
            },
            adjust_type => 'str',
        },
        family_filter => {
            db         => TRUE,
            type       => 'boolean',
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
            adjust_type => 'str',
        },
        fast_context => {
            db         => TRUE,
            type       => 'boolean',
            need_check => {
                type     => 'boolean',
                optional => TRUE,

            },
            adjust_type => 'str',
            api         => 1
        },
        only_picture => {
            db         => TRUE,
            type       => 'boolean',
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
            adjust_type => 'str',
            api         => 1
        },
        product_type => {
            get => sub {
                return 'mobile';
            },
            type => 'string',
        },
        view_images => {
            db         => TRUE,
            type       => 'boolean',
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
            adjust_type => 'str',
            api         => 1
        },
        performance_tgo_disable => {
            label      => d_gettext('Performance TGO Disable'),
            db         => 1,
            type       => 'boolean',
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
            api         => 1,
            adjust_type => 'str',
        },
    };
}

sub get_available_fields_depends {
    [qw(multistate)];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub get_add_fields {
    my ($self, $obj) = @_;

    my %fields = map {$_ => TRUE} qw(caption);

    foreach (qw(comment performance_tgo_disable)) {
        $fields{$_} = TRUE if $self->check_short_rights("edit_field__$_");
    }

    return \%fields;
}

sub collect_editable_fields {
    my ($self, $object) = @_;

    return {};
}

sub check_blocks_with_native {
    my ($self, $opts) = @_;

    my ($block_model)      = $self->get_block_model_names()->[0];
    my $mobile_app_rtb     = $self->$block_model;
    my $page_id_field_name = $mobile_app_rtb->get_page_id_field_name();
    my $rtb_table          = $mobile_app_rtb->partner_db_table();
    my $dsp_table          = $self->partner_db->media_sizes;
    my $query              = $self->partner_db->query->select(
        table  => $rtb_table,
        fields => [$page_id_field_name, 'id'],
        filter => [
            'AND',
            [
                [$page_id_field_name => '=', \$$opts{$self->get_page_id_field_name()}],
                ['multistate' => '=' => \$mobile_app_rtb->get_multistates_by_filter('not deleted')],
                [
                    'OR',
                    [
                        ['direct_block', '=', \'native'],
                        ['media_block',  '=', \'native'],
                        [\'native', '=', {'type' => $dsp_table}]
                    ],
                ],
                (
                    $mobile_app_rtb->is_block_table_with_multiple_models()
                    ? ['model', '=', \$mobile_app_rtb->accessor()]
                    : ()
                )
            ]
        ]
      )->join(
        table  => $dsp_table,
        fields => [],
        join_on =>
          ['AND', [['page_id', '=', {$page_id_field_name => $rtb_table}], ['block_id', '=', {'id' => $rtb_table}],]],
      )->group_by($page_id_field_name, 'id');

    my $blocks_with_native = $query->get_all();

    throw Exception::Validator::Fields gettext(
'To disable "Native" format on this page please delete all blocks with this format first. Blocks with "Native" format: %s',
        join(', ', map {$mobile_app_rtb->public_id($_)} @$blocks_with_native)
    ) if @$blocks_with_native;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    return {
        map {$_ => TRUE}
          qw(
          family_filter
          behavioral_targeting
          view_images
          bk_languages
          fast_context
          only_picture
          caption
          block_title
          excluded_domains
          excluded_phones
          false_click
          performance_tgo_disable
          )
    };
}

sub get_bk_data {
    my ($self, $page) = @_;

    my %custom_opts;
    push @{$custom_opts{'enable'}}, "mobile-false-click"
      if (defined($page->{'false_click'}) && $page->{'false_click'} == 1);
    push @{$custom_opts{'disable'}}, "mobile-false-click"
      if (defined($page->{'false_click'}) && $page->{'false_click'} == 0);

    if ($page->{'performance_tgo_disable'}) {
        push @{$custom_opts{'enable'}}, 'performance-tgo-disable';
    } else {
        push @{$custom_opts{'disable'}}, 'performance-tgo-disable';
    }

    my %data = (
        application_id  => $page->{'id'},
        banner_language => $page->{'bk_languages_name'},
        domain          => $page->{'store_id'},
        name            => $page->{'store_id'},
        description     => $page->{'store_id'},
        target_type     => $CONTEXT_TARGET_TYPE,
        mobile_app      => TRUE,
        options         => {
            dontshowbehavior => $page->{'behavioral_targeting'} ? 0 : 1,
            dontshowsex      => $page->{'family_filter'},
            BlockTitle       => $page->{'block_title'},
        },
        page_options => $self->page_options->get_options($page->{'page_id'}, %custom_opts),
    );

    return %data;
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    my $id;
    my $id_name;
    my $lng_table_name;
    my @to_update = qw(excluded_domains excluded_phones);

    if ($self->is_external_page()) {
        $id             = $self->hook_stash->get('settings')->{'context_page_id'};
        $id_name        = 'context_page_id';
        $lng_table_name = 'mobile_app_bk_language';

        if (defined($opts->{'assistants'})) {
            push @to_update, 'assistants';
        }
    }

    if ($self->is_internal_page()) {
        $id             = $self->hook_stash->get('id')->{'id'};
        $id_name        = 'application_id';
        $lng_table_name = 'internal_mobile_app_bk_language';
        push @to_update, 'managers';
    }

    my $mode = $self->hook_stash->mode('add') ? 'add' : 'edit';

    my $bk_languages = delete($opts->{'bk_languages'});
    if (defined($bk_languages)) {
        $self->partner_db->$lng_table_name->delete($self->partner_db->filter({$id_name => $id})) if $mode eq 'edit';
        $self->partner_db->$lng_table_name->add(
            {
                $id_name    => $id,
                language_id => $_,
            }
        ) foreach (@$bk_languages);
    }

    foreach (@to_update) {
        if (defined($opts->{$_})) {
            $self->$_->replace($id, delete($opts->{$_}), $mode);
        }
    }
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions('set_need_update'));
    if (   $self->need_update_in_bk($self->hook_stash->get('fields'))
        && $self->check_action($self->hook_stash->get('id'), 'set_need_update'))
    {
        $self->do_action($self->hook_stash->get('id'), 'set_need_update');
    }
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_processing_before_validation($opts);

    if (defined $opts->{'block_title'}) {
        $opts->{'block_title'} =~ s/[,;=']//g;    # workaround for gettext extractor bug
    }

    if ($self->hook_stash->mode('add')) {
        $opts->{'fast_context'} = 1;
        $opts->{'only_picture'} = 0;
        $opts->{'false_click'}  = 1;
        $opts->{'performance_tgo_disable'} //= 0;
    }
}

TRUE;
