package Application::Model::Page::SSP;

use qbit;

use base qw(
  Application::Model::Page
  Application::Model::Page::MIXIN::External
  );

consume qw(
  Application::Model::Role::Page::Has::MobileAppMode
  Application::Model::Role::Has::Page::Patch
  );

use Utils::MonitoringUtils qw/send_to_graphite/;

use Utils::PhoneBK;
use Utils::DomainBK;

use Utils::Logger qw(ERROR);

use PiConstants qw(
  $BOOKMAKER_FLAG_ID
  $CONTEXT_TARGET_TYPE
  );

sub get_page_id_field_name {'page_id'}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        ssp_seller       => 'Application::Model::Product::SSP::Seller',
        ssp_mirrors      => 'Application::Model::Product::SSP::Mirrors',
        api_http_maas    => 'QBit::Application::Model::API::Yandex::HTTPMAAS',
        ssp_imps         => 'Application::Model::Product::SSP::ImpressionLog',
        ssp_source       => 'Application::Model::Product',
        ssp_links        => 'Application::Model::Product',
        exception_dumper => 'Application::Model::ExceptionDumper',
        thematic_filters => 'Application::Model::ThematicFilters',
        filters          => 'Application::Model::Filters',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {%{$rights->[0]{'rights'}}, %{$self->get_external_structure_rights_to_register()},};

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $fields = {
        %{$self->SUPER::get_structure_model_fields()},
        %{$self->get_external_structure_model_fields()},

        source_id => {db => TRUE,},
        domain    => {
            depends_on => [qw(source_id ssp_source.domain)],
            get        => sub {
                $_[0]->{'ssp_source'}{$_[1]->{'source_id'}}{'domain'} // '';
              }
        },
        seller_id => {db => TRUE,},
        tokens    => {
            depends_on => [qw(id ssp_links.tokens)],
            get        => sub {
                $_[0]->{'ssp_links'}{$_[1]->{'id'}}{'tokens'} // '';
            },
        },
        banner_langs => {
            db  => TRUE,
            get => sub {
                return from_json($_[1]->{'banner_langs'} // '[]') // [];
              }
        },

        seller_name => {
            depends_on => ['seller_id', 'ssp_seller.name'],
            get        => sub {
                $_[0]->{'ssp_seller'}{$_[1]->{'seller_id'}}{'name'} // '';
            },
        },

        mirrors => {
            label      => d_gettext('Mirrors'),
            depends_on => ['id', 'mirrors.domain'],
            get        => sub {
                $_[0]->{'mirrors'}{$_[1]->{'id'}} // [];
            },
            type     => 'array',
            sub_type => 'string',
            api      => 1,
        },
        filters => {
            label      => d_gettext('Filters list'),
            depends_on => ['page_id', 'filters.filter_id'],
            get        => sub {
                defined($_[1]->{'page_id'}) ? $_[0]->{'filters'}{$_[1]->{'page_id'}} // [] : [];
            },
            type     => 'array',
            sub_type => 'number',
            api      => 1,
        },

        block_title          => {db => TRUE,},
        view_images          => {db => TRUE,},                                          #pictures_enabled
        behavioral_targeting => {db => TRUE,},
        fast_context         => {db => TRUE},
        only_picture         => {db => TRUE},
        family_filter        => {db => TRUE},
        cpa                  => {db => TRUE, label => d_gettext('CPA (conversion)')},

        editable_fields => {
            label => d_gettext('Editable fields'),
            get   => sub {
                return $_[0]->model->get_editable_fields($_[1]);
              }
        },
        available_fields => {
            depends_on => [qw(multistate)],
            label      => d_gettext('Available fields'),
            get        => sub {
                return $_[0]->model->get_available_fields($_[1]);
              }
        },
        available_blocks => {
            depends_on => [qw(multistate)],
            get        => sub {
                return $_[0]->model->get_available_block_model_names($_[1]);
              }
        },

        # заглушки (наследование)

        caption => {
            label      => d_gettext('Caption'),
            depends_on => ['public_id'],
            get        => sub {
                $_[1]->{'public_id'};
            },
        },
        is_tutby => {
            db      => TRUE,
            db_expr => \0,
        },
        images_first => {db => TRUE, db_expr => \1,},
    };

    return $fields;
}

sub get_actions_depends {
    return [qw(id multistate page_id cur_user_is_read_assistant)];
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $filter = $self->SUPER::get_structure_model_filter();

    $filter->{'fields'} = {
        %{$filter->{'fields'}},
        %{$self->get_external_structure_model_filter},
        source_id  => {type => 'number', label => d_gettext('Resource ID')},
        seller_id  => {type => 'number', label => d_gettext('Seller ID')},
        ssp_source => {
            type           => 'subfilter',
            model_accessor => 'ssp_source',
            fk_field       => 'id',
            field          => 'source_id',
            label          => d_gettext('SSP resource'),
        },
    };

    return $filter;
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    my $accessor = $self->accessor();

    return {
        empty_name  => d_gettext('New'),
        multistates => [
            [working            => d_pgettext('Campaign status', 'Working')],
            [testing            => d_pgettext('Campaign status', 'Testing')],
            [stopped            => d_pgettext('Campaign status', 'Stopped')],
            [deleted            => d_pgettext('Campaign status', 'Archived'), private => TRUE],
            [balance_registered => d_pgettext('Campaign status', 'Added to Balance'), private => TRUE],
            [read_only          => d_pgettext('Campaign status', 'Read only'), private => TRUE],
            [check_statistics   => d_pgettext('Campaign status', 'Check statistics'), private => TRUE],
            [blocked            => d_pgettext('Campaign status', 'Blocked')],
            [need_approve       => d_pgettext('Campaign status', 'Need approve')],
            [rejected           => d_pgettext('Campaign status', 'Rejected')],
            [protected          => d_pgettext('Campaign status', 'Protected')],
            [need_update        => d_pgettext('Campaign status', 'Need update')],
            [updating           => d_pgettext('Campaign status', 'Updating')],
        ],
        actions => {
            can_update_in_bk       => d_pgettext('Campaign action', 'Can update in bk'),
            reset_check_statistics => d_pgettext('Campaign action', 'Reset "check_statistics"'),
            restore_block          => d_pgettext('Campaign action', 'Restore block on campaign'),
            set_check_statistics   => d_pgettext('Campaign action', 'Set "check_statistics"'),
            set_need_approve       => d_pgettext('Campaign action', 'Set need approve'),
            start_block            => d_pgettext('Campaign action', 'Start block on campaign'),
            del_mirror             => d_pgettext('Campaign action', 'Remove mirror'),
        },
        right_actions => {
            add                 => d_pgettext('Campaign action', 'Add'),
            approve             => d_pgettext('Campaign action', 'Approve'),
            delete              => d_pgettext('Campaign action', 'Archive'),
            edit                => d_pgettext('Campaign action', 'Edit'),
            register_in_balance => d_pgettext('Campaign action', 'Register in Balance'),
            reject              => d_pgettext('Campaign action', 'Reject'),
            reset_blocked       => d_pgettext('Campaign action', 'Reset blocked'),
            reset_protected     => d_pgettext('Campaign action', 'reset protected'),
            reset_read_only     => d_pgettext('Campaign action', 'Reset read only'),
            restore             => d_pgettext('Campaign action', 'Restore'),
            set_blocked         => d_pgettext('Campaign action', 'Set blocked'),
            set_need_update     => d_pgettext('Campaign action', 'Set "need_update"'),
            set_protected       => d_pgettext('Campaign action', 'Set protected'),
            set_read_only       => d_pgettext('Campaign action', 'Set read only'),
            start               => d_pgettext('Campaign action', 'Start'),
            start_testing       => d_pgettext('Campaign action', 'Start testing'),
            start_update        => d_pgettext('Campaign action', 'Start update'),
            stop                => d_pgettext('Campaign action', 'Stop'),
            stop_update         => d_pgettext('Campaign action', 'Stop update'),
        },
        right_group        => [$accessor => d_gettext('Right to manage context campaigns')],
        right_name_prefix  => $accessor . '_',
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'register_in_balance',
                from      => 'not (balance_registered or deleted or need_approve or rejected or blocked)',
                set_flags => ['balance_registered'],
            },
            {
                action => 'start',
                from =>
                  'balance_registered and not (working or deleted or blocked or need_approve or rejected or protected)',
                set_flags   => ['working'],
                reset_flags => ['stopped', 'testing']
            },
            {
                action => 'start_testing',
                from =>
'balance_registered and not (testing or working or deleted or blocked or need_approve or rejected or protected)',
                set_flags   => ['testing'],
                reset_flags => ['stopped', 'working'],
            },
            {
                action      => 'stop',
                from        => '(working or testing) and not (stopped or need_approve or rejected or protected)',
                set_flags   => ['stopped'],
                reset_flags => ['working', 'testing'],
            },
            {
                action    => 'delete',
                from      => 'not (working or testing or deleted)',
                set_flags => ['deleted']
            },
            {
                action      => 'restore',
                from        => 'deleted and not blocked',
                reset_flags => ['deleted'],
            },
            {
                action => 'edit',
                from   => 'not (deleted or blocked or rejected)'
            },
            {
                action => 'del_mirror',
                from   => 'not (deleted or blocked or rejected or protected)'
            },
            {
                action    => 'set_read_only',
                from      => 'not read_only',
                set_flags => ['read_only'],
            },
            {
                action      => 'reset_read_only',
                from        => 'read_only',
                reset_flags => ['read_only'],
            },
            {
                action      => 'set_blocked',
                from        => 'not blocked',
                set_flags   => ['blocked'],
                reset_flags => ['need_approve'],
            },
            {
                action      => 'reset_blocked',
                from        => 'blocked',
                reset_flags => ['blocked'],
            },
            {
                action    => 'set_check_statistics',
                from      => '__EMPTY__',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action => 'start_block',
                from   => '(working or testing) and not protected',
            },
            {
                action => 'restore_block',
                from   => 'not deleted',
            },
            {
                action    => 'set_need_approve',
                from      => '__EMPTY__',
                set_flags => ['need_approve'],
            },
            {
                action      => 'approve',
                from        => 'need_approve',
                reset_flags => ['need_approve'],
            },
            {
                action      => 'reject',
                from        => 'need_approve',
                reset_flags => ['need_approve'],
                set_flags   => ['rejected'],
            },
            {
                action => 'can_update_in_bk',
                from   => 'not (need_approve or rejected or deleted or protected)',
            },
            {
                action    => 'set_protected',
                from      => 'not protected',
                set_flags => ['protected'],
            },
            {
                action      => 'reset_protected',
                from        => 'protected',
                reset_flags => ['protected'],
            },
            {
                action    => 'set_need_update',
                from      => 'not (need_approve or rejected or deleted or protected)',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update or updating',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ]
    };
}

sub pre_process_fields {
    my ($self, $fields, $result, %opts) = @_;

    my $ids      = $opts{'ids'}      //= array_uniq(map {$_->{'id'}      // ()} @$result);
    my $page_ids = $opts{'page_ids'} //= array_uniq(map {$_->{'page_id'} // ()} @$result);

    $self->SUPER::pre_process_fields($fields, $result, %opts);

    if ($fields->need('owner')) {
        my $tmp_rights = $self->app->add_tmp_rights('users_view_all', 'users_view_field__client_id');
        $fields->{'__OWNERS__'} = {
            map {$_->{'id'} => $_} @{
                $self->users->get_all(
                    fields => [qw(id login client_id business_unit name midname lastname roles is_tutby)],
                    filter => {id => array_uniq(map {$_->{'owner_id'} // ()} @$result)},
                )
              }
        };
        undef($tmp_rights);
    }

    if ($fields->need('assistants')) {
        $fields->{'__ASSISTANTS__'} = {};

        foreach (@{$self->assistants->get_assistants()}) {
            my $page_id = delete($_->{'page_id'});

            $fields->{'__ASSISTANTS__'}{$page_id} //= [];
            push(@{$fields->{'__ASSISTANTS__'}{$page_id}}, $_);
        }
    }
}

sub get_available_fields {{}}

sub get_add_fields {
    my ($self) = @_;

    return {
        map {$_ => TRUE}
          qw(
          banner_langs
          behavioral_targeting
          block_title
          context
          cpa
          create_date
          family_filter
          fast_context
          filters
          mirrors
          mobile_app_mode
          only_picture
          owner_id
          page_id
          seller_id
          source_id
          view_images
          ),
    };
}

sub get_editable_fields {{}}

sub collect_editable_fields {{}}

sub add {
    my ($self, %opts) = @_;

    $opts{'banner_langs'} = to_json($opts{'banner_langs'});

    my $id;
    $self->partner_db->transaction(
        sub {
            my $mirrors          = delete($opts{'mirrors'});
            my $excluded_domains = delete($opts{'excluded_domains'}) // [];
            my $excluded_phones  = delete($opts{'excluded_phones'}) // [];
            my $filters          = delete($opts{'filters'}) // [];

            $id = $opts{'id'} = $opts{'page_id'};
            $self->partner_db_table->add(\%opts);

            $self->ssp_mirrors->add($id, $mirrors);
            $self->excluded_domains->replace($opts{'page_id'}, $excluded_domains, 'add') if @$excluded_domains;
            $self->excluded_phones->replace($opts{'page_id'}, $excluded_phones, 'add') if @$excluded_phones;
            $self->filters->replace($opts{'page_id'}, array_uniq(@$filters, $BOOKMAKER_FLAG_ID), 'add');

            $self->do_action($id, "add", %opts);

            $self->do_action($id, 'register_in_balance');

            $self->do_action($id, 'start');
        }
    );

    return $id;
}

sub on_action_register_in_balance {
    my ($self, $obj) = @_;

    my $object = $self->_get_object_fields($obj, [qw(domain client_id page_id)]);

    $self->api_balance->create_or_update_place(
        operator_uid  => 0,
        client_id     => $object->{'client_id'},
        page_id       => $object->{'page_id'},
        domain        => $object->{'domain'},
        campaign_type => 3,
    );
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_robot_assistant_or_owner($filter);

    return $filter;
}

sub get_bk_data {
    my ($self, $page) = @_;

    my $user_id = $page->{'owner_id'};

    my @user_global_excluded_phones =
      map {$_->{'phone'}} @{$self->user_global_excluded_phones->get_all(filter => {user_id => $user_id})};

    my @user_global_excluded_domains =
      map {$_->{'domain'}} @{$self->user_global_excluded_domains->get_all(filter => {user_id => $user_id})};

    my @domains = check_domains_for_absorption([@{$page->{'excluded_domains'}}, @user_global_excluded_domains]);

    my %hash_excluded_domains = map {$_ => TRUE} @domains,
      bs_format_phone([@{$page->{'excluded_phones'}}, @user_global_excluded_phones]),
      @{$page->{'mirrors'}}, $page->{'domain'};

    my $disabled_flags;
    if (exists($page->{'filters'})) {
        my %filter_name =
          map {$_->{'id'} => $_->{'bk_name'}} @{$self->thematic_filters->get_all(fields => [qw(id bk_name)])};

        $disabled_flags = [map {$filter_name{$_}} @{$page->{'filters'}}];
    }

    my %bk_data = (
        product_id  => $self->accessor,
        target_type => $CONTEXT_TARGET_TYPE,
        is_pi2      => '1',

        ssp_source_tokens => $page->{'tokens'},
        ssp_seller_id     => $page->{'seller_id'},

        cpa         => $page->{'cpa'},
        create_date => $page->{'create_date'},

        mirrors => $page->{'mirrors'} // [],
        excluded_domains => [sort keys(%hash_excluded_domains)],

        options => {
            dontshowbehavior => $page->{'behavioral_targeting'} ? 0 : 1,
            dontshowsex      => $page->{'family_filter'},
            BlockTitle       => $page->{'block_title'},
        },

        fast_context => $page->{'fast_context'},
        only_picture => $page->{'only_picture'},
        page_options => $self->page_options->get_options($page->{'page_id'}),

        ($page->{'banner_langs'} ? (banner_language => $page->{'banner_langs'}) : ()),
        ($disabled_flags         ? (disabled_flags  => $disabled_flags)         : ()),
    );

    return %bk_data;
}

sub api_available_actions { }

sub check_action {&Application::Model::Page::MIXIN::External::check_action}

sub create_from_impression {
    my ($self, $imp) = @_;

    my $tmp_field = $self->get_tmp_field_name();

    my $seller =
      $self->ssp_seller->get($imp->{'seller_id'}, fields => [qw(id user_id client_id business_unit), $tmp_field]);

    throw Exception gettext('Incorrect json in filed "%s" for SSP seller "%d"', $tmp_field, $imp->{'seller_id'})
      if !$seller->{$tmp_field};

    my $tmpl = from_json($seller->{$tmp_field});
    $tmpl->{'mincpm'} //= '0';

    my $source = $self->ssp_source->get($imp->{'source_id'}, fields => [qw(domain)]);

    my $page_id = $self->api_utils_partner2->get_next_page_id();

    my %opts = (
        %$tmpl,
        create_date => curdate(oformat => 'db_time'),
        page_id     => $page_id,
        source_id   => $imp->{'source_id'},
        owner_id    => $seller->{'user_id'},
        seller_id   => $seller->{'id'},
    );

    my $id;
    $self->partner_db->transaction(
        sub {
            my $add_fields = $self->get_add_fields();

            $id = $self->add(hash_transform(\%opts, [sort keys(%$add_fields)]));

            my $block_model = $self->get_block_model_names()->[0];

            my $block_add_fields = $self->$block_model->get_add_fields();

            $self->$block_model->add(hash_transform(\%opts, [sort keys(%$block_add_fields)]), campaign_id => $page_id);
        }
    );

    return $id;
}

sub is_external_page {
    # переопределение, чтобы не делалась проверка договора...
    # может она нужна?!
    return FALSE;
}

sub related_models {
    my ($self) = @_;

    my $fk_link_field = $self->ssp_links->get_campaign_id_field();

    return {
        %{$self->SUPER::related_models()},
        %{$self->external_related_models()},
        ssp_seller => {
            accessor => 'ssp_seller',
            filter   => sub {
                return {id => array_uniq(map {$_->{'seller_id'} // ()} @{$_[1]}),};
            },
            key_fields => ['id'],
        },
        mirrors => {
            accessor => 'ssp_mirrors',
            filter   => sub {
                return {campaign_id => array_uniq(map {$_->{'id'} // ()} @{$_[1]})};
            },
            key_fields => ['campaign_id'],
            value_type => 'array_domain',
        },
        ssp_source => {
            accessor => 'ssp_source',
            filter   => sub {
                return {id => array_uniq(map {$_->{'source_id'} // ()} @{$_[1]})};
            },
            key_fields => ['id'],
        },
        ssp_links => {
            accessor => 'ssp_links',
            filter   => sub {
                return {$fk_link_field => array_uniq(map {$_->{'id'} // ()} @{$_[1]})};
            },
            key_fields => [$fk_link_field],
        },
        filters => {
            accessor => 'filters',
            filter   => sub {
                return {page_id => array_uniq(map {$_->{'page_id'} // ()} @{$_[1]})};
            },
            key_fields => ['page_id'],
            value_type => 'array_filter_id',
        },
    };
}

#
# not used (перетащить эти две сабы в базовый для links)
#

sub update_data {
    my ($self, $date) = @_;

    #TODO: change prefix
    my $prefix = 'SSP.' . $self->accessor();

    send_to_graphite(
        value => dates_delta_days($date, curdate(oformat => 'db'), iformat => 'db'),
        path  => "$prefix.link_process_age"
    );

    my $imps = $self->ssp_imps->get_sources($date);

    send_to_graphite(value => scalar(@$imps), path => "$prefix.link_count_sources");

    my ($cnt_page_updated, $cnt_page_created) = (0, 0);

    foreach my $imp (@$imps) {
        next unless $self->check_log_line($imp);

        if (defined($imp->{'site_id'}) && defined($imp->{'application_id'})) {
            my $exception = Exception->new("site_id and application_id defined together: " . to_json($imp));
            $self->exception_dumper->dump_as_html_file($exception);

            next;
        }

        $imp->{'source_id'} = $imp->{'application_id'} // $imp->{'site_id'};

        if ($self->update_from_impression($imp)) {
            $cnt_page_updated += 1;
        } elsif ($self->create_from_impression($imp)) {
            $cnt_page_created += 1;
        }
    }

    send_to_graphite(value => $cnt_page_updated, path => "$prefix.link_count_updated");
    send_to_graphite(value => $cnt_page_created, path => "$prefix.page_count_created");
}

sub update_from_impression {
    my ($self, $imp) = @_;

    # get old page
    my $pages = $self->get_all(
        fields => [
            qw(
              id
              source_id
              page_id
              tokens
              seller_id
              )
        ],
        filter => {
            source_id => $imp->{'source_id'},
            seller_id => $imp->{'seller_id'}
        }
    );

    return 0 unless @$pages;

    throw 'Expected one page...' if @$pages > 1;

    my $page = $pages->[0];

    # new token
    my $new_tokens = arrays_difference($imp->{'source_token'}, $page->{'tokens'});

    if (@$new_tokens) {
        # save data in db
        $self->partner_db_table->edit($page, {tokens => to_json([@$new_tokens, @{$page->{'tokens'}}]),});

        # save data in bk (context)
        if ($page->{'page_id'}) {
            $self->do_action($page, 'set_need_update');
        }
    }

    return 1;
}

TRUE;
