package Application::Model::Page::Site;

use qbit;

use base qw(Application::Model::Page);

use PiConstants qw(@BK_LANGUAGES_BY_DEFAULT);

use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        bk_language  => 'Application::Model::BKLanguage',
        mirrors      => 'QBit::Application::Model',
        quality_coef => 'Application::Model::QualityCoef',
        site         => 'QBit::Application::Model',
        partner_db   => 'Application::Model::PartnerDB',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        map {$self->get_description_right($_)}
          qw(
          view_field__banner_lang
          edit_field__banner_lang
          view_field__reload_timeout
          edit_field__reload_timeout
          view_field__block_title
          edit_field__block_title
          view_field__quality_coef
          view_field__domain_id
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        domain_id => {
            default      => TRUE,
            db           => TRUE,
            label        => d_gettext('Domain id'),
            check_rights => $self->get_right('view_field__domain_id'),
            type         => 'number',
            api          => 1,
            need_check   => {type => 'int_un',},
            adjust_type  => 'str',
        },
        domain => {
            label             => d_gettext('Domain'),
            forced_depends_on => ['domain_id', 'site.domain'],
            get               => sub {
                $_[0]->{'site'}{$_[1]->{'domain_id'}}{'domain'} // '';
            },
            type       => 'string',
            api        => 1,
            need_check => {
                len_min => 1,
                len_max => 255,
                check   => sub {
                    my ($qv, $value) = @_;
                    throw Exception::Validator::Fields gettext('Invalid domain') unless defined $value;
                },
            },
        },
        quality_coef => {
            depends_on   => ['page_id'],
            check_rights => $self->get_right('view_field__quality_coef'),
            get          => sub {
                my ($self, $obj) = @_;

                # TODO удалить следующую строчку в одном из следующих ПР, например по PI-15793
                return undef unless $obj->{'page_id'};

                return $self->{'__QUALITY_COEF__'}{$obj->{'page_id'}};
            },
            type        => 'number',
            api         => 1,
            adjust_type => 'str',
        },
        reload_timeout => {
            db           => TRUE,
            type         => 'number',
            check_rights => $self->get_right('view_field__reload_timeout'),
            api          => 1,
            need_check   => {
                min => 1,
                max => 30,
            },
            adjust_type => 'str',
        },
        #no in mobile
        family_filter => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean',},
            adjust_type => 'str',
        },
        banner_lang => {
            depends_on   => [qw(id)],
            label        => d_gettext('Banner language ids'),
            check_rights => $self->get_right('view_field__banner_lang'),
            get          => sub {
                [sort keys(%{$_[0]->{'__BANNER_LANG__'}->{$_[1]->{'id'}}})];
            },
            type       => 'array',
            sub_type   => 'number',
            api        => 1,
            need_check => {
                type  => 'array',
                check => sub {
                    my ($qv, $banner_lang) = @_;

                    my %uniq_banner_lang;

                    foreach my $lang_id (@$banner_lang) {
                        throw Exception::Validator::Fields gettext('Elements must be integer number')
                          unless $lang_id =~ m/^\d+\z/;
                        $uniq_banner_lang{$lang_id} = TRUE;
                    }

                    $banner_lang = [sort keys %uniq_banner_lang];

                    my %known = map {$_->{id} => TRUE} @{
                        $qv->app->bk_language->get_all(
                            fields => [qw(id)],
                            filter => {id => $banner_lang},
                        )
                      };

                    my @unknown = sort {$a <=> $b} grep {not $known{$_}} @$banner_lang;

                    throw Exception::Validator::Fields gettext('Unknown language ID(s): %s', join(',', @unknown))
                      if @unknown;
                },
            },
            adjust_type => 'array_str',
        },
        banner_lang_name => {
            depends_on   => [qw(id)],
            label        => d_gettext('Banner languages'),
            check_rights => $self->get_right('view_field__banner_lang'),
            get          => sub {
                [sort values(%{$_[0]->{'__BANNER_LANG__'}->{$_[1]->{'id'}}})];
            },
            type     => 'array',
            sub_type => 'string',
            api      => 1,
        },
        page_lang => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                type  => 'int_un',
                check => sub {
                    my ($qv, $page_lang) = @_;
                    throw Exception::Validator::Fields gettext('Page language with ID=%s does not exist', $page_lang)
                      unless $qv->app->bk_language->get($page_lang);
                },
            },
            adjust_type => 'str',
        },
        page_lang_name => {
            depends_on => [qw(page_lang)],
            get        => sub {
                $_[0]->{'__BK_LANG__'}->{$_[1]->{'page_lang'}};
            },
            type => 'string',
            api  => 1,
        },
        page_lang_display_name => {
            depends_on => [qw(page_lang)],
            get        => sub {
                $_[0]->{'__BK_LANG_DISPLAY_NAME__'}->{$_[1]->{'page_lang'}};
            },
            type => 'string',
        },
        block_title => {
            db           => TRUE,
            type         => 'string',
            check_rights => $self->get_right('view_field__block_title'),
            api          => 1,
            need_check   => {
                len_max  => 255,
                optional => TRUE,
            },
        },
        # Счетчик заведенный автоматом по крону (#PI-10338)
        # TODO: Перенести в миксину или роль
        metrica_counter => {
            depends_on => ['page_id'],
            type       => 'number',
            api        => 1,
            need_check => {
                type     => 'int_un',
                optional => TRUE,
            },
            get => sub {
                # NOTE! page_id может и н быть (#PI-12865)
                $_[0]->{'__METRIKA_COUNTERS__'}->{$_[1]->{'page_id'} // ''};
            },
            adjust_type => 'str',
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $filter = $self->SUPER::get_structure_model_filter();

    $filter->{'fields'} = {
        %{$filter->{'fields'}},
        domain_id   => {type => 'number', label => d_gettext('Domain ID')},
        domain_text => {type => 'alias',  path  => [qw(domain domain)]},
        domain      => {
            type           => 'subfilter',
            model_accessor => 'site',
            field          => 'domain_id',
            fk_field       => 'id',
            label          => d_gettext('Domains'),
        },
        all_domain => {
            type  => 'domain_mirror',
            label => d_gettext('All domains'),
        },
        metrica_counters => {type => 'text'},
    };

    return $filter;
}

sub pre_process_fields {
    my ($self, $fields, $result, %opts) = @_;

    my $ids      = $opts{'ids'}      //= array_uniq(map {$_->{'id'}      // ()} @$result);
    my $page_ids = $opts{'page_ids'} //= array_uniq(map {$_->{'page_id'} // ()} @$result);

    $self->SUPER::pre_process_fields($fields, $result, %opts);

    if (   $fields->need('banner_lang')
        || $fields->need('banner_lang_name')
        || $fields->need('page_lang_name')
        || $fields->need('page_lang_display_name'))
    {
        my $bk_language = $self->bk_language->get_all(fields => [qw(id language display_name)]);
        $fields->{__BK_LANG__}              = {map {$_->{'id'} => $_->{'language'}} @$bk_language};
        $fields->{__BK_LANG_DISPLAY_NAME__} = {map {$_->{'id'} => $_->{'display_name'}} @$bk_language};
    }

    if ($fields->need('banner_lang') || $fields->need('banner_lang_name')) {
        my @banner_lang_by_campaign_id = @{
            $self->db_banner_lang->get_all(
                fields => [qw(campaign_id language_id)],
                filter => {campaign_id => $ids},
            );
          };

        $fields->{'__BANNER_LANG__'}{$_->{'campaign_id'}}{$_->{'language_id'}} =
          $fields->{__BK_LANG__}{$_->{'language_id'}}
          foreach @banner_lang_by_campaign_id;

        my %bk_lang = reverse(%{$fields->{__BK_LANG__}});

        foreach my $id (@$ids) {
            $fields->{'__BANNER_LANG__'}{$id} ||=
              {map {$bk_lang{$_} => $_} grep {$bk_lang{$_}} @BK_LANGUAGES_BY_DEFAULT};
        }
    }

    if ($fields->need('quality_coef')) {
        $fields->{'__QUALITY_COEF__'} = $self->quality_coef->get_pages_coef(pages => $page_ids);
    }

    if ($fields->need('metrica_counter')) {

        my $data = $self->partner_db->query->select(
            table  => $self->partner_db->metrika_counters,
            fields => [qw( page_id  counter_id )],
            filter => [page_id => 'IN' => \$page_ids],
        )->get_all();

        $fields->{'__METRIKA_COUNTERS__'} = {map {$_->{'page_id'} => $_->{'counter_id'}} @$data};
    }
}

sub _get_common_add_edit_fields {
    my ($self, $add_fields, $edit_fields) = @_;

    my $fields = $self->get_fields_by_right(
        res_fields   => $self->SUPER::_get_common_add_edit_fields(),
        right_fields => {edit => ['block_title']}
    );

    return $fields;
}

sub on_action_register_in_balance {
    my ($self, $obj) = @_;

    my $campaign = $self->_get_object_fields($obj, [qw(id owner domain client_id page_id create_date)])
      // throw Exception::Validation::BadArguments gettext('Campaign with ID=%s does not exists', $obj->{'id'});

    my $is_internal = $self->is_internal_page();

    $self->api_balance->create_or_update_place(
        operator_uid => $self->get_option(cur_user => {})->{'id'},
        client_id    => $campaign->{'client_id'},
        page_id      => $campaign->{'page_id'},
        domain       => $campaign->{'domain'},
        # sub campaign_type?
        campaign_type => $self->is_context_page()
        ? 3     # Контекстная
        : 2,    # Поисковая
        ($is_internal ? (viptype => 2) : ()),
        is_tutby    => $campaign->{'owner'}{'is_tutby'},
        create_date => $campaign->{'create_date'},
    );

    if ($is_internal) {
        $self->do_action($obj, 'start') if $self->check_action($obj->{'id'}, 'start');
    } else {
        $self->do_action($obj, 'start_testing') if $self->check_action($obj->{'id'}, 'start_testing');
    }
}

sub get_and_check_mirrors {
    my ($self, $mirrors) = @_;

    my @mirrors;

    foreach my $domain (split(/[\s,;]+/, $mirrors)) {
        $domain = $self->get_and_check_domain($domain)
          // throw Exception::Validation::BadArguments gettext('Invalid domain: "%s"', $domain);
        push(@mirrors, $domain);
    }

    return join(', ', sort @mirrors);
}

sub related_models {
    my ($self) = @_;

    return {
        %{$self->SUPER::related_models()},
        site => {
            accessor => 'site',
            filter   => sub {
                return {id => array_uniq(map {$_->{'domain_id'} // ()} @{$_[1]})};
            },
            key_fields => ['id'],
        },
    };
}

sub get_fields_to_trim {qw(block_title caption)}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    # workaround for gettext extractor bug
    $opts->{block_title} =~ s/[,;=']//g if exists $opts->{block_title};

    $opts->{'domain'} = $self->get_and_check_domain($opts->{'domain'}) if exists $opts->{'domain'};

    $self->SUPER::hook_fields_processing_before_validation($opts);
}

sub hook_preparing_fields_to_save {
    my ($self, $opts) = @_;

    if (exists($opts->{metrica_counters})) {
        $opts->{metrica_counters} = to_json($opts->{metrica_counters});
    }
    $self->SUPER::hook_preparing_fields_to_save($opts);
}

sub safe_block_title {
    my ($class, $block_title) = @_;
    # workaround for gettext extractor bug
    $block_title =~ s/[,;=']//g;
    return trim($block_title);
}

TRUE;
