package Application::Model::Page::Site::External;

use qbit;

use base qw(
  Application::Model::Page::Site
  Application::Model::Page::MIXIN::External
  );

use PiConstants qw($MODERATION_MAIL $BOOKMAKER_FLAG_ID);

consume qw(
  Application::Model::Role::Has::Page::FilterSelfDomain
  Application::Model::Role::Has::Page::Patch
  );

use Utils::PhoneBK;
use Utils::DomainBK;

use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;
use Utils::Logger qw(WARN);

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        site               => 'Application::Model::Product::AN::Site',
        cookie_match       => 'Application::Model::CookieMatch',
        mail_notification  => 'Application::Model::MailNotification',
        kv_store           => 'QBit::Application::Model::KvStore',
        user_notifications => 'Application::Model::UserNotifications',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        %{$self->get_external_structure_rights_to_register()},
        map {$self->get_description_right($_)}
          qw(
          view_field__banner_lang
          edit_field__banner_lang
          view_field__block_title
          edit_field__block_title
          view_field__quality_coef
          edit_field__market_api
          add_field__user_synchronization
          view_field__user_synchronization
          edit_field__user_synchronization
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        %{$self->get_external_structure_model_fields()},

        source_id => {
            db          => TRUE,
            db_expr     => 'domain_id',
            type        => 'string',
            api         => 1,
            adjust_type => 'str',
        },
        user_synchronization => {
            db           => TRUE,
            type         => 'boolean',
            check_rights => $self->get_right('view_field__user_synchronization'),
            need_check   => {
                optional => TRUE,
                type     => 'boolean'
            },
            api         => 1,
            adjust_type => 'str',
        },
        tag_id => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                check    => sub {
                    my ($qv, $value) = @_;

                    my @has_tag_id =
                      grep {$_->{'id'} == $value} @{$qv->app->cookie_match->get_by_user_id($qv->data->{owner_id})};

                    throw Exception::Validator::Fields gettext('Tag "%s" not found', $value)
                      unless @has_tag_id;
                },
            },
            adjust_type => 'str',
        },
        cookie_match => {
            label      => d_gettext('Tag'),
            depends_on => ['tag_id'],
            get        => sub {
                return defined($_[1]->{'tag_id'}) ? $_[0]->{'__TAGS__'}->{$_[1]->{'tag_id'}} // {} : {};
            },
            type => 'cookie_match',
            api  => 1,
        },
        pi_id => {
            db          => TRUE,
            type        => 'number',
            label       => d_gettext('Old Interface ID'),
            api         => 1,
            adjust_type => 'str',
        },
        market_api => {
            db         => TRUE,
            type       => 'boolean',
            api        => 1,
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
            adjust_type => 'str',
        },
        is_graysite => {
            label             => d_gettext('Gray site'),
            forced_depends_on => ['domain_id', 'site.is_graysite'],
            get               => sub {
                $_[0]->{'site'}{$_[1]->{'domain_id'}}{'is_graysite'} // '';
            },
            type        => 'boolean',
            adjust_type => 'str',
        },
        mirrors => {
            label      => d_gettext('Mirrors'),
            depends_on => ['id', 'mirrors.domain', 'mirrors.multistate', 'mirrors.moderation_status', 'mirrors.auto'],
            get        => sub {
                my ($self, $obj) = @_;
                return [sort {$a->{'domain'} cmp $b->{'domain'}} @{$self->{'mirrors'}{$obj->{'id'}} // []}];
            },
            type     => 'complex',
            fix_type => sub {
                my ($model, $value) = @_;

                for (@$value) {
                    $_->{'id'} += 0;
                    $_->{'domain'} .= '';
                    $_->{'multistate'}  += 0;
                    $_->{'campaign_id'} += 0;
                    $_->{'moderation_status'} .= '';
                }

                return $value;
            },
            need_check => {
                type     => 'array',
                optional => TRUE,
            },
            api => 1,
        },
        is_site_approved => {
            forced_depends_on => ['domain_id', 'site.is_approved'],
            get               => sub {
                $_[0]->{'site'}{$_[1]->{'domain_id'}}{'is_approved'} // '';
            },
            type        => 'boolean',
            adjust_type => 'str',
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $filter = $self->SUPER::get_structure_model_filter();

    $filter->{'fields'} = {
        %{$filter->{'fields'}},
        %{$self->get_external_structure_model_filter()},
        source_id => {
            type      => 'number',
            label     => d_gettext('Source ID'),
            db_filter => sub {
                return ['domain_id' => $_[1]->[1] => \$_[1]->[2]];
            },
        },
        tag_id     => {type => 'number',  label => d_gettext('Tag ID')},
        market_api => {type => 'boolean', label => d_gettext('Market API')},
        pi_id      => {type => 'number',  label => d_gettext('Old Interface ID')},
        mirror     => {
            type           => 'subfilter',
            model_accessor => 'mirrors',
            field          => 'id',
            fk_field       => 'campaign_id',
            label          => d_gettext('Mirrors'),
        },
    };

    return $filter;
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        (
            $self->check_rights('view_search_filters__login') ? {name => 'owner.login', label => gettext('Login')}
            : ()
        ),
        (
            $self->check_rights('view_search_filters__assistants_login')
            ? {name => 'assistants.user.login', label => gettext('Assistant\'s login')}
            : ()
        ),
        (
            $self->check_rights('view_search_filters__user_type') ? {name => 'user_type', label => gettext('User type')}
            : ()
        ),
        {name => 'page_id',    label => gettext('Page ID')},
        {name => 'pi_id',      label => gettext('Old Interface ID')},
        {name => 'all_domain', label => gettext('Domain and mirror')},
        {name => 'multistate', label => gettext('Status')},
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    my $accessor = $self->accessor();
    my ($rigth_prefix, $right_group) = $accessor =~ /((.+)_)campaign$/;

    return {
        empty_name  => d_gettext('New'),
        multistates => [
            [working            => d_pgettext('Campaign status', 'Working')],
            [testing            => d_pgettext('Campaign status', 'Testing')],
            [stopped            => d_pgettext('Campaign status', 'Stopped')],
            [deleted            => d_pgettext('Campaign status', 'Archived'), private => TRUE],
            [balance_registered => d_pgettext('Campaign status', 'Added to Balance'), private => TRUE],
            [read_only          => d_pgettext('Campaign status', 'Read only'), private => TRUE],
            [
                check_statistics => d_pgettext('Campaign status', 'Check statistics'),
                private          => TRUE,
                deprecated       => TRUE
            ],
            [blocked      => d_pgettext('Campaign status', 'Blocked')],
            [need_approve => d_pgettext('Campaign status', 'Need approve')],
            [rejected     => d_pgettext('Campaign status', 'Rejected')],
            [protected    => d_pgettext('Campaign status', 'Protected')],
            [need_update  => d_pgettext('Campaign status', 'Need update')],
            [updating     => d_pgettext('Campaign status', 'Updating')],
        ],
        actions => {
            can_update_in_bk       => d_pgettext('Campaign action', 'Can update in bk'),
            reset_check_statistics => d_pgettext('Campaign action', 'Reset "check_statistics"'),
            restore_block          => d_pgettext('Campaign action', 'Restore block on campaign'),
            set_check_statistics   => d_pgettext('Campaign action', 'Set "check_statistics"'),
            set_need_approve       => d_pgettext('Campaign action', 'Set need approve'),
            start_block            => d_pgettext('Campaign action', 'Start block on campaign'),
            del_mirror             => d_pgettext('Campaign action', 'Remove mirror'),
        },
        right_actions => {
            add                 => d_pgettext('Campaign action', 'Add'),
            approve             => d_pgettext('Campaign action', 'Approve'),
            delete              => d_pgettext('Campaign action', 'Archive'),
            edit                => d_pgettext('Campaign action', 'Edit'),
            register_in_balance => d_pgettext('Campaign action', 'Register in Balance'),
            reject              => d_pgettext('Campaign action', 'Reject'),
            reset_blocked       => d_pgettext('Campaign action', 'Reset blocked'),
            reset_protected     => d_pgettext('Campaign action', 'reset protected'),
            reset_read_only     => d_pgettext('Campaign action', 'Reset read only'),
            restore             => d_pgettext('Campaign action', 'Restore'),
            set_blocked         => d_pgettext('Campaign action', 'Set blocked'),
            set_need_update =>
              {label => d_pgettext('Campaign action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            set_protected => d_pgettext('Campaign action', 'Set protected'),
            set_read_only => d_pgettext('Campaign action', 'Set read only'),
            start         => d_pgettext('Campaign action', 'Start'),
            start_testing => d_pgettext('Campaign action', 'Start testing'),
            start_update => {label => d_pgettext('Campaign action', 'Start update'), dont_write_to_action_log => TRUE},
            stop => d_pgettext('Campaign action', 'Stop'),
            stop_update => {label => d_pgettext('Campaign action', 'Stop update'), dont_write_to_action_log => TRUE},
        },
        right_group => [$right_group => d_gettext('Right to manage context campaigns')],
        right_name_prefix =>
          $rigth_prefix,    # TODO: переименовать в $self->accessor . '_' (не хватает "campaign")
        multistate_actions => [
            {
                action    => 'add',
                from      => '__EMPTY__',
                set_flags => ['need_approve'],
            },
            {
                action    => 'register_in_balance',
                from      => 'not (balance_registered or deleted or need_approve or rejected or blocked)',
                set_flags => ['balance_registered'],
            },
            {
                action => 'start',
                from =>
                  'balance_registered and not (working or deleted or blocked or need_approve or rejected or protected)',
                set_flags   => ['working'],
                reset_flags => ['stopped', 'testing']
            },
            {
                action => 'start_testing',
                from =>
'balance_registered and not (testing or working or deleted or blocked or need_approve or rejected or protected)',
                set_flags   => ['testing'],
                reset_flags => ['stopped', 'working'],
            },
            {
                action      => 'stop',
                from        => '(working or testing) and not (stopped or need_approve or rejected or protected)',
                set_flags   => ['stopped'],
                reset_flags => ['working', 'testing'],
            },
            {
                action    => 'delete',
                from      => 'not (working or testing or deleted)',
                set_flags => ['deleted']
            },
            {
                action      => 'restore',
                from        => 'deleted and not blocked',
                reset_flags => ['deleted'],
            },
            {
                action => 'edit',
                from   => 'not (deleted or blocked or rejected)'
            },
            {
                action => 'del_mirror',
                from   => 'not (deleted or blocked or rejected or protected)'
            },
            {
                action    => 'set_read_only',
                from      => 'not read_only',
                set_flags => ['read_only'],
            },
            {
                action      => 'reset_read_only',
                from        => 'read_only',
                reset_flags => ['read_only'],
            },
            {
                action      => 'set_blocked',
                from        => 'not blocked',
                set_flags   => ['blocked'],
                reset_flags => ['need_approve'],
            },
            {
                action      => 'reset_blocked',
                from        => 'blocked',
                reset_flags => ['blocked'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action => 'start_block',
                from   => '(working or testing) and not protected',
            },
            {
                action => 'restore_block',
                from   => 'not deleted',
            },
            {
                action      => 'set_need_approve',
                from        => 'rejected and not blocked',
                set_flags   => ['need_approve'],
                reset_flags => ['rejected'],
            },
            {
                action      => 'approve',
                from        => 'need_approve',
                reset_flags => ['need_approve'],
            },
            {
                action      => 'reject',
                from        => 'not rejected',
                reset_flags => ['need_approve'],
                set_flags   => ['rejected'],
            },
            {
                action => 'can_update_in_bk',
                from   => 'not (need_approve or rejected or deleted or protected)',
            },
            {
                action    => 'set_protected',
                from      => 'not protected',
                set_flags => ['protected'],
            },
            {
                action      => 'reset_protected',
                from        => 'protected',
                reset_flags => ['protected'],
            },
            {
                action    => 'set_need_update',
                from      => 'not (need_approve or rejected or deleted)',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update or updating',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ]
    };
}

sub pre_process_fields {
    my ($self, $fields, $result, %opts) = @_;

    my $ids      = $opts{'ids'}      //= array_uniq(map {$_->{'id'}      // ()} @$result);
    my $page_ids = $opts{'page_ids'} //= array_uniq(map {$_->{'page_id'} // ()} @$result);

    $self->SUPER::pre_process_fields($fields, $result, %opts);
    $self->Application::Model::Page::MIXIN::External::pre_process_fields($fields, $result, %opts);

    if ($fields->need('cookie_match')) {
        $fields->{'__TAGS__'} = {};

        my $tmp_rights = $self->app->add_tmp_rights(qw(cookie_match_view_all));

        foreach (
            @{
                $self->cookie_match->get_all(
                    fields => [qw(id tag data_key js_redir skip_data_key)],
                    filter => {id => array_uniq(map {$_->{'tag_id'} // ()} @$result)}
                )
            }
          )
        {
            $fields->{'__TAGS__'}{$_->{'id'}} = $_;
        }
    }
}

sub get_actions_depends {
    [qw(id multistate cur_user_is_read_assistant is_site_approved)];
}

sub get_editable_fields_depends {
    [qw(id multistate user_synchronization page_id owner_id  cur_user_is_read_assistant is_read_only)];
}

sub get_available_fields_depends {
    [
        qw(
          multistate
          owner_id
          user_synchronization
          )
    ];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    $self->handle_field_assistants($obj, \%fields, 'view_field__assistants');

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor
              . '_view_field__%s' => [qw(comment login reload_timeout domain_id block_title quality_coef is_tutby)],
            $accessor . '_view_field__owner'       => [qw(owner)],
            $accessor . '_view_field__domain_id'   => [qw(domain_id source_id)],
            $accessor . '_view_field__banner_lang' => [qw(banner_lang banner_lang_name)],
        }
    );

    delete($fields{'user_synchronization'})
      unless $self->check_short_rights('view_field__user_synchronization') && $obj->{'user_synchronization'}
          || $self->check_short_rights('edit_field__user_synchronization');

    # user_synchronization details (dependent fields) are only visible when user_synchronization is turned on
    unless ($obj->{'user_synchronization'}) {
        delete($fields{$_}) foreach (qw(tag_id cookie_match));
    }

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    $fields->{'domain_id'} = TRUE;

    $fields->{'login'} = TRUE if $self->check_short_rights('add_other');

    return $fields;
}

sub collect_editable_fields {
    my ($self, $obj) = @_;

    my $fields = {};
    $self->handle_field_assistants($obj, $fields, 'edit_field__assistants');

    return $fields;
}

sub _get_common_add_edit_fields {
    my ($self, $add_fields, $edit_fields) = @_;

    my $fields = $self->get_fields_by_right(
        res_fields      => $self->SUPER::_get_common_add_edit_fields(),
        no_right_fields => ['caption'],
    );

    return $fields;
}

sub hook_stash_add_fields {
    my ($self) = @_;

    my %fields = $self->SUPER::hook_stash_add_fields;

    push @{$fields{once}}, 'site';

    return %fields;
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    my $default_settings = $self->get_default_settings();
    my $settings = $self->hook_stash->mode('edit') ? $self->hook_stash->get('current') : {};
    for my $field (keys %$default_settings) {
        $opts->{$field} //= $settings->{$field} // $default_settings->{$field};
    }

    $self->SUPER::hook_fields_processing_before_validation($opts);
}

sub hook_fields_validation {
    my ($self, $opts) = @_;
    $self->SUPER::hook_fields_validation($opts);

    my %sites = map {$_->{'id'} => $_} @{$self->site->get_domain_for_adding($self->hook_stash->get('user')->{'id'})};
    my $site = $sites{$opts->{'domain_id'}} // throw Exception::Validation::BadArguments gettext('Unknown domain');

    throw Exception::Validation::BadArguments gettext('Domain %s is blocked', $site->{domain})
      if $site->{is_blocked};

    throw Exception::Validation::BadArguments gettext('Domain %s is rejected by moderation', $site->{domain})
      if ($site->{is_rejected}
        && ($self->get_moderation_timeout($site->{moderation_reason}) < 0));

    $self->hook_stash->set('site', $site);
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    my $site = $self->hook_stash->get('site');
    my $id   = $self->hook_stash->get('id');
    my $user = $self->hook_stash->get('user');

    if ($self->site->check_multistate_flag($site->{'multistate'}, 'approved')) {
        my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions(qw(approve)));

        $self->do_action($id, "approve");
    } elsif ($self->site->check_multistate_flag($site->{'multistate'}, 'rejected')) {
        my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions(qw(reject)));

        $self->do_action($id, "reject");
    } else {
        my $uri = URI->new("https://partner2.yandex.ru");
        $uri->path("/v2/sites/");
        $uri->query_form(search => to_json({domain => [LIKE => $site->{'domain'}]}));

        my $body = {
            context_on_site_campaign => d_gettext(
                'context_on_site_campaign__site_moderation_needed_email__login=%s,domain=%s,site_id=%s,url=%s',
                $user->{login}, $site->{'domain'}, $site->{'id'}, $uri
            ),
            search_on_site_campaign => d_gettext(
                'search_on_site_campaign__site_moderation_needed_email__login=%s,domain=%s,site_id=%s,url=%s',
                $user->{login}, $site->{'domain'}, $site->{'id'}, $uri
            ),
        };

        $self->mail_notification->add(
            type    => 0,
            user_id => 0,
            opts    => {
                check_alive => FALSE,
                subject     => gettext('New domain for moderation in PI2'),
                to          => $MODERATION_MAIL,
                values      => {
                    message_body       => $body->{$self->accessor()}->(),
                    plain_text_wrapper => TRUE,
                },
            },
        );
    }
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    my $id       = $self->hook_stash->get('id');
    my $settings = $self->hook_stash->get('settings');
    my $related  = $self->hook_stash->get('fields_from_related_models');
    my $mode     = $self->hook_stash->mode('add') ? 'add' : 'edit';

    my @to_update = qw(excluded_domains excluded_phones assistants);

    if (my $banner_lang = $related->{banner_lang}) {
        $self->db_banner_lang->delete($self->partner_db->filter({campaign_id => $id->{'id'}})) if $mode eq 'edit';
        $self->db_banner_lang->add(
            {
                campaign_id => $id->{'id'},
                language_id => $_,
            }
        ) foreach (@$banner_lang);
    }

    if (my $mirrors = $related->{mirrors}) {
        $self->mirrors->replace($id, $mirrors);
    }

    for my $model (@to_update) {
        if (defined(my $value = $related->{$model})) {
            $self->$model->replace($settings->{'page_id'}, $value, $mode);
        }
    }
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    if (   $self->need_update_in_bk($self->hook_stash->get('fields'))
        && $self->check_action($self->hook_stash->get('id'), 'set_need_update'))
    {
        $self->do_action($self->hook_stash->get('id'), 'set_need_update');
    }
}

sub can_action_reset_blocked {
    my ($self, $obj) = @_;

    my $object = $self->_get_object_fields($obj, [qw(is_site_approved)]);

    return $object->{'is_site_approved'};
}

sub on_action_set_blocked {
    my ($self, $obj, %opts) = @_;

    $self->do_action($obj, 'stop') if $self->check_action($obj, 'stop');

    my $object = $self->_get_object_fields($obj, [qw(caption owner_id page_id)]);
    $object->{accessor} = $self->accessor;
    unless ($opts{suppress_mail_notification}) {
        $self->app->mail_notification->add_when_blocking_page($object);
    }
}

sub query_filter {
    my ($self, $filter, $opts) = @_;

    $filter = $self->limit_filter_by_tutby_and_assistant_and_robot_assistant($filter, $opts);

    return $filter;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my $fields = $self->SUPER::get_need_update_in_bk_fields();

    $fields->{$_} = TRUE foreach (
        qw(
        banner_lang
        page_lang
        block_title
        excluded_domains
        excluded_phones
        family_filter
        no_mobile_app
        reload_timeout
        caption
        tag_id
        )
    );

    return $fields;
}

sub get_page_mirrors {
    my ($self, $page) = @_;
    my $mirrors = [
        map {$_->{'domain'}}
        grep {$self->mirrors->check_multistate_flag($_->{'multistate'}, 'approved')} @{$page->{mirrors} // []}
    ];
    return $mirrors;
}

sub get_bk_data {
    my ($self, $page) = @_;

    my $user_id = $page->{'owner_id'};

    my @user_global_excluded_phones =
      map {$_->{'phone'}} @{$self->user_global_excluded_phones->get_all(filter => {user_id => $user_id})};

    my @user_global_excluded_domains =
      map {$_->{'domain'}} @{$self->user_global_excluded_domains->get_all(filter => {user_id => $user_id})};

    my $excluded_domains = array_uniq(
        check_domains_for_absorption([@{$page->{'excluded_domains'}}, @user_global_excluded_domains]),
        bs_format_phone([@{$page->{'excluded_phones'}}, @user_global_excluded_phones])
    );

    my %data = (
        cpa              => 100,
        mirrors          => $self->get_page_mirrors($page),
        excluded_domains => [sort @$excluded_domains],
        is_graysite      => $page->{'is_graysite'} ? 1 : 0,
        (%{$page->{'cookie_match'}} ? (cookie_match_tag => $page->{'cookie_match'}{'tag'}) : ()),
    );

    return %data;
}

# API

sub api_available_actions {
    return qw();
}

sub api_can_edit {TRUE}

sub approve_for_site {
    my ($self, $domain_id) = @_;

    my $campaigns = $self->get_all(
        fields => [
            qw(
              id
              multistate
              cur_user_is_read_assistant
              is_site_approved
              domain
              caption
              client_id
              business_unit
              owner_id
              ),
            keys(%{$self->get_default_settings()})
        ],
        filter => {domain_id => $domain_id, multistate => 'need_approve and not protected'}
    );

    my %owner_has_not_deleted_pages;
    foreach (@$campaigns) {
        $self->do_action($_, 'approve')
          if $self->check_action($_->{'id'}, 'approve');

        $owner_has_not_deleted_pages{$_->{owner_id}} = TRUE
          unless $self->check_multistate_flag($_->{multistate}, 'deleted');
    }

    return \%owner_has_not_deleted_pages;
}

sub check_action {&Application::Model::Page::MIXIN::External::check_action}

sub reject_for_site {
    my ($self, $domain_id, %opts) = @_;

    my $protected_pages =
      $self->get_all(fields => [qw(id)], filter => {domain_id => $domain_id, multistate => 'protected'});
    WARN {
        message => sprintf("Site with protected pages is rejected. Protected pages: %s",
            join(', ', (map {$_->{id}} @$protected_pages))),
        fingerprint => [$self->accessor(), 'Site with protected pages is rejected']
      }
      if @$protected_pages;

    my $campaigns = $self->get_all(
        fields => [qw(id multistate cur_user_is_read_assistant is_site_approved owner_id caption)],
        filter => {domain_id => $domain_id, multistate => 'not rejected and not protected'}
    );

    my %owner_has_not_deleted_pages;
    foreach (@$campaigns) {
        $self->do_action($_, 'stop')
          if $self->check_action($_->{'id'}, 'stop');
        $self->do_action($_, 'reject', %opts)
          if $self->check_action($_->{'id'}, 'reject', %opts);

        $owner_has_not_deleted_pages{$_->{owner_id}} = TRUE
          unless $self->check_multistate_flag($_->{multistate}, 'deleted');
    }

    return \%owner_has_not_deleted_pages;
}

sub return_to_moderation_for_site {
    my ($self, $domain_id) = @_;

    my $campaigns = $self->get_all(
        fields => [qw(id multistate cur_user_is_read_assistant is_site_approved)],
        filter => {domain_id => $domain_id, multistate => 'rejected'}
    );

    foreach (@$campaigns) {
        $self->do_action($_, 'set_need_approve')
          if $self->check_action($_->{'id'}, 'set_need_approve');
    }
}

sub related_models {
    my ($self) = @_;

    return {
        %{$self->SUPER::related_models()},
        %{$self->external_related_models()},

        mirrors => {
            accessor => 'mirrors',
            filter   => sub {
                return {
                    campaign_id => array_uniq(map {$_->{'id'} // ()} @{$_[1]}),
                    multistate => 'not (deleted or rejected)'
                };
            },
            key_fields => ['campaign_id'],
            value_type => 'array',
        },
    };
}

TRUE;
