package Application::Model::Page::Site::Internal;

use qbit;

use base qw(
  Application::Model::Page::Site
  Application::Model::Page::MIXIN::Internal
  );

use PiConstants qw(
  $ADINSIDE_CLIENT_ID
  $INTERNAL_PAGE_OWNER_LOGIN
  $CONTEXT_TARGET_TYPE
  $SEARCH_TARGET_TYPE
  );

consume qw(
  Application::Model::Role::Has::Page::FilterSelfDomain
  Application::Model::Role::Has::Page::Patch
  );

use Utils::PhoneBK;
use Utils::DomainBK;

use Exception::Denied;
use Exception::Validation::BadArguments;

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        internal_site => 'Application::Model::InternalSite',
        api_yt_bk     => 'QBit::Application::Model::API::Yandex::YT::BK',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        %{$self->get_internal_structure_rights_to_register()},
        map {$self->get_description_right($_)}
          qw(
          view_field__block_title
          edit_field__block_title
          view_field__quality_coef
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        %{$self->get_internal_structure_model_fields()},
        site_id =>
          {default => TRUE, db => TRUE, label => d_gettext('Site id'), type => 'number', adjust_type => 'str',},
        domain_id => {
            #redefine
            db          => TRUE,
            label       => d_gettext('Domain ID'),
            db_expr     => 'site_id',
            type        => 'number',
            adjust_type => 'str',
        },
        caption => {
            depends_on => [qw(domain)],
            get        => sub {
                return $_[1]->{'caption'} || $_[1]->{'domain'};
            },
            type       => 'string',
            db         => TRUE,
            api        => 1,
            need_check => {len_max => 255,},
        },
        mirrors => {
            label      => d_gettext('Mirrors'),
            depends_on => ['id', 'mirrors.domain'],
            get        => sub {
                $_[0]->{'mirrors'}{$_[1]->{'id'}} // [];
            },
            type       => 'array',
            sub_type   => 'string',
            need_check => {
                type     => 'array',
                optional => TRUE,
            },
            api => 1,
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $filter = $self->SUPER::get_structure_model_filter();

    $filter->{'fields'} = {
        %{$filter->{'fields'}}, %{$self->get_internal_structure_model_filter()},
        site_id => {type => 'number', label => d_gettext('Site_id')},
        #redefine
        caption_text => {type => 'alias', path => [qw(domain domain)]},
        caption      => {
            type      => 'text',
            label     => d_gettext('Caption'),
            db_filter => sub {
                return $_[0]->partner_db->filter(
                    [
                        'site_id' => '= ANY' => $_[0]->site->query(
                            fields => $_[0]->site->_get_fields_obj(['id']),
                            filter => $_[0]->get_db_filter(['domain' => $_[1]->[1] => $_[1]->[2]])
                        )
                    ]
                );
            },
        },
        domain_id => {
            type      => 'number',
            label     => d_gettext('Site_id'),
            db_filter => sub {
                return ['site_id' => $_[1]->[1] => \$_[1]->[2]];
            },
        },
        domain => {
            type           => 'subfilter',
            model_accessor => 'site',
            field          => 'site_id',
            fk_field       => 'id',
            label          => d_gettext('Domains'),
        },
        mirror => {
            type           => 'subfilter',
            model_accessor => 'mirrors',
            field          => 'id',
            fk_field       => 'campaign_id',
            label          => d_gettext('Mirrors'),
        },
    };

    return $filter;
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [
            {name => 'managers.manager.login', label => gettext('Manager\'s login')},
            {name => 'id',                     label => gettext('Page ID')},
        ],
        [
            {name => 'all_domain', label => gettext('Domain and mirror')},
            {name => 'multistate', label => gettext('Status')},
        ],
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    my $accessor = $self->accessor();
    my ($rigth_prefix, $right_group) = $accessor =~ /((.+)_)campaign$/;

    return {
        empty_name  => 'New',
        multistates => [
            [working            => d_pgettext('Campaign status', 'Working')],
            [testing            => d_pgettext('Campaign status', 'Testing')],
            [stopped            => d_pgettext('Campaign status', 'Stopped')],
            [deleted            => d_pgettext('Campaign status', 'Archived')],
            [balance_registered => d_pgettext('Campaign status', 'Added to Balance')],
            [read_only          => d_pgettext('Campaign status', 'Read only')],
            [check_statistics   => d_pgettext('Campaign status', 'Check statistics')],
            [protected          => d_pgettext('Campaign status', 'Protected')],
            [need_update        => d_pgettext('Campaign status', 'Need update')],
            [updating           => d_pgettext('Campaign status', 'Updating')],
        ],
        actions => {
            can_update_in_bk       => d_pgettext('Campaign action', 'Can update in bk'),
            reset_check_statistics => d_pgettext('Campaign action', 'Reset "check_statistics"'),
            restore_block          => d_pgettext('Campaign action', 'Restore block on campaign'),
            set_check_statistics   => d_pgettext('Campaign action', 'Set "check_statistics"'),
            start_block            => d_pgettext('Campaign action', 'Start block on campaign'),
            del_mirror             => d_pgettext('Campaign action', 'Remove mirror'),
        },
        right_actions => {
            add                 => d_pgettext('Campaign action', 'Add'),
            bk_import_deleted   => d_pgettext('Campaign action', 'Import pages from BK with status deleted'),
            bk_import_testing   => d_pgettext('Campaign action', 'Import pages from BK with status testing'),
            bk_import_working   => d_pgettext('Campaign action', 'Import pages from BK with status working'),
            delete              => d_pgettext('Campaign action', 'Archive'),
            edit                => d_pgettext('Campaign action', 'Edit'),
            register_in_balance => d_pgettext('Campaign action', 'Register in Balance'),
            reset_protected     => d_pgettext('Campaign action', 'Reset protected'),
            reset_read_only     => d_pgettext('Campaign action', 'Reset read only'),
            restore             => d_pgettext('Campaign action', 'Restore'),
            set_need_update =>
              {label => d_pgettext('Campaign action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            set_protected => d_pgettext('Campaign action', 'Set protected'),
            set_read_only => d_pgettext('Campaign action', 'Set read only'),
            start         => d_pgettext('Campaign action', 'Start'),
            start_testing => d_pgettext('Campaign action', 'Start testing'),
            start_update => {label => d_pgettext('Campaign action', 'Start update'), dont_write_to_action_log => TRUE},
            stop         => d_pgettext('Campaign action', 'Stop'),
            stop_testing => d_pgettext('Campaign action', 'Stop testing'),
            stop_update => {label => d_pgettext('Campaign action', 'Stop update'), dont_write_to_action_log => TRUE},
        },
        right_group => [$right_group => d_gettext('Right to manage internal context campaigns')],
        right_name_prefix =>
          $rigth_prefix,    # TODO: переименовать в $self->accessor . '_' (не хватает "campaign")
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'register_in_balance',
                from      => 'not balance_registered and not deleted',
                set_flags => ['balance_registered'],
            },
            {
                action      => 'start',
                from        => 'balance_registered and not working and not deleted and not protected',
                set_flags   => ['working'],
                reset_flags => ['stopped']
            },
            {
                action      => 'start_testing',
                from        => 'stopped and not protected',
                set_flags   => ['testing'],
                reset_flags => ['stopped'],
            },
            {
                action      => 'stop_testing',
                from        => 'testing and not protected',
                set_flags   => ['stopped'],
                reset_flags => ['testing'],
            },
            {
                action      => 'stop',
                from        => '(working or testing) and not protected and not stopped',
                set_flags   => ['stopped'],
                reset_flags => ['working'],
            },
            {
                action    => 'delete',
                from      => 'not (working or testing) and not deleted',
                set_flags => ['deleted']
            },
            {
                action      => 'restore',
                from        => 'deleted and not protected',
                reset_flags => ['deleted', 'stopped', 'testing'],
                set_flags   => ['working'],
            },
            {
                action => 'edit',
                from   => 'not deleted'
            },
            {
                action => 'del_mirror',
                from   => 'not deleted and not protected'
            },
            {
                action    => 'set_read_only',
                from      => 'not read_only',
                set_flags => ['read_only'],
            },
            {
                action      => 'reset_read_only',
                from        => 'read_only',
                reset_flags => ['read_only'],
            },
            {
                action    => 'set_protected',
                from      => 'not protected',
                set_flags => ['protected'],
            },
            {
                action      => 'reset_protected',
                from        => 'protected',
                reset_flags => ['protected'],
            },
            {
                action    => 'set_check_statistics',
                from      => '__EMPTY__',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action => 'start_block',
                from   => '(working or testing) and not protected',
            },
            {
                action => 'restore_block',
                from   => 'not deleted',
            },
            {
                action      => 'bk_import_working',
                from        => 'not working',
                set_flags   => ['working'],
                reset_flags => ['testing', 'deleted'],
            },
            {
                action      => 'bk_import_testing',
                from        => 'not testing',
                set_flags   => ['testing'],
                reset_flags => ['working', 'deleted'],
            },
            {
                action      => 'bk_import_deleted',
                from        => 'not deleted',
                set_flags   => ['deleted'],
                reset_flags => ['working', 'testing'],
            },
            {
                action => 'can_update_in_bk',
                from   => 'not (deleted or protected)',
            },
            {
                action    => 'set_need_update',
                from      => 'not (deleted)',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update or updating',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ]
    };
}

sub get_actions_depends {
    [qw(id multistate is_my)];
}

sub get_available_fields_depends {
    [qw(multistate)];
}

sub get_editable_fields_depends {
    [qw(id multistate is_my is_read_only)];
}

sub pre_process_fields {
    my ($self, $fields, $result, %opts) = @_;

    $self->SUPER::pre_process_fields($fields, $result, %opts);
    $self->Application::Model::Page::MIXIN::Internal::pre_process_fields($fields, $result, %opts);
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s'          => [qw(comment block_title quality_coef)],
            $accessor . '_view_field__managers'    => [qw(managers manager_ids)],
            $accessor . '_view_field__banner_lang' => [qw(banner_lang banner_lang_name)],
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    $fields->{'domain'} = TRUE;

    return $fields;
}

sub collect_editable_fields {{}}

sub hook_stash_add_fields {
    my ($self) = @_;

    my %fields = $self->SUPER::hook_stash_add_fields;

    push @{$fields{once}}, 'page_options';

    return %fields;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    my $busy_domain = $self->internal_site->get_all(filter => {domain => $opts->{domain}});

    my $site_id;
    if (@$busy_domain) {
        $site_id = $busy_domain->[0]{'id'};
    } else {
        $site_id = $self->internal_site->add($opts->{domain});
    }
    $opts->{site_id} = $site_id;
    $opts->{create_date} = curdate(oformat => 'db_time');
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    my $id = $self->hook_stash->get('id');

    $self->do_action($id, 'add', %$opts);
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    my $id      = $self->hook_stash->get('id');
    my $related = $self->hook_stash->get('fields_from_related_models');
    my $mode    = $self->hook_stash->mode('add') ? 'add' : 'edit';

    my @to_update = qw(excluded_domains excluded_phones assistants managers);

    if (my $banner_lang = $related->{banner_lang}) {
        $self->db_banner_lang->delete($self->partner_db->filter({campaign_id => $id->{'id'}})) if $mode eq 'edit';
        $self->db_banner_lang->add(
            {
                campaign_id => $id->{'id'},
                language_id => $_,
            }
        ) foreach (@$banner_lang);
    }

    if (my $mirrors = $related->{mirrors}) {
        $self->mirrors->replace($id, $mirrors);
    }

    for my $model (@to_update) {
        if (defined(my $value = $related->{$model})) {
            $self->$model->replace($id->{'id'}, $value, $mode);
        }
    }
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    if ($self->need_update_in_bk($self->hook_stash->get('fields'))) {
        $self->do_action($self->hook_stash->get('id'), 'set_need_update');
    }
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_managers_and_robot_assistant($filter);

    return $filter;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my $fields = $self->SUPER::get_need_update_in_bk_fields();

    $fields->{$_} = TRUE foreach (
        qw(
        caption
        banner_lang
        page_lang
        block_title
        excluded_domains
        excluded_phones
        family_filter
        no_mobile_app
        reload_timeout
        )
    );
    return $fields;
}

sub get_page_mirrors {
    my ($self, $page) = @_;
    my $mirrors = [@{$page->{'mirrors'} // []}];
    return $mirrors;
}

sub get_bk_data {
    my ($self, $page) = @_;

    my $excluded_domains =
      array_uniq(check_domains_for_absorption($page->{'excluded_domains'}),
        bs_format_phone($page->{'excluded_phones'}));

    my %data = (
        cpa              => 100,
        mirrors          => $self->get_page_mirrors($page),
        excluded_domains => [sort @$excluded_domains],
    );

    return %data;
}

sub check_action {&Application::Model::Page::MIXIN::Internal::check_action}

sub related_models {
    my ($self) = @_;

    return {
        %{$self->SUPER::related_models()},
        %{$self->internal_related_models()},

        mirrors => {
            accessor => 'mirrors',
            filter   => sub {
                return {campaign_id => array_uniq(map {$_->{'id'} // ()} @{$_[1]})};
            },
            key_fields => ['campaign_id'],
            value_type => 'array_domain',
        },
    };
}

TRUE;
