package Application::Model::Page::Video;

use qbit;

use base qw(
  Application::Model::Page
  Application::Model::Page::MIXIN::External
  RestApi::MultistateModel
  Application::Model::ValidatableMixin
  );

consume qw(
  Application::Model::Role::Has::Filters
  Application::Model::Role::Has::Tier
  Application::Model::Role::Has::UnmoderatedRtbAuction
  Application::Model::Role::Page::Has::AllowDirectVASTRequest
  Application::Model::Role::Page::Has::Assessor
  Application::Model::Role::Page::Has::BlocksLimit
  Application::Model::Role::Page::Has::MetricaCounters
  Application::Model::Role::Page::Has::MoneyMap::VideoResources
  Application::Model::Role::Page::Has::PartnerReadOnly
  Application::Model::Role::Page::Has::Tags
  Application::Model::Role::Has::Page::FilterSelfDomain
  Application::Model::Role::Has::Page::Patch
  Application::Model::Role::Has::ModerationReason::Page
  );

use Utils::DomainBK;
use Utils::PhoneBK;

use Utils::JSON qw(fix_type_for_complex);

use PiConstants qw(
  $ADINSIDE_CLIENT_ID
  $BALANCE_VIPTYPE_YANDEX
  $CONTEXT_TARGET_TYPE
  $DEFAULT_VIDEO_SKIN
  $DSP_DIRECT_ID
  $VIDEO_PARTNER_ROLE_ID
  $VIDEO_PLATFORM_FLASH
  $VIDEO_PLATFORM_HTML5
  $VIDEO_PLATFORM_MOBILE
  $VIDEO_PLATFORM_SMART_TV
  $VIDEO_PLATFORMS
  );

use Exception;
use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

my $PARTNER_TYPES = {
    'kp+rsya'    => d_gettext('Kinopoisk and YAN'),
    'rsya'       => d_gettext('YAN'),
    'kp'         => d_gettext('Kinopoisk'),
    'yahosting'  => d_gettext('Yahosting'),
    'kp_partner' => d_gettext('Kinopoisk partner'),
};

my $VALID_DELAYS = [2, 5, 10, 15, 60, 180];

sub accessor               {'video_an_site'}
sub db_table_name          {'video_an_site'}
sub get_opts_schema_name   {'video_an_site_opts'}
sub get_product_name       {gettext('video_an_site')}
sub get_page_id_field_name {'id'}

sub get_block_model_names {
    my ($self) = @_;
    return [map {"video_an_site_$_"} qw(instream inpage fullscreen)];
}
sub block_seq_db_table {$_[0]->partner_db->video_an_site_block_seq}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        video_an_site_instream   => 'Application::Model::Product::VideoAN::Site::InStream',
        video_an_site_inpage     => 'Application::Model::Product::VideoAN::Site::InPage',
        video_an_site_fullscreen => 'Application::Model::Product::VideoAN::Site::Fullscreen',
        video_an_site_mirrors    => 'Application::Model::Product::VideoAN::Site::Video::Mirrors',
        video_an_site_blockings  => 'Application::Model::Product::VideoAN::Site::Video::Blockings',
        video_stat_files         => 'Application::Model::Product::VideoAN::Site::Video::StatFiles',
        agreement_checker        => 'Application::Model::AgreementChecker',
        video_an_site_categories => 'Application::Model::Product::VideoAN::Site::Categories',
        mirrors                  => 'QBit::Application::Model',
        cookie_match             => 'Application::Model::CookieMatch',
        block_dsps               => 'Application::Model::Block::DSPs',
        video_scenaries          => 'Application::Model::VideoScenaries',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        %{$self->get_external_structure_rights_to_register()},
        $self->get_description_right('add_adfox_block', d_gettext('Right to add adfox video blocks')),
        map {$self->get_description_right($_)}
          qw(
          edit_field__partner_type
          edit_field__settings_articles
          view_field__bk_stat
          view_field__user_synchronization
          edit_field__user_synchronization
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        %{$self->get_external_structure_model_fields()},
        available_blocks => {
            depends_on => [qw(multistate  platform block_types_on_page categories)],
            get        => sub {
                return $_[0]->model->get_available_block_model_names($_[1]);
            },
            type     => 'array',
            sub_type => 'string',
            api      => 1
        },
        domain => {
            default    => TRUE,
            db         => TRUE,
            label      => d_gettext('Domain'),
            type       => 'string',
            api        => 1,
            need_check => {
                len_max => 255,
                check   => sub {
                    my ($qv, $domain) = @_;
                    throw Exception::Validator::Fields gettext('Invalid domain')
                      unless get_domain($domain);
                },
            },
        },
        platform => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                check => sub {
                    my ($qv, $platform) = @_;
                    #throw Exception gettext('Type must be defined') unless defined($type);

                    throw Exception::Validator::Fields gettext('Unknown platform with id = "%s"', $platform)
                      unless exists($VIDEO_PLATFORMS->{$platform});
                },
            },
            adjust_type => 'str',
        },
        skin => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                check    => sub {
                    my ($qv, $value) = @_;
                    if ($value ne '') {
                        my $urls = $qv->app->partner_db->mds->get_all(filter => {url => [$value]});
                        throw Exception::Validator::Fields gettext("Can't use url %s for skin", $value)
                          if @$urls == 0;
                    }
                },
            },
        },
        skin_file_name => {
            depends_on => ['skin'],
            get        => sub {
                return '' if !defined($_[1]->{'skin'}) || $_[1]->{'skin'} eq '';
                return $_[0]->{'__SKINS__'}->{$_[1]->{'skin'}};
            },
            type => 'string',
            api  => 1
        },
        title => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {len_max => 255,},
        },
        skip_delay => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                check => sub {
                    my ($qv, $value) = @_;

                    throw Exception::Validator::Fields gettext('For "%s" select one of the values: %s', 'skip_delay',
                        join(', ', @$VALID_DELAYS))
                      if $qv->data->{'skip_time_left_show'} && !in_array($value, $VALID_DELAYS);
                },
            },
            adjust_type => 'str',
        },
        skip_time_left_show => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean',},
            adjust_type => 'str',
        },
        time_left_show => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean',},
            adjust_type => 'str',
        },
        skin_timeout => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'int_un',
                min      => 1,
                max      => 10_000,
            },
            adjust_type => 'str',
        },
        vast_version => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'int_un',
                min      => 2,
                max      => 3,
            },
            adjust_type => 'str',
        },
        vast_timeout => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'int_un',
                min      => 1,
                max      => 10_000,
            },
            adjust_type => 'str',
        },
        video_timeout => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'int_un',
                min      => 1,
                max      => 10_000,
            },
            adjust_type => 'str',
        },
        buffer_full_timeout => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'int_un',
                min      => 1,
                max      => 10_000,
            },
            adjust_type => 'str',
        },
        buffer_empty_limit => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'int_un',
                min      => 1,
                max      => 20,
            },
            adjust_type => 'str',
        },
        vpaid_enabled => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean', optional => TRUE,},
            adjust_type => 'str',
        },
        vpaid_timeout => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                check    => sub {
                    my ($qv, $value) = @_;
                    throw Exception::Validator::Fields gettext('Must be a whole number from 1 to 10000')
                      if $qv->data->{'vpaid_enabled'} && ($value !~ /^[0-9]+$/ || $value < 1 || $value > 10_000);
                },
            },
            adjust_type => 'str',
        },
        wrapper_timeout => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'int_un',
                min      => 1,
                max      => 10_000,
            },
            adjust_type => 'str',
        },
        wrapper_max_count => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'int_un',
                in       => [0, 1, 2, 3, 4, 5],
            },
            adjust_type => 'str',
        },
        partner_type => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                in       => [sort keys %$PARTNER_TYPES],
            },
        },
        pixels => {
            db         => TRUE,
            type       => 'string',
            need_check => {
                optional => TRUE,
                type     => 'pixels',
            },
            get => sub {
                return from_json($_[1]->{'pixels'} // '{}');
            },
        },
        partner_type_label => {
            depends_on => ['partner_type'],
            get        => sub {
                defined($_[1]->{'partner_type'}) ? $PARTNER_TYPES->{$_[1]->{'partner_type'}}->() : '';
              }
        },
        categories => {
            label      => d_gettext('Categories'),
            depends_on => [
                'id',                   'partner_type',    'categories.parent_id', 'categories.id',
                'categories.parent_id', 'categories.name', 'categories.comment',   'categories.link',
                'categories.archived',
            ],
            get => sub {
                my %ignore_categories = ();

                if (defined($_[1]->{'partner_type'})) {
                    if ($_[1]->{'partner_type'} ne 'yahosting') {
                        %ignore_categories =
                          map {$_ => TRUE} $_[0]->model->video_an_site_categories->get_category_ids_for_yahosting();
                    }
                } else {
                    %ignore_categories =
                      map {$_ => TRUE}
                      $_[0]->model->video_an_site_categories->get_category_ids_for_yandex_video_search(),
                      $_[0]->model->video_an_site_categories->get_category_ids_for_yahosting();
                }

                my @res = ();
                foreach my $el (@{$_[0]->{'categories'}{$_[1]->{'id'}} // []}) {
                    next if $ignore_categories{$el->{'id'}};

                    delete($el->{'page_id'});

                    push(@res, $el);
                }

                return \@res;
            },
            type       => 'array',
            sub_type   => 'video_an_site_categories',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'array',
                check    => sub {
                    my ($qv, $value) = @_;

                    throw Exception::Validator::Fields gettext("No category with id = 0")
                      unless in_array(0, [map {$_->{id}} @$value]);

                    if (my $categories = $qv->data->{categories}) {
                        my @missing_ids = @{arrays_difference([map {$_->{id}} @$categories], [map {$_->{id}} @$value])};

                        throw Exception::Validator::Fields gettext("Missing category ids %s", join(', ', @missing_ids))
                          if @missing_ids;
                    }
                },
            },
        },
        blockings => {
            label      => d_gettext('Blockings'),
            depends_on => ['id', 'blockings.content_id', 'blockings.name', 'blockings.type',],
            get        => sub {
                my $blockings = $_[0]->{'blockings'}{$_[1]->{'id'}} // [];

                foreach my $el (@$blockings) {
                    delete($el->{'site_id'});
                }

                return $blockings;
            },
            need_check => {
                optional => TRUE,
                type     => 'array',
            },
            type     => 'array',
            sub_type => 'video_an_site_blockings',
            api      => 1
        },
        platform_caption => {
            depends_on => 'platform',
            label      => d_gettext('Platform caption'),
            get        => sub {
                $VIDEO_PLATFORMS->{$_[1]->{'platform'}}{'label'}();
            },
            type => 'string',
            api  => 1
        },
        mirrors => {
            label      => d_gettext('Mirrors'),
            depends_on => ['id', 'mirrors.domain'],
            get        => sub {
                $_[0]->{'mirrors'}{$_[1]->{'id'}} // [];
            },
            type       => 'array',
            sub_type   => 'string',
            need_check => {
                type     => 'array',
                optional => TRUE,
            },
            api => 1,
        },
        bk_stat => {
            label        => d_gettext('BK stat'),
            check_rights => 'video_an_site_view_field__bk_stat',
            depends_on   => ['id'],
            get          => sub {
                $_[0]->{'__VIDEO_STAT_FILES__'}->{$_[1]->{'id'}} || [];
            },
            type        => 'array',
            sub_type    => 'video_stat_files',
            api         => 1,
            adjust_type => 'array_hash_str',
        },
        product_type => {
            get => sub {
                return 'video';
            },
        },
        data_pixels => {
            depends_on => [qw(pixels)],
            label      => d_gettext('Data pixels'),
            get        => sub {
                $_[1]->{pixels};
            },
            type => 'complex',
            api  => 1
        },
        block_types_on_page => {
            depends_on => [qw(page_id)],
            get        => sub {
                return $_[0]->{'__BLOCK_TYPES_ON_PAGE__'}{$_[1]->{'page_id'}} // {};
              }
        },
        user_synchronization => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {optional => TRUE,},
            adjust_type => 'str',
        },
        tag_id => {
            db         => TRUE,
            type       => 'number',
            api        => 1,
            need_check => {
                optional => TRUE,
                check    => sub {
                    my ($qv, $value) = @_;
                    my %tag_ids =
                      map {$_->{'id'} => TRUE} @{$qv->app->cookie_match->get_by_user_id($qv->data->{owner_id})};

                    throw Exception::Validator::Fields gettext('Tag "%s" not found', $value)
                      unless $tag_ids{$value};
                },
            },
            adjust_type => 'str',
        },
        cookie_match => {
            label      => d_gettext('Tag'),
            depends_on => ['tag_id'],
            get        => sub {
                return defined($_[1]->{'tag_id'}) ? $_[0]->{'__TAGS__'}->{$_[1]->{'tag_id'}} // {} : {};
            },
            type => 'cookie_match',
            api  => 1,
        },
        caption => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {len_max => 255,},
        },
        vmaps => {
            label      => d_gettext('Page VMaps'),
            depends_on => [
                'id',                      'vmaps.id',
                'vmaps.caption',           'vmaps.single_ad_session',
                'vmaps.use_better_places', 'vmaps.ad_session_interval',
                'vmaps.multistate'
            ],
            get => sub {
                my ($self, $obj) = @_;
                return $self->{'vmaps'}{$obj->{'id'}} // [];
            },
            type       => 'complex',
            need_check => {
                type     => 'array',
                optional => TRUE,
            },
            api => 1,
        },
        source_id => {db => TRUE, type => 'number', db_expr => 'id', api => 1, adjust_type => 'str',},
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $filter = $self->SUPER::get_structure_model_filter();

    $filter->{'fields'} = {
        %{$filter->{'fields'}},
        %{$self->get_external_structure_model_filter()},
        domain      => {type => 'text',  label => d_gettext('Domain')},
        domain_text => {type => 'alias', path  => [qw(domain)]},
        platform    => {
            type   => 'dictionary',
            label  => gettext('Platforms'),
            values => sub {
                [
                    map {
                        {hash_transform($_, ['id', 'label'])}
                      } @{$_[0]->get_platforms()}
                ];
              }
        },
        page_id => {
            type      => 'number',
            label     => d_gettext('Page ID'),
            db_filter => sub {
                return ['id' => $_[1]->[1] => \$_[1]->[2]];
            },
        },
        source_id     => {type => 'alias',  path  => [qw(page_id)]},
        tag_id        => {type => 'number', label => d_gettext('Tag ID')},
        vpaid_enabled => {
            type  => 'boolean',
            label => d_gettext('VPAID/MRAID support')
        },
    };

    return $filter;
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        (
            $self->check_rights('view_search_filters__login')
            ? ({name => 'owner.login', label => gettext('Login'), login_filter => 'video_an_partner=1'})
            : ()
        ),
        (
            $self->check_rights('view_search_filters__assistants_login')
            ? {name => 'assistants.user.login', label => gettext('Assistant\'s login')}
            : ()
        ),
        (
            $self->check_rights('view_search_filters__user_type') ? {name => 'user_type', label => gettext('User type')}
            : ()
        ),
        {name => 'id',         label => gettext('ID')},
        {name => 'page_id',    label => gettext('Page ID')},
        {name => 'platform',   label => gettext('Platforms')},
        {name => 'caption',    label => gettext('Caption')},
        {name => 'domain',     label => gettext('Resource')},
        {name => 'multistate', label => gettext('Status')}
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => 'New',
        multistates => [
            [working          => d_pgettext('Application status', 'Working')],
            [testing          => d_pgettext('Application status', 'Testing')],
            [stopped          => d_pgettext('Application status', 'Stopped')],
            [deleted          => d_pgettext('Application status', 'Archived')],
            [check_statistics => d_pgettext('Application status', 'No statistics')],
            [protected        => d_pgettext('Application status', 'Protected')],
            [need_update      => d_pgettext('Application status', 'Need update')],
            [updating         => d_pgettext('Application status', 'Updating')],
            [need_approve     => d_pgettext('Campaign status',    'Need approve')],
            [rejected         => d_pgettext('Campaign status',    'Rejected')],
            [blocked          => d_pgettext('Application status', 'Blocked')],
        ],
        actions => {
            start_block      => d_pgettext('Application action', 'Start block on site'),
            restore_block    => d_pgettext('Application action', 'Restore block on site'),
            can_update_in_bk => d_pgettext('Application action', 'Can update in bk'),
        },
        right_group       => [video_an_site => d_gettext('Right to manage video an sites')],
        right_name_prefix => $self->accessor() . '_',
        right_actions     => {
            add                    => d_pgettext('Application action', 'Add'),
            delete                 => d_pgettext('Application action', 'Archive'),
            restore                => d_pgettext('Application action', 'Restore'),
            start                  => d_pgettext('Application action', 'Start'),
            stop                   => d_pgettext('Application action', 'Stop'),
            edit                   => d_pgettext('Application action', 'Edit'),
            start_testing          => d_pgettext('Application action', 'Start testing'),
            stop_testing           => d_pgettext('Application action', 'Stop testing'),
            set_check_statistics   => d_pgettext('Application action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Application action', 'Reset "check_statistics"'),
            add_blocking           => d_pgettext('Application action', 'Add blocking'),
            del_mirror             => d_pgettext('Application action', 'Remove mirror'),
            del_blocking           => d_pgettext('Application action', 'Remove blocking'),
            set_protected          => d_pgettext('Application action', 'Set protected'),
            reset_protected        => d_pgettext('Application action', 'Reset protected'),
            set_need_update =>
              {label => d_pgettext('Application action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update =>
              {label => d_pgettext('Application action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update => {label => d_pgettext('Application action', 'Stop update'), dont_write_to_action_log => TRUE},
            set_need_approve => d_pgettext('Application action', 'Set "need approve"'),
            reject           => d_pgettext('Application action', 'Reject'),
            approve          => d_pgettext('Application action', 'Approve'),
            set_blocked      => d_pgettext('Application action', 'Set blocked'),
            reset_blocked    => d_pgettext('Application action', 'Reset blocked'),
        },
        multistate_actions => [
            {
                action    => 'add',
                from      => '__EMPTY__',
                set_flags => ['stopped'],
            },
            {
                action      => 'start_testing',
                from        => 'stopped and not (deleted or protected or blocked or need_approve or rejected)',
                set_flags   => ['testing'],
                reset_flags => ['stopped'],
            },
            {
                action      => 'stop_testing',
                from        => 'testing and not protected',
                set_flags   => ['stopped'],
                reset_flags => ['testing'],
            },
            {
                action      => 'start',
                from        => 'testing and not (protected or blocked or need_approve or rejected)',
                set_flags   => ['working'],
                reset_flags => ['testing']
            },
            {
                action      => 'stop',
                from        => 'working and not (stopped or protected)',
                set_flags   => ['stopped'],
                reset_flags => ['working'],
            },
            {
                action    => 'delete',
                from      => 'not (working or testing or deleted)',
                set_flags => ['deleted']
            },
            {
                action      => 'restore',
                from        => 'deleted and not (protected or blocked)',
                reset_flags => ['deleted', 'stopped', 'working'],
                set_flags   => ['testing'],
            },
            {
                action => 'edit',
                from   => 'not (deleted or blocked)'
            },
            {
                action    => 'set_check_statistics',
                from      => '__EMPTY__',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action => 'start_block',
                from   => '(working or testing) and not protected',
            },
            {
                action => 'restore_block',
                from   => 'not (deleted or blocked)',
            },
            {
                action => 'can_update_in_bk',
                from   => 'not (deleted or protected)',
            },
            {
                action    => 'set_protected',
                from      => 'not protected',
                set_flags => ['protected'],
            },
            {
                action      => 'reset_protected',
                from        => 'protected',
                reset_flags => ['protected'],
            },
            {
                action    => 'set_need_update',
                from      => 'not (deleted)',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update or updating',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
            {
                action    => 'set_blocked',
                from      => 'not blocked',
                set_flags => ['blocked'],
            },
            {
                action      => 'reset_blocked',
                from        => 'blocked',
                reset_flags => ['blocked'],
            },
            {
                action      => 'set_need_approve',
                from        => 'not need_approve',
                reset_flags => ['rejected'],
                set_flags   => ['need_approve'],
            },
            {
                action      => 'approve',
                from        => 'need_approve',
                reset_flags => ['need_approve'],
            },
            {
                action      => 'reject',
                from        => 'not rejected',
                reset_flags => ['need_approve'],
                set_flags   => ['rejected'],
            },
          ]

      }
}

sub pre_process_fields {
    my ($self, $fields, $result, %opts) = @_;

    $self->SUPER::pre_process_fields($fields, $result, %opts);
    $self->Application::Model::Page::MIXIN::External::pre_process_fields($fields, $result, %opts);

    if ($fields->need('skin_file_name')) {

        my $data = $self->partner_db->mds->get_all(fields => [qw(url file_name)],);

        my %h = map {$_->{url} => $_->{file_name}} @$data;

        $fields->{'__SKINS__'} = \%h;
    }

    if ($fields->need('bk_stat')) {
        $fields->{'__VIDEO_STAT_FILES__'} = {};

        foreach (
            @{
                $self->video_stat_files->get_all(
                    fields => [qw(id page_id)],
                    filter => {page_id => array_uniq(map {$_->{'id'} // ()} @$result),},
                    order_by => [['date', TRUE]]
                )
            }
          )
        {
            $fields->{'__VIDEO_STAT_FILES__'}{$_->{'page_id'}} //= [];
            push(@{$fields->{'__VIDEO_STAT_FILES__'}{$_->{'page_id'}}}, $_);
        }
    }

    if ($fields->need('block_types_on_page')) {
        $fields->{'__BLOCK_TYPES_ON_PAGE__'} = $self->get_block_types_on_pages([map {$_->{'page_id'}} @$result]) // {};
    }

    if ($fields->need('cookie_match')) {
        $fields->{'__TAGS__'} = {};

        my $tmp_rights = $self->app->add_tmp_rights(qw(cookie_match_view_all));

        foreach (
            @{
                $self->cookie_match->get_all(
                    fields => [qw(id tag data_key js_redir skip_data_key)],
                    filter => {id => array_uniq(map {$_->{'tag_id'} // ()} @$result)}
                )
            }
          )
        {
            $fields->{'__TAGS__'}{$_->{'id'}} = $_;
        }
    }
}

sub get_actions_depends {
    [qw(id multistate cur_user_is_read_assistant)];
}

sub get_available_fields_depends {
    [qw(multistate partner_type user_synchronization owner_id)];
}

sub get_editable_fields_depends {
    [qw(id multistate partner_type cur_user_is_read_assistant user_synchronization owner_id)];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);
    delete($fields{'blockings'});

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s'           => [qw(comment login bk_stat is_tutby)],
            $accessor . '_view_field__owner'        => [qw(owner client_id)],
            $accessor . '_view_field__partner_type' => [qw(partner_type partner_type_label)],
        }
    );

    $self->handle_field_assistants($obj, \%fields, 'view_field__assistants');

    delete(@fields{'pixels', 'data_pixels'}) unless defined($obj->{'partner_type'});

    delete($fields{'user_synchronization'})
      unless $self->check_short_rights('view_field__user_synchronization') && $obj->{'user_synchronization'}
          || $self->check_short_rights('edit_field__user_synchronization');

    # user_synchronization details (dependent fields) are only visible when user_synchronization is turned on
    unless ($obj->{'user_synchronization'}) {
        delete($fields{$_}) foreach (qw(tag_id cookie_match));
    }

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    foreach (qw(domain platform)) {
        $fields->{$_} = TRUE;
    }

    if ($self->check_short_rights('add_other')) {
        $fields->{login} = TRUE;
    } else {
        delete $fields->{login};
    }

    return $fields;
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my $res = $self->_get_common_add_edit_fields($data);

    $res->{$_} = TRUE foreach (
        qw(
        skin_timeout
        vast_timeout
        video_timeout
        buffer_full_timeout
        wrapper_timeout
        buffer_empty_limit
        vpaid_timeout
        vast_version
        vpaid_enabled
        wrapper_max_count
        categories
        excluded_domains
        excluded_phones)
    );

    $res->{'pixels'} = TRUE if defined($data->{'partner_type'});

    $res->{'mirrors'} = TRUE if $self->check_rights('do_video_an_site_del_mirror');

    foreach (qw(user_synchronization)) {
        $res->{$_} = TRUE if $self->check_short_rights("edit_field__$_");
    }

    $self->handle_field_assistants($data, $res, "edit_field__assistants");

    if (   $self->check_short_rights('edit_field__user_synchronization')
        || $data->{'user_synchronization'})
    {
        $res->{'tag_id'} = TRUE;
    }

    if ($self->check_multistate_flag($data->{'multistate'}, 'protected')
        && !$self->check_rights('edit_protected_pages'))
    {
        foreach (qw(mirrors blockings), keys(%{$self->get_need_update_in_bk_fields()})) {
            delete($res->{$_});
        }
    }

    return $res;
}

sub _get_common_add_edit_fields {
    my ($self, $data) = @_;

    my $fields = $self->get_fields_by_right(
        res_fields      => $self->SUPER::_get_common_add_edit_fields(),
        no_right_fields => [
            qw(
              caption
              skin
              skip_delay
              skip_time_left_show
              time_left_show
              title
              vpaid_enabled
              vpaid_timeout
              )
        ],
        right_fields => {edit => ['partner_type']}
    );

    return $fields;
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    $opts->{'skin'} //= '';

    if (exists $opts->{'skip_time_left_show'}) {
        unless ($opts->{'skip_time_left_show'}) {
            $opts->{'skip_delay'} = 0;
        }
    } else {
        delete $opts->{'skip_delay'};
    }

    if (exists $opts->{'vpaid_enabled'}) {
        unless ($opts->{'vpaid_enabled'}) {
            $opts->{'vpaid_timeout'} = 0;
        }
    } else {
        delete $opts->{'vpaid_timeout'};
    }

    $self->SUPER::hook_fields_processing_before_validation($opts);
}

sub hook_fields_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_validation($opts);

    my $roles = $self->rbac->get_roles_by_user_id($self->hook_stash->get('user')->{'id'});
    throw Exception::Validation::BadArguments gettext("Username must belong to the video partner")
      unless $roles->{$VIDEO_PARTNER_ROLE_ID};
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $opts->{categories} = $self->video_an_site_categories->get_default_video_categories($opts->{'domain'});
}

sub update_block_dsps {
    my ($self, $obj, $vpaid_enabled) = @_;

    my $page_id = $obj->{id};

    if (!$vpaid_enabled) {

        my $res = $self->partner_db->block_dsps->get_all(
            fields => {'block_id' => 'block_id', 'dsp_id'     => 'dsp_id'},
            filter => {'page_id'  => $page_id,   'is_deleted' => 0},
        );

        my %blocks_with_non_direct_dsp;
        foreach (@$res) {
            $blocks_with_non_direct_dsp{$_->{block_id}} += $_->{dsp_id} == $DSP_DIRECT_ID ? 0 : 1;
        }
        throw Exception::Validation::BadArguments gettext(
            'Ads cannot be displayed within the current settings. For example, enable VPAID')
          if grep {$_ == 0} (values %blocks_with_non_direct_dsp);
    }

    my %fields = map {$_->{name} => \undef} @{$self->partner_db->block_dsps->fields};
    $fields{page_id}    = 'page_id';
    $fields{block_id}   = 'id';
    $fields{dsp_id}     = \$DSP_DIRECT_ID;
    $fields{is_deleted} = $vpaid_enabled ? \0 : \1;

    my $query = $self->partner_db->query->select(
        table  => $self->partner_db->all_blocks,
        fields => \%fields,
        filter => ['page_id' => '=' => \$page_id],
    );

    $self->block_dsps->partner_db_table->add_multi([$query], replace => TRUE);
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    $self->SUPER::hook_save_fields_from_related_models($opts);

    my $id      = $self->hook_stash->get('id');
    my $related = $self->hook_stash->get('fields_from_related_models');
    my $mode    = $self->hook_stash->mode('add') ? 'add' : 'edit';

    if (my $categories = $related->{categories}) {
        $self->video_an_site_categories->replace($id->{id}, $categories, $mode);
    }

    if (my $mirrors = $related->{mirrors}) {
        $self->mirrors->replace($id, $mirrors);
    }

    foreach my $model (qw(excluded_domains excluded_phones assistants)) {
        if (defined(my $value = $related->{$model})) {
            $self->$model->replace($id->{'id'}, $value, $mode);
        }
    }
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    $self->SUPER::hook_processing_after_insert($opts);

    my $id   = $self->hook_stash->get('id');
    my $user = $self->hook_stash->get('user');

    $self->api_balance->create_or_update_place(
        operator_uid  => $user->{'id'},
        client_id     => $user->{'client_id'},
        page_id       => $id->{id},
        domain        => $opts->{'domain'},
        campaign_type => 3,
        ($user->{'client_id'} eq $ADINSIDE_CLIENT_ID ? (viptype => $BALANCE_VIPTYPE_YANDEX) : ()),
        is_tutby    => $user->{'is_tutby'},
        create_date => $opts->{'create_date'},
    );

}

sub hook_preparing_fields_to_save {
    my ($self, $opts) = @_;

    if (exists($opts->{'pixels'})) {
        $opts->{'pixels'} = to_json($opts->{'pixels'});
    }

    if (exists($opts->{metrica_counters})) {
        $opts->{metrica_counters} = to_json($opts->{metrica_counters});
    }

    $self->SUPER::hook_preparing_fields_to_save($opts);
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    $self->SUPER::hook_processing_after_update($opts);

    my $id = $self->hook_stash->get('id');

    if (exists($opts->{'partner_type'})) {
        my $partner_type = $self->hook_stash->get('current')->{'partner_type'};

        my @categories_ids = ();

        if (defined($partner_type)) {
            unless (defined($opts->{'partner_type'})) {
                @categories_ids = grep {$_ != 0} $self->video_an_site_categories->get_system_category_ids();
            } elsif ($partner_type eq 'yahosting' && $opts->{'partner_type'} ne 'yahosting') {
                @categories_ids = $self->video_an_site_categories->get_category_ids_for_yahosting();
            }
        }

        if (@categories_ids) {
            foreach my $block_model (@{$self->get_block_model_names()}) {
                my $blocks = $self->$block_model->get_all(
                    fields => [qw(page_id id)],
                    filter => {category_id => \@categories_ids, page_id => $id->{'id'}, multistate => 'not deleted'}
                );

                foreach my $block (@$blocks) {
                    $self->$block_model->maybe_do_action($block, 'delete');
                }
            }
        }
    }

    if (   $self->need_update_in_bk($self->hook_stash->get('fields'))
        && $self->check_action($id, 'set_need_update'))
    {
        $self->do_action($id, 'set_need_update');
    }

    $self->update_block_dsps($id, $opts->{'vpaid_enabled'}) if exists $opts->{'vpaid_enabled'};
}

sub can_action_edit {TRUE}

sub can_action_delete {TRUE}

sub on_action_delete {
    my ($self, $obj) = @_;

    $self->SUPER::on_action_delete($obj);

    $self->do_all_scenaries_action($obj, 'delete_with_page');
}

sub on_action_add_blocking {
    my ($self, $obj, %opts) = @_;

    $self->video_an_site_blockings->add(%opts, site_id => $obj->{'id'});

    $self->do_action($obj, 'set_need_update');
}

sub on_action_approve { }

sub on_action_del_blocking {
    my ($self, $obj, %opts) = @_;

    $self->video_an_site_blockings->delete(%opts, site_id => $obj->{'id'});

    $self->do_action($obj, 'set_need_update');
}

sub can_action_reject {TRUE}

sub on_action_reject {
    my ($self, $obj, %opts) = @_;

    $self->maybe_do_action($obj, 'stop');
    $self->maybe_do_action($obj, 'stop_testing');
}

sub on_action_reset_blocked { }

sub can_action_restore {TRUE}

sub on_action_restore {
    my ($self, $obj) = @_;

    $self->SUPER::on_action_restore($obj);

    $self->do_all_scenaries_action($obj, 'restore_with_page');
}

sub can_action_set_blocked {TRUE}

sub on_action_set_blocked {
    my ($self, $obj, %opts) = @_;

    $self->maybe_do_action($obj, 'stop');
    $self->maybe_do_action($obj, 'stop_testing');
}

sub on_action_set_need_approve {
    my ($self, $obj, %opts) = @_;

    $self->do_action($obj, 'approve');
}

sub do_all_scenaries_action {
    my ($self, $obj, $action) = @_;

    my $page_id = $self->_get_object_fields($obj, [qw(page_id)])->{'page_id'};

    my @scenaries = @{
        $self->video_scenaries->get_all(
            fields => $self->video_scenaries->get_depends_for_field('actions'),
            filter => {page_id => $page_id, multistate => $self->video_scenaries->get_multistate_by_action($action)}
        )
      };

    foreach my $scenario (@scenaries) {
        $self->video_scenaries->do_action($scenario, $action)
          if $self->video_scenaries->check_action($scenario, $action);
    }
}

sub query_filter {
    my ($self, $filter, $opts) = @_;

    $filter = $self->limit_filter_by_tutby_and_assistant_and_robot_assistant($filter, $opts);

    return $filter;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my $fields = $self->SUPER::get_need_update_in_bk_fields();

    $fields->{$_} = TRUE foreach (
        qw(
        buffer_empty_limit
        buffer_full_timeout
        caption
        categories
        partner_type
        skin
        skin_timeout
        skip_delay
        skip_time_left_show
        tag_id
        time_left_show
        title
        vast_timeout
        vast_version
        video_timeout
        vpaid_enabled
        vpaid_timeout
        wrapper_max_count
        wrapper_timeout
        )
    );

    #show_count
    #interval

    return $fields;
}

sub get_page_mirrors {
    my ($self, $page, $with_default) = @_;

    my $mirrors = [@{$page->{'mirrors'} // []}];

    # pushing after so DEFAULT_VIDEO_MIRRORS wont be in `excluded_domains`
    push(@$mirrors, @DEFAULT_VIDEO_MIRRORS) if $with_default;

    return $mirrors;
}

sub get_bk_data {
    my ($self, $page) = @_;

    my $user_id = $page->{'owner_id'};

    my @user_global_excluded_phones =
      map {$_->{'phone'}} @{$self->user_global_excluded_phones->get_all(filter => {user_id => $user_id})};

    my @user_global_excluded_domains =
      map {$_->{'domain'}} @{$self->user_global_excluded_domains->get_all(filter => {user_id => $user_id})};

    my $excluded_domains = array_uniq(
        check_domains_for_absorption([@{$page->{'excluded_domains'}}, @user_global_excluded_domains]),
        bs_format_phone([@{$page->{'excluded_phones'}}, @user_global_excluded_phones])
    );

    my %data = (
        mirrors          => $self->get_page_mirrors($page, TRUE),
        excluded_domains => [sort @$excluded_domains],
        target_type      => $CONTEXT_TARGET_TYPE,
        rtb_video        => {
            Categories =>
              {map {$_->{'id'} => {Name => $_->{'name'}, Archive => $_->{'archived'}}} @{$page->{'categories'}}},
            Contents => {map {$_->{'content_id'} => {CPM => 1_005_000_000}} @{$page->{'blockings'}}},
            Skin => $page->{'skin'} || $DEFAULT_VIDEO_SKIN,
            SkipDelay         => $page->{'skip_delay'},
            SkipTimeLeftShow  => $page->{'skip_time_left_show'},
            TimeLeftShow      => int($page->{'time_left_show'}),
            Title             => $page->{'title'},
            SkinTimeout       => $page->{'skin_timeout'},
            VASTTimeout       => $page->{'vast_timeout'},
            VideoTimeout      => $page->{'video_timeout'},
            BufferFullTimeout => $page->{'buffer_full_timeout'},
            WrapperTimeout    => $page->{'wrapper_timeout'},
            BufferEmptyLimit  => $page->{'buffer_empty_limit'},
            VPAIDEnabled      => int($page->{'vpaid_enabled'}),
            VPAIDTimeout      => $page->{'vpaid_timeout'},
            WrapperMaxCount   => $page->{'wrapper_max_count'},
            Platform          => $VIDEO_PLATFORMS->{$page->{'platform'}}{'type'},

            in_array('instream_vmap', $page->{owner}->{features})
            ? (
                VmapIDs => {
                    map {
                        $_->{id} => {
                            SingleVideoSession         => $_->{single_ad_session},
                            SingleVideoSessionInterval => $_->{ad_session_interval} // 0,
                            UseInterestingPoints       => $_->{use_better_places},
                          }
                      } grep {
                        $self->video_scenaries->check_multistate_flag($_->{'multistate'}, 'working')
                      } @{$page->{vmaps}},
                },
              )
            : (),
        },
        page_options => $self->page_options->get_options($page->{'page_id'}),
        (%{$page->{'cookie_match'}} ? (cookie_match_tag => $page->{'cookie_match'}{'tag'}) : ()),
    );

    return %data;
}

# API

sub api_available_actions {
    return qw(
      del_mirror
      delete
      edit
      start
      stop
      restore
      start_testing
      stop_testing
      add_blocking
      del_blocking
      set_blocked
      reset_blocked
      set_need_approve
      approve
      reject
      );
}

sub api_can_edit {TRUE}
sub api_can_add  {TRUE}

sub check_action {&Application::Model::Page::MIXIN::External::check_action}

sub get_block_types_on_pages {
    my ($self, $pages, $id) = @_;

    my $block_types_on_pages = {};

    foreach my $product ($self->get_block_models) {
        my $page_field = $product->get_page_id_field_name();

        my @filter = (
            [$page_field  => '=' => \$pages],
            ['multistate' => '=' => \$product->get_multistates_by_filter('not deleted')]
        );

        push(@filter, [id => '<>' => \$id]) if defined($id);

        my $data = $self->partner_db->query->select(
            table  => $product->partner_db_table(),
            fields => {$page_field => '', 'category_id' => '', 'type' => '', count => {'COUNT' => ['id']}},
            filter => ['AND' => \@filter]
        )->group_by($page_field, 'category_id', 'type')->get_all();

        foreach my $row (@$data) {
            $block_types_on_pages->{$row->{$page_field}}{$row->{'category_id'}}{$row->{'type'}} += $row->{'count'};
        }
    }

    return $block_types_on_pages;
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = ();

    if ($need_fields->{'platform'}) {
        $result{'platform'} = $self->get_platforms();
    }

    if ($need_fields->{'partner_type'}) {
        $result{'partner_type'} = $self->get_partner_types();
    }

    if ($need_fields->{'skip_time_left_show'}) {
        $result{'skip_time_left_show'} = [2, 5, 10, 15, 60, 180];
    }

    return \%result;
}

sub get_fields_depends {
    return {
        #если поменялось поле из ключ, то
        #нужно перезапросить поля из значения
        depends => {},
        #для поля из ключа обязятельно нужно передать поля из значения
        required => {},
    };
}

sub get_next_category_id {
    my ($self, $site_id) = @_;

    my $id = $self->partner_db->video_an_site_category_seq->get(
        $site_id,
        fields     => ['next_category_id'],
        for_update => TRUE
    );

    my $cur_id;
    if (defined($id)) {
        $cur_id = $id->{'next_category_id'};

        my $next_id = $cur_id + 1;
        my %vc = map {$_->{'id'} => TRUE} @{$self->video_an_site_categories->get_default_video_categories()};
        $next_id++ while $vc{$next_id};

        $self->partner_db->video_an_site_category_seq->edit($site_id, {next_category_id => $next_id});
    } else {
        $cur_id = 1;
        $self->partner_db->video_an_site_category_seq->add({site_id => $site_id, next_category_id => $cur_id + 1});
    }

    return $cur_id;
}

sub get_partner_types {
    my ($self) = @_;

    return [map {{id => $_, label => $PARTNER_TYPES->{$_}->()}} sort keys(%$PARTNER_TYPES)];
}

sub get_platforms {
    my ($self) = @_;

    my $platforms = clone($VIDEO_PLATFORMS);
    delete $platforms->{$VIDEO_PLATFORM_FLASH};

    return [
        map {{id => $_, label => $platforms->{$_}{'label'}(), type => $platforms->{$_}{'type'}}}
        sort {$a <=> $b} keys(%$platforms)
    ];
}

sub related_models {
    my ($self) = @_;

    return {
        %{$self->SUPER::related_models()},
        %{$self->external_related_models()},

        mirrors => {
            accessor => 'video_an_site_mirrors',
            filter   => sub {
                return {site_id => array_uniq(map {$_->{'id'} // ()} @{$_[1]})};
            },
            key_fields => ['site_id'],
            value_type => 'array_domain',
        },
        categories => {
            accessor => 'video_an_site_categories',
            filter   => sub {
                my $result = {page_id => array_uniq(map {$_->{'id'} // ()} @{$_[1]})};
                return $result;
            },
            key_fields => ['page_id'],
            value_type => 'array',
        },
        blockings => {
            accessor => 'video_an_site_blockings',
            filter   => sub {
                my $result = {site_id => array_uniq(map {$_->{'id'} // ()} @{$_[1]})};
                return $result;
            },
            key_fields => ['site_id'],
            value_type => 'array',
        },
        vmaps => {
            accessor => 'video_scenaries',
            filter   => sub {
                my $result = {page_id => array_uniq(map {$_->{'id'} // ()} @{$_[1]})};
                return $result;
            },
            key_fields => ['page_id'],
            value_type => 'array',
        },
    };
}

sub get_fields_to_trim {qw(domain skin title partner_type pixels caption)}

sub get_next_scenario_id {
    my ($self, $page_id) = @_;

    throw gettext('Must call in transaction') unless $self->partner_db->{'__SAVEPOINTS__'};

    my $scenario_seq_table = $self->partner_db->video_an_site_scenario_seq;
    my $pk                 = $scenario_seq_table->primary_key->[0];

    my $id = $scenario_seq_table->get(
        $page_id,
        fields     => ['next_scenario_id'],
        for_update => TRUE
    );

    my $cur_id;
    if (defined($id)) {
        $cur_id = $id->{'next_scenario_id'};

        $scenario_seq_table->edit($page_id, {next_scenario_id => $cur_id + 1});
    } else {
        $cur_id = 1;
        $scenario_seq_table->add({$pk => $page_id, next_scenario_id => $cur_id + 1});
    }

    return $cur_id;
}

TRUE;
