package Application::Model::PageMinimal;

=encoding UTF-8

=cut

=head1 DESCRIPTION

Базовый класс для всех пейджей с минимально допустимым набором полей для
- проверки прав по владельцу
- поддержки статусов (need_update, protected, read_only)
- отправки в БК
- валидации
- ведения action логов

В отличие от Application::Model::Page
- без блоков (get_block_model_names)
- без регистрации в Балансе
- без домена, зеркал (mirrors, excluded_domains, excluded_phones)
- без статистики (available_levels)
- без асистентов

=cut

use qbit;
use Exception::BK::Protected;
use Exception::Denied;

use base qw(
  Application::Model::Product
  Application::Model::ValidatableMixin
  );

consume qw(
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::CreateDate
  Application::Model::Role::Has::CreatorId
  Application::Model::Role::Has::SendTime
  Application::Model::Role::Has::EditableFields);

use Utils::Logger qw( ERROR WARN INFOF );

use PiConstants qw(
  $MYSQL_MIN_DATETIME
  $MYSQL_DEFAULT_DATETIME
  );

use Exception::Denied;
use Exception::Validation::BadArguments;

my $LIMIT = 1000;

__PACKAGE__->abstract_methods(qw(get_page_id_field_name _data_for_bk get_product_name));

sub check_read_only_and_protected_status {
    my ($self, %opts) = @_;

    my $page_field_name = $self->get_page_id_field_name();
    my $has_testing     = $self->get_multistate_by_name('testing');

    my $id_filter = [id => '>' => 0];
    my @filter = (
        {'multistate' => 'working' . ($has_testing ? ' or testing' : '')},
        [send_time => '>' => $MYSQL_MIN_DATETIME],    # хотя бы раз отправлялась
        [send_time => '<=' => date_sub(curdate(), minute => 60, oformat => 'db_time')],
        $id_filter
    );
    if ($opts{page_id}) {
        push @filter, [$page_field_name => IN => $opts{page_id}];
    }
    my @ignored_pages;
    my %extra_pages;
    my $failed  = FALSE;
    my $fixed   = 0;
    my $checked = 0;
    while (1) {
        my $rows = $self->get_all(
            fields => [$page_field_name, qw(id multistate)],
            filter   => [AND => \@filter],
            limit    => $LIMIT,
            order_by => ['id']
        );
        last unless @$rows;
        $id_filter->[2] = $rows->[-1]{id};
        my %campaigns = map {$_->{$page_field_name} => $_} @$rows;
        my @pages = sort keys %campaigns;
        $checked += @pages;

        try {
            my $all_settings = $self->api_yt_bk->export_table(
                table_name  => 'Page',
                field_names => [qw(PageID ReadOnly OptionsProtected)],
                page_id     => \@pages
            );

            foreach my $settings (@$all_settings) {
                try {
                    my $obj = delete $campaigns{$settings->{'PageID'}};
                    if ($obj) {
                        $fixed += $self->change_status_if_need($obj, 'protected', $settings->{'OptionsProtected'});
                        if ($opts{check_ro_status}) {
                            $fixed += $self->change_status_if_need($obj, 'read_only', $settings->{'ReadOnly'});
                        }
                    } else {
                        $extra_pages{$settings->{'PageID'}} = $settings;
                    }
                }
                catch {
                    my ($exception) = @_;

                    $failed = TRUE;
                    ERROR $exception;
                }
            }
            if (keys %campaigns) {
                push @ignored_pages, keys %campaigns;
            }
        }
        catch {
            my ($exception) = @_;

            $failed = TRUE;

            ERROR {
                exception   => $exception,
                message     => sprintf('check_read_only_and_protected_status FAILED for "%s"', $self->accessor()),
                extra       => {accessor => $self->accessor(),},
                fingerprint => ['check_read_only_and_protected_status', $self->accessor()],
            };
        }
    }

    if (@ignored_pages || keys(%extra_pages)) {
        $failed = TRUE;
        ERROR {
            message => $self->accessor() . ': requested and received page_ids are different',
            extra   => {
                extra_pages   => \%extra_pages,
                ignored_pages => \@ignored_pages,
            },
        };
    }

    return ($checked, $fixed, $failed);
}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        api_bk             => 'Application::Model::API::Yandex::BK',
        api_java_bk_data   => 'Application::Model::API::Yandex::JavaBKData',
        api_utils_partner2 => 'Application::Model::API::Yandex::UtilsPartner2',
        api_yt_bk          => 'QBit::Application::Model::API::Yandex::YT::BK',
        kv_store           => 'QBit::Application::Model::KvStore',
        partner_db         => 'Application::Model::PartnerDB',
        rbac               => 'Application::Model::RBAC',
        resources          => 'Application::Model::Resources',
        users              => 'Application::Model::Users',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            name        => $self->accessor(),
            description => d_gettext('Right to manage %s', $self->accessor()),
            rights      => {
                (
                    map {$self->get_description_right($_)}
                      qw(
                      view
                      view_all
                      view_action_log
                      edit
                      edit_all
                      view_field__comment
                      edit_field__comment
                      view_field__page_bk_data
                      ),
                ),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $FIELDS_DEPENDS;

    return {
        id => {
            default     => TRUE,
            db          => TRUE,
            pk          => TRUE,
            label       => d_gettext('ID'),
            type        => 'string',
            api         => 1,
            need_check  => {type => 'int_un'},
            adjust_type => 'str',
        },
        page_id => {
            #redefine
            default     => TRUE,
            db          => TRUE,
            db_expr     => $self->get_page_id_field_name(),
            label       => d_gettext('Page ID'),
            hint        => d_gettext('The unique campaign identifier (the same in BK)'),
            type        => 'number',
            api         => 1,
            need_check  => {type => 'int_un', optional => TRUE},
            adjust_type => 'str',
        },
        public_id => {
            db          => TRUE,
            db_expr     => 'id',
            type        => 'string',
            api         => 1,
            adjust_type => 'str',
        },
        owner_id => {
            default     => TRUE,
            db          => TRUE,
            label       => d_gettext('Owner id'),
            need_check  => {type => 'int_un',},
            type        => 'number',
            api         => 1,
            adjust_type => 'str',
        },
        caption => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                len_min => 1,
                len_max => 255,
            },
        },
        comment => {
            db           => TRUE,
            label        => d_gettext('Campaign\'s comment'),
            check_rights => $self->get_right('view_field__comment'),
            type         => 'string',
            need_check   => {
                type     => 'scalar',
                optional => TRUE,
                len_max  => 255,
            },
            api => 1,
        },
        multistate => {
            db          => TRUE,
            label       => d_gettext('Status (raw)'),
            type        => 'number',
            api         => 1,
            adjust_type => 'str',
        },
        multistate_name => {
            label      => d_gettext('Status'),
            depends_on => ['multistate'],
            get        => sub {
                return $_[0]->model->get_multistate_name($_[1]->{'multistate'},
                    private => $_[0]->model->check_short_rights('view_full_multistate_name'));
            },
            type => 'string',
            api  => 1,
        },
        #'actions',
        #'available_fields',
        #'editable_fields'
        page_bk_data => {
            depends_on   => [qw(id)],
            label        => 'page_bk_data',
            check_rights => $self->get_right('view_field__page_bk_data'),
            type         => 'complex',
            need_check   => {type => 'page_bk_data', optional => TRUE},
            get          => sub {
                my ($fields, $obj) = @_;
                my $data = $fields->model->_get_page_bk_data($obj, page_only => TRUE);
                return $data;
            },
            lazy => TRUE,
            api  => 1,
        },
        bk_state_name => {
            depends_on => [qw(multistate)],
            get        => sub {
                $_[0]->model->get_bk_state_name_by_multistate($_[1]->{'multistate'});
            },
            type => 'string',
            api  => 1,
        },
        bk_state_id => {
            depends_on => [qw(multistate)],
            get        => sub {
                if ($_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'working')) {
                    return 1;
                } elsif ($_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'testing')) {
                    return 0;
                } else {
                    return -1;
                }
            },
            type        => 'number',
            api         => 1,
            adjust_type => 'int',
        },
        is_updating => {
            forced_depends_on => ['multistate'],
            get               => sub {
                my ($fields, $page) = @_;
                return $fields->model->check_multistate_flag($page->{'multistate'}, 'need_update')
                  || $fields->model->check_multistate_flag($page->{'multistate'}, 'updating');
            },
            type        => 'boolean',
            api         => 1,
            adjust_type => 'str',
        },
        is_stopped => {
            depends_on => ['multistate'],
            get        => sub {
                return $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'stopped');
            },
            type        => 'boolean',
            api         => 1,
            adjust_type => 'str',
        },
        is_deleted => {
            depends_on => ['multistate'],
            get        => sub {
                return $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'deleted');
            },
            type        => 'boolean',
            api         => 1,
            adjust_type => 'str',
        },
        is_read_only => {
            depends_on => [qw(multistate)],
            get        => sub {
                return $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'read_only');
            },
            type        => 'boolean',
            api         => 1,
            adjust_type => 'str',
        },
        is_protected => {
            depends_on => ['multistate'],
            get        => sub {
                return $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'protected');
            },
            type        => 'boolean',
            api         => 1,
            adjust_type => 'str',
        },
        update_time    => {db => TRUE, label => d_gettext('Time queuing for updating'), type => 'string', api => 1,},
        fields_depends => {
            get => sub {
                $FIELDS_DEPENDS //= $_[0]->model->get_fields_depends();

                return $FIELDS_DEPENDS;
            },
            type => 'complex',
            api  => 1,
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        db_accessor => 'partner_db',
        fields      => {
            caption      => {type => 'text',       label => d_gettext('Caption')},
            caption_text => {type => 'alias',      path  => [qw(caption)]},
            id           => {type => 'number',     label => d_gettext('ID'), pk => TRUE},
            multistate   => {type => 'multistate', label => d_gettext('Status')},
            page_id      => {type => 'number',     label => d_gettext('Page ID'),},
            update_time  => {type => 'date',       label => d_gettext('Time queuing for updating')},
        },
    };
}

sub collect_editable_fields      {{}}
sub get_actions_depends          {[]}
sub get_available_fields_depends {[]}
sub get_editable_fields_depends  {[]}
sub pre_process_fields           { }
sub get_available_fields         {{}}

sub get_editable_fields {
    my ($self, $object) = @_;

    return {}
      unless $self->check_action($object, 'edit');

    return $self->collect_editable_fields($object);
}

sub _get_common_add_edit_fields {
    my ($self, $add_fields, $edit_fields) = @_;

    my $fields = $self->get_fields_by_right(right_fields => {edit => ['comment']});

    return $fields;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $opts->{id} //= $opts->{$self->get_page_id_field_name()} //= $self->api_utils_partner2->get_next_page_id();
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;
    $self->SUPER::hook_fields_processing_before_validation($opts);

    $opts->{'update_time'} = $MYSQL_DEFAULT_DATETIME;
}

sub on_action_restore {
    my ($self, $obj) = @_;

    my $model             = $self->accessor();
    my $related_accessors = $self->rbac->get_accessors_for_partner_owner_roles();
    if (exists($related_accessors->{$model})) {
        my $need_owner_role = $related_accessors->{$model}->{'id'};
        my $user_id         = $self->_get_object_fields($obj, ['owner_id'])->{'owner_id'};
        my $roles           = $self->rbac->get_roles_by_user_id($user_id);
        unless (exists($roles->{$need_owner_role})) {
            throw Exception::Denied gettext(
                "It is impossible to restore page from an archive until the owner login is not active.");
        }
    }

}

sub on_action_set_need_update {
    my ($self, $obj, %opts) = @_;
    $self->partner_db_table()->edit($obj, {'update_time' => curdate(oformat => 'db_time'),});
}

sub on_action_stop_update {
    my ($self, $obj) = @_;
    $self->partner_db_table()->edit($obj, {'send_time' => curdate(oformat => 'db_time'),});
}

sub throw_error_by_action {
    my ($self, $object, $action) = @_;

    if ($action eq 'delete' && $self->check_multistate_flag($object->{'multistate'}, 'deleted')) {
        throw Exception::Validation::BadArguments gettext('This page already deleted');
    } elsif (
        $action eq 'delete'
        && (   $self->check_multistate_flag($object->{'multistate'}, 'working')
            || $self->check_multistate_flag($object->{'multistate'}, 'testing'))
      )
    {
        throw Exception::Validation::BadArguments gettext('You cannot delete working page. First stopped it');
    } elsif ($action eq 'start' && $self->check_multistate_flag($object->{'multistate'}, 'working')) {
        throw Exception::Validation::BadArguments gettext('You cannot start a working page');
    }

    if ($self->check_multistate_flag($object->{'multistate'}, 'protected')
        && !$self->check_rights('edit_protected_pages'))
    {
        throw Exception::Validation::BadArguments gettext('You can not edit protected page');
    }

    return FALSE;
}

sub get_need_update_in_bk_fields {{}}

sub update_in_bk {
    my ($self, $filter, %opts) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights();

    my $page_internal_id = -1;
    my $page_id          = -1;
    {
        my $pages = $self->get_all(
            fields => ['id', 'page_id'],
            filter => $filter
        );

        throw Exception::Validation::BadArguments gettext('Campaigns not found')
          if @$pages == 0;

        throw gettext("Can't work with several campaigns") if @$pages > 1;

        ($page_internal_id, $page_id) = @{$pages->[0]}{qw(id page_id)};
    }

    my ($is_send_to_bssoap, undef, undef) = $self->api_bk->get_transports_to_send($self->accessor(), $page_id);

    $opts{do_not_change_multistate} = TRUE if ($opts{logbrocker_only} || $opts{protobuf_only}) && $is_send_to_bssoap;

    $self->do_action($page_internal_id, 'start_update') unless $opts{do_not_change_multistate};

    my $page = $self->get($page_internal_id, fields => ['*'],);

    my %bk_opts = $self->app->bkdata->_get_bk_data_opts(%opts);
    my $data_for_bk = $self->_data_for_bk($page, (is_from_java => $opts{is_from_java} // $bk_opts{is_from_java},));

    try {
        my $opts;
        foreach (qw(logbrocker_only protobuf_only is_regular_update)) {
            $opts->{$_} = 1 if $opts{$_};
        }
        $page_id = $self->api_bk->create_or_update_campaign($data_for_bk, $opts);

        # TODO удалить этот блок в одном из следующих ПР, например по PI-15793
        unless (defined($page->{'page_id'})) {
            my $page_id_field_name = $self->get_page_id_field_name();
            $self->partner_db_table()->edit($page, {$page_id_field_name => $page_id});
        }
    }
    catch Exception::BK::Protected with {
        $self->change_status_if_need($page, 'protected', TRUE);
        INFOF 'Page ID %s is closed for editing in BK (protected)', $page->{page_id};
    }
    catch {
        my ($e) = shift;
        $e->{text} .= sprintf("\n(page_id=%d)", $page->{page_id});
        throw $e;
    };

    if ($self->can('do_all_block_action')) {
        $self->do_all_block_action($page, 'stop_update');
    }

    unless ($opts{do_not_change_multistate}) {
        $self->maybe_do_action($page, 'stop_update');
    } else {
        $self->partner_db_table()->edit($page, {send_time => curdate(oformat => 'db_time')});
    }

    return $page_id;
}

sub change_status_if_need {
    my ($self, $object, $status_name, $real_value) = @_;

    $object = $self->_get_object_fields($object, [@{$self->get_pk_fields()}, qw(multistate)]);

    my $result = 0;
    if ($real_value) {
        unless ($self->check_multistate_flag($object->{'multistate'}, $status_name)) {
            $self->do_action($object, "set_$status_name");
            $result = 1;
        }
    } else {
        if ($self->check_multistate_flag($object->{'multistate'}, $status_name)) {
            $self->do_action($object, "reset_$status_name");
            $result = 1;
        }
    }
    return $result;
}

sub do_action {
    my ($self, $object, $action, %opts) = @_;

    my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions(qw(set_need_update)));

    my $result = $self->SUPER::do_action($object, $action, %opts);

    return $result;
}

sub do_all_block_action { }

sub need_update_in_bk {
    my ($self, $fields) = @_;

    my $fields_need_update = $self->get_need_update_in_bk_fields();

    foreach (@$fields) {
        return TRUE if $fields_need_update->{$_};
    }

    return FALSE;
}

sub related_models {{}}

sub may_send_not_balance_registered {TRUE;}

my %SKIP_KEY = map {$_ => TRUE} (
    'mirrors',    # входящий формат отличается от используемого
    'filters',
);

sub hook_post_validation {
    my ($self, $obj) = @_;

    $self->SUPER::hook_post_validation($obj);

    my $data = $self->_get_page_bk_data($obj, page_only => TRUE);

    $self->app->validator->check(
        data     => $data,
        app      => $self,
        template => {type => 'page_bk_data',},
        throw    => TRUE,
    );

    return TRUE;
}

sub _get_page_bk_data {
    my ($self, $obj, %opts) = @_;

    my $id = $self->hook_stash->inited ? $self->hook_stash->get('id')->{id} : $obj->{id};

    my $tmp_rights = $self->app->add_all_tmp_rights();
    my $full = $self->get($id, fields => ['*']);
    undef $tmp_rights;

    unless ($self->hook_stash->inited && $self->hook_stash->mode('add')) {
        for my $key (grep {!$SKIP_KEY{$_}} keys %$obj) {
            $full->{$key} = $obj->{$key};
        }
    }

    my $data_for_bk = $self->_data_for_bk(
        $full,
        read_only => TRUE,
        page_only => $opts{page_only}
    );

    my %bk_data = $self->app->api_bk->_get_bk_data([$data_for_bk]);
    my $data    = (values %bk_data)[0];

    return $data;
}

1;
