package Application::Model::PartnerActs;

use qbit;

use base qw(QBit::Application::Model);

use QBit::Validator;

use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;
use Utils::Logger qw/INFO/;

use PiConstants qw($SELFEMPLOYED_STATUS_READY $FIRM_ID_YANDEX_ADVERTISING);

sub accessor {'partner_acts'}

__PACKAGE__->model_accessors(
    api_oebs              => 'Application::Model::API::Yandex::OEBS',
    api_selfemployed_oebs => 'Application::Model::API::Yandex::SelfemployedOeBS',
    users                 => 'Application::Model::Users',
    documents             => 'Application::Model::Documents',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'partner_acts',
            description => d_gettext('Rights to use partner acts'),
            rights      => {
                partner_acts_view               => d_gettext('Right to view partner acts'),
                partner_acts_view_filter__login => d_gettext('Right to view filter "login"'),
                partner_acts_view_content       => d_gettext('Right to view partner act content'),
            }
        }
    ]
);

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        $self->check_short_rights('view_filter__login')
        ? ({name => 'login', label => gettext('Login'), login_filter => 'partner=1'})
        : ()
    ];
}

sub get_all_with_meta {
    my ($self, %opts) = @_;

    throw Exception::Denied gettext('Access denied') unless $self->check_short_rights('view');

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                filter => {
                    type     => 'array',
                    contents => [
                        {in => 'AND'},
                        {
                            type     => 'array',
                            contents => [{type => 'array', contents => [{in => 'login'}, {in => '='}, {}]}]
                        }
                    ],
                    optional => !$self->check_short_rights('view_filter__login'),
                    check    => sub {
                        my ($qv, $filter) = @_;

                        throw Exception::Validator::Fields gettext('Extra option "filter"')
                          if !$self->check_short_rights('view_filter__login')
                              && $filter;
                      }
                }
            }
        },
        throw => TRUE,
    );

    my $partner_info = $self->_get_partner_info($opts{'filter'}->[1][0][2]);

    my %contracts =
      map {$_->{'Contract'}{'contract2_id'} => $_}
      @{$self->documents->get_contracts(client_id => $partner_info->{client_id})};

    my $data = $self->api_oebs->get_partner_act_headers($partner_info->{client_id}) // [];

    my @result = ();
    foreach (@$data) {
        next unless $_->{'send_status'} eq 'Y';

        my $contract = $contracts{$_->{'contract_id'}};
        unless ($contract) {
            INFO('NOT FOUND IN CONTRACTS: ' . to_json($_, pretty => TRUE));
            next;
        }

        $_->{'external_id'} = $contract->{'Contract'}{'external_id'};

        $_->{'multistate_name'} = $self->_get_act_status($_, $contract);

        push(@result, $_);
    }

    # if selfemployed
    if ($SELFEMPLOYED_STATUS_READY == ($partner_info->{self_employed} // 0)) {
        try {
            my $receipt_list = $self->api_selfemployed_oebs->get_yandex_receipts($partner_info->{inn});
            push @result, @$receipt_list;
        }
        catch {
            my ($e) = @_;
            INFO 'get_yandex_receipts failed: ', $e->message();
        };
    }

    return {
        data => [sort       {$b->{'period_start_date'} cmp $a->{'period_start_date'}} @result],
        meta => {found_rows => scalar @result},
    };
}

sub get_content {
    my ($self, %opts) = @_;

    throw Exception::Denied gettext('Access denied') unless $self->check_short_rights('view_content');

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                act_id        => {type     => 'int_un'},
                login         => {optional => TRUE},
                encode_base64 => {type     => 'boolean', optional => TRUE},
            },
        },
        throw => TRUE,
    );

    my $act_id = delete($opts{'act_id'});
    my $login  = delete($opts{'login'});

    return $self->api_oebs->get_partner_act_contents($self->_get_client_id($login), $act_id, %opts);
}

sub get_db_filter_fields {
    return {login => {label => gettext('Login'), type => 'client_id_by_login'},};
}

sub _get_act_status {
    my ($self, $act, $contract) = @_;

    my $status;

    my $nds_0 = defined $contract->{'Contract'}{'nds'} && $contract->{'Contract'}{'nds'} == 0;
    my $is_ya_adv = $contract->{'Contract'}{'firm'} == $FIRM_ID_YANDEX_ADVERTISING;
    my $all_received =
         ($act->{'receive_sf_status'} eq 'Y' || $nds_0)
      && $act->{'receive_schet_status'} eq 'Y'
      && $act->{'receive_akt_status'}   eq 'Y';

    my $nothing_received =
         ($act->{'receive_sf_status'} eq 'N' || $nds_0)
      && $act->{'receive_schet_status'} eq 'N'
      && $act->{'receive_akt_status'}   eq 'N';

    if ($nds_0 && !$is_ya_adv) {
        $status = gettext('Documents provided');
    } elsif ($all_received) {
        if ($act->{'send_original_status'} eq 'Y') {
            $status = gettext('Got the original, originals sent');
        } else {
            $status = gettext('Got the original');
        }
    } elsif ($nothing_received) {
        $status =
          $act->{'receive_status'} eq 'Y'
          ? gettext('Got a copy, send the original')
          : gettext('Send the original');
    } else {
        my $not_received_current =
          $act->{'receive_status'} eq 'Y'
          ? gettext('got a copy, send the original')
          : gettext('send the original');

        my @statuses = ();

        if ($contract->{'Contract'}{'nds'} && $contract->{'Contract'}{'nds'} != 0) {
            push(@statuses,
                gettext('Invoice') . ' — '
                  . ($act->{'receive_sf_status'} eq 'Y' ? gettext('got the original') : $not_received_current));
        }

        push(@statuses,
            gettext('Invoice for payment') . ' — '
              . ($act->{'receive_schet_status'} eq 'Y' ? gettext('got the original') : $not_received_current));

        push(@statuses,
            gettext('Act') . ' — '
              . ($act->{'receive_akt_status'} eq 'Y' ? gettext('got the original') : $not_received_current));

        $status = join("\n", @statuses);
    }

    return $status;
}

sub _get_client_id {
    my ($self, $login) = @_;

    my $client_id;

    my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_field__client_id ));

    if ($self->check_short_rights('view_filter__login')) {
        my $user = $self->users->get_by_login($login, fields => [qw(client_id)])
          // throw Exception::Validation::BadArguments gettext('User with login "%s" not found', $login);

        $client_id = $user->{'client_id'};
    } else {
        $client_id = $self->get_option('cur_user', {})->{'client_id'};
    }

    throw gettext('Client ID not found') unless $client_id;

    return $client_id;
}

sub _get_partner_info {
    my ($self, $login) = @_;

    my $partner_info;

    my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_field__client_id ));

    if ($self->check_short_rights('view_filter__login')) {
        $partner_info = $self->users->get_by_login($login, fields => [qw(client_id inn self_employed)])
          // throw Exception::Validation::BadArguments gettext('User with login "%s" not found', $login);
    } else {
        $partner_info =
          $self->app->users->get($self->get_option('cur_user', {}), fields => [qw(client_id inn self_employed)]);
    }

    throw gettext('Client ID not found') unless $partner_info->{client_id};

    return $partner_info;
}

TRUE;
