package Application::Model::PartnerDocuments;

use qbit;

use base qw(QBit::Application::Model);

use QBit::Validator;

use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

sub accessor {'partner_documents'}

__PACKAGE__->model_accessors(
    api_oebs => 'Application::Model::API::Yandex::OEBS',
    users    => 'Application::Model::Users',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'partner_documents',
            description => d_gettext('Rights to use partner documents'),
            rights      => {
                partner_documents_view_filter__login => d_gettext('Right to view filter "login"'),
                partner_documents_view               => d_gettext('Right to view partner documents'),
                partner_documents_view_content       => d_gettext('Right to view partner document content'),
            }
        }
    ]
);

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        $self->check_short_rights('view_filter__login')
        ? ({name => 'login', label => gettext('Login'), login_filter => 'partner=1'})
        : ()
    ];
}

sub get_all_with_meta {
    my ($self, %opts) = @_;

    throw Exception::Denied gettext('Access denied') unless $self->check_short_rights('view');

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                filter => {
                    type     => 'array',
                    contents => [
                        {in => 'AND'},
                        {
                            type     => 'array',
                            contents => [{type => 'array', contents => [{in => 'login'}, {in => '='}, {}]}]
                        }
                    ],
                    optional => !$self->check_short_rights('view_filter__login'),
                    check    => sub {
                        my ($qv, $filter) = @_;

                        throw Exception::Validator::Fields gettext('Extra option "filter"')
                          if !$self->check_short_rights('view_filter__login')
                              && $filter;
                      }
                }
            }
        },
        throw => TRUE,
    );

    my $client_id = $self->_get_client_id($opts{'filter'}->[1][0][2]);

    my $data = $self->api_oebs->get_partner_documents($client_id);

    return {
        data => $data,
        meta => {found_rows => scalar(@$data)},
    };
}

sub get_content {
    my ($self, %opts) = @_;

    throw Exception::Denied gettext('Access denied') unless $self->check_short_rights('view_content');

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                doc_id        => {type     => 'int_un'},
                login         => {optional => TRUE},
                encode_base64 => {type     => 'boolean', optional => TRUE},
            },
        },
        throw => TRUE,
    );

    my $doc_id = delete($opts{'doc_id'});
    my $login  = delete($opts{'login'});

    return $self->api_oebs->get_partner_document_contents($self->_get_client_id($login), $doc_id, %opts);
}

sub get_db_filter_fields {
    return {login => {label => gettext('Login'), type => 'client_id_by_login'},};
}

sub _get_client_id {
    my ($self, $login) = @_;

    my $client_id;

    my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_field__client_id ));

    if ($self->check_short_rights('view_filter__login')) {
        my $user = $self->users->get_by_login($login, fields => [qw(client_id)])
          // throw Exception::Validation::BadArguments gettext('User with login "%s" not found', $login);

        $client_id = $user->{'client_id'};
    } else {
        $client_id = $self->get_option('cur_user', {})->{'client_id'};
    }

    throw gettext('Client ID not found') unless $client_id;

    return $client_id;
}

TRUE;
