package Application::Model::PartnerMonitoring;

use qbit;

use Exception::Denied;
use Utils::MonitoringUtils;
use PiConstants qw(
  $SUPPORT_MAIL
  $AUTHORIZATION_USER_FIELDS
  $MOL_REPORT_TYPE_MAIN
  $CATEGORY_MOL_BASIC
  $CATEGORY_MOL_FINANCIAL
  );

use base qw(QBit::Application::Model);
use Utils::Logger qw(ERROR);
use URI::Escape qw(uri_escape_utf8);

sub accessor {'partner_monitoring'}

__PACKAGE__->model_accessors(
    product_manager   => 'Application::Model::ProductManager',
    mail_notification => 'Application::Model::MailNotification',
    partner_db        => 'Application::Model::PartnerDB',
    users             => 'Application::Model::Users',
    bk_statistics     => 'Application::Model::BKStatistics',
);

my $URL = 'https://partner2.yandex.ru/v2/statistics/';

sub send_partner_monitoring_products {
    my ($self, %opts) = @_;
    # %opts = (
    #   login => 'foo, bar',
    #   date  => '2017-01-01'
    #);

    my $filter = $self->users->get_db_filter(
        {
            role_id => [9, 20, 27],    # Sites Partner, Video Partner, Applications Partner
            no_stat_monitoring_emails => 0
        }
    );

    my ($instances_count, $instance_number) = @opts{qw(instances_count instance_number)};
    if ($opts{'login'}) {
        my $logins = $opts{'login'};
        $logins =~ s/^\s+//;
        $logins =~ s/\s+$//;
        $filter->and([login => IN => \[split(/\s*,\s*/, $logins)]]);
    } elsif ($instances_count and $instance_number) {
        $instance_number--;
        $filter->and([{'MOD' => ['id', \$instances_count]}, '=', \$instance_number]);
    }

    my $users = $self->users->get_all(
        fields => $AUTHORIZATION_USER_FIELDS,
        filter => $filter,
    );

    my $cur_user = $self->app->get_option('cur_user');

    # reinstall proper rights checking
    my $old_check_rights = \&QBit::Cron::check_rights;
    {
        no warnings 'redefine';
        *QBit::Cron::check_rights = \&QBit::Application::check_rights;
    }

    my $yan_title       = gettext('Yandex Advertising Network');
    my $russian_support = {$SUPPORT_MAIL => $yan_title};
    my $subject         = gettext('Statistics monitoring in new interface');

    my $date = $opts{'date'} || date_sub(curdate(), day => 1, oformat => 'db');
    my $week_ago = date_sub($date, day => 7, iformat => 'db', oformat => 'db');

    my $report_title = gettext(
        'Comparison statistics for the %s on %s',
        format_date($date,     '%d.%m.%Y', iformat => 'db'),
        format_date($week_ago, '%d.%m.%Y', iformat => 'db')
      ),
      my $report_sub_title = gettext('Income change');

    foreach my $user (@$users) {
        $self->app->set_cur_user($user);

        my @emails = split(/\s*,\s*/, $user->{'email'});

        my $data = $self->get_data_for_monitoring_mol(date => $date, compare_date => $week_ago, user => $user);

        if (@$data) {
            $self->mail_notification->add(
                type    => 0,
                user_id => $user->{id},
                opts    => {
                    from     => $russian_support,
                    subject  => $subject,
                    template => 'monitoring/partner_monitoring.html.tt2',
                    values   => {
                        title     => $report_title,
                        sub_title => $report_sub_title,
                        login     => $user->{login},
                        data      => $data,
                    },
                    vars => {map {$_ => $_} qw(title sub_title login data)},
                },
            );
        }

    }

    $self->app->set_cur_user($cur_user);

    {
        no warnings 'redefine';
        *QBit::Cron::check_rights = $old_check_rights;
    }

    return TRUE;
}

sub get_data_for_monitoring_mol {
    my ($self, %opts) = @_;

    my $is_available = 0;
    try {
        $is_available = $self->bk_statistics->_check_access() && $self->bk_statistics->is_available();
    }
    catch Exception::Denied with {
        return [];
    };
    return [] unless $is_available;

    my $date         = $opts{'date'};
    my $compare_date = $opts{'compare_date'};

    my @MON = (
        {
            id     => 'payment',
            label  => gettext('Payment'),
            fields => [qw(partner_wo_nds)],
        },
        (
            map +{
                id     => $_->{id},
                label  => $_->{label},
                filter => ['block_level', 'IN', [$_->{id}]]
            },
            @{$self->bk_statistics->_get_block_model_values}
        ),
    );

    my @fields_data = map +{
        %$_,
        id    => $_->{partner2_id},
        title => $_->{label},
        type  => $_->{frontend_type},
      },
      grep {
        $_->{'select'}
          && ($_->{frontend_category} == $CATEGORY_MOL_FINANCIAL || $_->{frontend_category} == $CATEGORY_MOL_BASIC)
      } $self->bk_statistics->_get_fields($MOL_REPORT_TYPE_MAIN);

    my @sorted_data;

    foreach my $mon (@MON) {
        my @actual_fields_data;
        if ($mon->{fields}) {
            my %h = map {$_ => undef} @{$mon->{fields}};
            @actual_fields_data = grep {exists $h{$_->{id}}} @fields_data;
        } else {
            @actual_fields_data = @fields_data;
        }
        my %hash_fields = map {$_->{id} => $_} @actual_fields_data;
        my @fields = keys %hash_fields;

        my $level    = $mon->{id};
        my %stat_all = ();

        my $stat_result;
        try {
            $stat_result = $self->bk_statistics->get_statistics2(
                period => [[$date, $date], [$compare_date, $compare_date]],
                ($mon->{filter} ? (levels => [{filter => ['AND', [$mon->{filter}]]}]) : ()),
                fields           => \@fields,
                dimension_fields => ["date|day"],
            );
        }
        catch {
            my ($e) = @_;
            my $user = $opts{user} // {};
            ERROR {
                exception => $e,
                extra     => {
                    user_id         => $user->{id},
                    user_multistate => $user->{multistate},
                    user_login      => $user->{login}
                },
                fingerprint => ['PartnerMonitoring', 'get_data_for_monitoring_mol'],
            };
        };

        next unless defined($stat_result) && @{$stat_result->{points}};

        my @data = _format_data_for_monitoring_email_letter(
            $level,
            $stat_result->{points}[0]{measures}[1],
            $stat_result->{points}[0]{measures}[0],
            \%hash_fields
        );

        my %args = (
            'page[size]'      => 10,
            'page[number]'    => 1,
            'period[endDate]' => date_add(
                $date,
                iformat => 'db',
                oformat => 'sec',
                hour    => 23,
                minute  => 59,
                second  => 59,
              )
              . '999',
            'period[startDate]' => trdate(db => sec => $date) . '000',
            'comparisonStart'   => trdate(db => sec => $compare_date) . '000',
            'statType'          => $MOL_REPORT_TYPE_MAIN,
            'dimensionFields'   => 'date|day',
        );

        if ($mon->{filter}) {
            $args{filter} = '["AND",[' . to_json($mon->{filter}) . ']]';
        }
        my $stat_url = 'https://partner.yandex.ru/v2/statistics/constructor/?'
          . uri_escape_utf8(join('&', map("$_=$args{$_}", sort keys %args), map("fields=$_", sort @fields)));

        my $result = {
            product    => "<a href=\"$stat_url\">" . html_encode($mon->{label}) . "</a>",
            fields     => \@actual_fields_data,
            currencies => [{id => 'RUB', code => 2, data => \@data}],
        };

        push @sorted_data, $result;
    }

    return \@sorted_data;
}

TRUE;
