package Application::Model::Product::AN::ContextOnSite::AdBlock;

use qbit;

use base qw(Application::Model::Block::External::Site RestApi::MultistateModel Application::Model::ValidatableMixin);

consume qw(
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::DSPSWithoutMode
  Application::Model::Role::Has::ShowVideo
  Application::Model::Role::Has::Strategies
  Application::Model::Role::Has::CustomBkData
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Block::Has::BlocksLimit
  );

use PiConstants qw(
  $DSP_DIRECT_ID
  @VIDEO_AWAPS_DSPS
  $MIN_CPM_STRATEGY_ID
  $DSP_MEDIA_TYPE_ID
  );

use Application::Model::Product::BlockTypes::RTB qw(
  direct_block_types
  %DIRECT_BLOCKS
  );

use Partner::DSP::RuleSet;
use Partner::DSP::Rule;
use Partner::DSP::Rule::Part;
use Partner::DSP::Rule::Filter;

use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

sub accessor      {'context_on_site_adblock'}
sub db_table_name {'context_on_site_adblock'}

sub get_campaign_model_name {'context_on_site_campaign'}

sub get_product_name {gettext('context_on_site_adblock')}

sub get_page_id_field_name {'page_id'}

sub public_id_prefix {'R-AB-'}

sub storage_method_of_strategies {'db'}

sub get_dsp_type {[$DSP_MEDIA_TYPE_ID]}

sub get_video_dsp_list {@VIDEO_AWAPS_DSPS}

sub get_video_dsp_list_depends {()}

sub get_video_dsp_turn_on_block_filter {
    my ($self, $dsp) = @_;

    if (grep {$_ == $dsp->{id}} @VIDEO_AWAPS_DSPS) {
        return [show_video => '=' => 1];
    } else {
        return;
    }
}

sub role_show_video_get_fields_depends {[qw(show_video)]}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        context_on_site_campaign => 'Application::Model::Product::AN::ContextOnSite::Campaign',
        tns_dict_brand           => 'Application::Model::TNSDict::Brand',
        tns_dict_article         => 'Application::Model::TNSDict::Article',
        dsp                      => 'Application::Model::DSP',
        geo_base                 => 'Application::Model::GeoBase',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        @{$self->SUPER::get_structure_rights_to_register()},
        {
            name        => $self->accessor(),
            description => d_gettext('Right to manage AdBlock'),
            rights      => {
                (
                    map {$self->get_description_right($_)}
                      qw(
                      edit_field__blind
                      edit_field__comment
                      edit_field__is_custom_format_direct
                      view_field__blind
                      view_field__comment
                      view_filter__login
                      ),
                ),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        direct_block => {
            db         => TRUE,
            label      => d_gettext('Direct block format'),
            type       => 'string',
            api        => 1,
            need_check => {
                check => sub {
                    my ($qv, $direct_block) = @_;

                    throw Exception::Validator::Fields gettext('Invalid direct block type')
                      unless defined($direct_block);
                    throw Exception::Validator::Fields gettext('Invalid direct block type')
                      unless $DIRECT_BLOCKS{$direct_block} || $qv->data->{'is_custom_format_direct'};
                },
            },
        },
        limit => {
            db         => TRUE,
            label      => d_gettext('Ads limit'),
            type       => 'number',
            api        => 1,
            need_check => {
                type  => 'int_un',
                check => sub {
                    my ($qv, $limit) = @_;

                    throw Exception::Validator::Fields gettext('Invalid limit')
                      unless defined($limit) && looks_like_number($limit) && $limit =~ /^[0-9]+$/;

                    my $direct_block = $qv->data->{'direct_block'};
                    throw Exception::Validator::Fields gettext('Invalid limit')
                      unless $DIRECT_BLOCKS{$direct_block}
                          && $limit >= $DIRECT_BLOCKS{$direct_block}{'min_banners'}
                          && $limit <= $DIRECT_BLOCKS{$direct_block}{'max_banners'}
                          || $qv->data->{'is_custom_format_direct'};
                },
            },
            adjust_type => 'str',
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $parent_filter = $self->SUPER::get_structure_model_filter();

    return {
        db_accessor => 'partner_db',
        fields      => {
            %{$parent_filter->{'fields'}},
            page_id => {type => 'number', label => d_gettext('Page ID')},
            page    => {
                type           => 'subfilter',
                model_accessor => $self->get_campaign_model_name(),
                field          => 'page_id',
                fk_field       => 'page_id',
                label          => d_gettext('Page'),
            },
            login      => {type => 'alias', path => [qw(page owner login)]},
            show_video => {type => 'boolean'},
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        (
            $self->check_rights('view_search_filters__login') ? {name => 'login', label => gettext('Login')}
            : ()
        ),
        (
            $self->check_rights('view_search_filters__user_type') ? {name => 'user_type', label => gettext('User type')}
            : ()
        ),
        {name => 'page_id',         label => gettext('Page ID')},
        {name => 'id',              label => gettext('Block ID')},
        {name => 'caption',         label => gettext('Block\'s caption')},
        {name => 'page.all_domain', label => gettext('Domain and mirror')},
        {name => 'multistate',      label => gettext('Status')},
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_pgettext('Block multistate', 'New'),
        multistates => [
            [deleted           => d_pgettext('Block multistate', 'Archived')],
            [working           => d_pgettext('Block multistate', 'Working')],
            [check_statistics  => d_pgettext('Block multistate', 'Check statistics')],
            [deleted_with_page => d_pgettext('Block multistate', 'Archived with page'), private => TRUE],
            [need_update       => d_pgettext('Block multistate', 'Need update')],
            [updating          => d_pgettext('Block multistate', 'Updating')],
        ],
        actions => {
            edit                   => d_pgettext('Block action', 'Edit'),
            delete                 => d_pgettext('Block action', 'Archive'),
            delete_with_page       => d_pgettext('Block action', 'Archive with page'),
            restore                => d_pgettext('Block action', 'Restore'),
            restore_with_page      => d_pgettext('Block action', 'Restore with page'),
            start                  => d_pgettext('Block action', 'Start'),
            stop                   => d_pgettext('Block action', 'Stop'),
            set_check_statistics   => d_pgettext('Block action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Block action', 'Reset "check_statistics"'),
            duplicate              => d_pgettext('Block action', 'Duplicate'),
        },
        right_name_prefix => $self->accessor . '_',
        right_actions     => {
            add => d_pgettext('Block action', 'Add'),
            set_need_update =>
              {label => d_pgettext('Block action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Block action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Block action', 'Stop update'),  dont_write_to_action_log => TRUE},
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action    => 'delete_with_page',
                from      => 'not deleted',
                set_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page',
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action    => 'start',
                from      => 'not working and not deleted',
                set_flags => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
            },
            {
                action    => 'set_check_statistics',
                from      => 'not (deleted or check_statistics)',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action => 'duplicate',
                from   => 'not (deleted or deleted_with_page)',
            },
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ],
    };
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor      = $self->accessor();
    my $page_accessor = $self->get_campaign_model_name();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s' => [qw( comment  is_tutby)],

            $page_accessor . '_view_field__%s'    => [qw(login domain_id)],
            $page_accessor . '_view_field__owner' => [qw(owner_client_id)],
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    $fields->{'page_id'} = TRUE;

    $fields->{'comment'} = TRUE if $self->check_short_rights('edit_field__comment');

    $self->set_show_video_add_fields({}, $fields);

    return $fields;
}

sub get_actions_depends {
    [qw(id page_id multistate page cur_user_is_read_assistant is_protected)];
}

sub get_available_fields_depends {
    [qw(multistate)];
}

sub get_editable_fields_depends {
    [qw(page_id page id multistate is_protected cur_user_is_read_assistant)];
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    return $self->_get_common_add_edit_fields();
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        no_right_fields => [
            qw(
              caption
              direct_block
              limit
              )
        ],
        right_fields => {edit => ['comment']}
    );

    return $fields;
}

#Пока не переведено на хуки - не убирать!
#TODO - после перевода на хуки, убрать эти виртуальные методы. Добавлено при введении ограничения на блоки.
sub hook_fields_validation { }

sub hook_processing_after_insert { }

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied gettext('Access denied')
      unless $self->check_rights($self->get_rights_by_actions('add'));

    $self->_trim_opts(\%opts);

    my $allowed_fields = $self->get_add_fields(\%opts);

    $self->check_bad_fields_in_add(\%opts, $allowed_fields);

    $self->app->validator->check(
        data  => \%opts,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => [keys(%{$self->get_add_fields})]),
    );

    $self->hook_fields_validation(\%opts);

    my $block_id;
    $self->partner_db->transaction(
        sub {
            my $id = $self->get_page_model->get_next_block_id($opts{'page_id'});

            my $dsps = delete($opts{'dsps'});

            $block_id = $self->partner_db_table()->add(
                {
                    id => $id,
                    %opts
                }
            );

            # TODO: Has::DSPS::hook_set_initialize_settings
            unless ($self->check_short_rights('edit_field__dsps')) {
                $dsps = [map {$_->{'id'}} @{$self->get_default_dsps({show_video => $opts{'show_video'} // 0})}];
            }

            # TODO: Has::DSPS::hook_save_fields_from_related_models
            if (defined($dsps)) {
                my $tmp_rights = $self->app->add_tmp_rights('block_dsps_edit');
                $self->block_dsps->replace({page_id => $opts{'page_id'}, block_id => $block_id->[1]}, $dsps, 'add');
            }

            $self->do_action($block_id, 'add', %opts);
            $self->maybe_do_action($block_id, 'start');

            $self->update_in_bk($block_id);
        }
    );

    $self->hook_processing_after_insert($block_id);

    return $self->public_id({page_id => $block_id->[0], id => $block_id->[1]});
}

sub on_action_edit {
    my ($self, $block, %data) = @_;

    $self->_trim_opts(\%data);

    my $editable_fields = $self->check_bad_fields($block, \%data);

    my @fields_for_validation = (sort keys(%$editable_fields), qw(show_video));

    my $old_block_settings = $self->get($block, fields => \@fields_for_validation);
    my %new_block_settings = (%$old_block_settings, %data);

    # TODO: Has::DSPS::hook_fields_processing_before_validation
    my $rules = $self->get_dsp_rule_set();
    $rules->edit_dsp_list($old_block_settings, \%data);
    $new_block_settings{dsps} = $data{dsps} if exists($data{dsps});

    $self->app->validator->check(
        data  => \%new_block_settings,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => \@fields_for_validation),
    );

    my @fields = keys(%data);

    # TODO: Has::DSPS::hook_save_fields_from_related_models
    if (exists($data{'dsps'})) {
        my $tmp_right_edit_dsps = $self->app->add_tmp_rights('block_dsps_edit');
        $self->block_dsps->replace({page_id => $block->{'page_id'}, block_id => $block->{'id'}}, $data{'dsps'}, 'edit');

        delete($data{'dsps'});
    }

    $self->partner_db_table()->edit($block, \%data) if %data;

    $self->update_in_bk($block) if $self->need_update_in_bk(\@fields);

    return TRUE;
}

sub can_action_duplicate {
    my ($self, $obj, %opts) = @_;

    return FALSE unless $self->check_rights($self->get_rights_by_actions('add'));

    my $object = $self->_get_object_fields($obj, [qw(is_protected)]);

    return !$object->{'is_protected'} || $self->check_rights('edit_protected_pages') ? TRUE : FALSE;
}

sub on_action_duplicate {
    my ($self, $obj, %opts) = @_;

    my $tmp_rights = $self->app->add_tmp_rights(
        'block_dsps_edit',
        $self->get_right('edit_field__dsps'),
        $self->get_right('edit_field__is_custom_format_direct'),
        $self->get_right('edit_field__show_video')
    );

    my $entity = $self->get($obj, fields => [sort keys(%{$self->get_add_fields()})],);

    my $result = $self->add(%$entity);

    $self->hook_processing_after_insert($obj);

    return {public_id => $result};
}

sub fix_template {
    my ($self, $qv) = @_;

    my $template = $qv->template;

    my $direct_block = $qv->data->{'direct_block'};

    if (defined($direct_block)
        && exists($DIRECT_BLOCKS{$direct_block}->{'skip_fields'}))
    {
        foreach (@{$DIRECT_BLOCKS{$direct_block}->{'skip_fields'}}) {
            $template->{'fields'}{$_} = {eq => undef};
        }
    }

    $qv->template($template);
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_campaign($filter);

    return $filter;
}

sub get_bk_common_data {
    return ();
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = map {$_ => TRUE} qw(
      dsps
      caption
      direct_block
      limit
      );

    return \%fields_need_update;
}

# API
#
sub api_available_fields {
    my ($self) = @_;

    my @fields = $self->SUPER::api_available_fields();

    return sort @fields;
}

sub api_available_actions {
    return qw(duplicate);
}

sub api_can_add {1}

sub api_can_edit {1}

sub direct_blocks {
    return \%DIRECT_BLOCKS;
}

sub get_types_direct_blocks {
    my ($self) = @_;

    my $DIRECT_BLOCKS = $self->direct_block_types();

    my %direct_blocks = map {$_ => TRUE} qw(vertical);

    return [] unless %direct_blocks;

    my @result = ();
    foreach (@$DIRECT_BLOCKS) {
        next unless $direct_blocks{$_->{'id'}};

        $_->{'short_caption'} = $_->{'short_caption'}();
        $_->{'caption'}       = $_->{'caption'}();

        push(@result, {hash_transform($_, [qw(id short_caption caption min_banners max_banners skip_fields banners)])});
    }

    return \@result;
}

sub get_types_media_blocks {[]}

sub get_types_dsp_blocks {[]}

sub get_all_campaigns_for_adding {
    my ($self, %opts) = @_;

    my $is_assistant = $self->check_rights('is_assistant');

    my @filter = ();
    push(@filter, $opts{'filter'}) if exists($opts{'filter'});
    push(@filter, ['tag_id', 'IS NOT', undef]);    # can only add adblock to pages with a valid tag

    push(@{$opts{'fields'}}, 'owner_id') if $is_assistant;

    my $pages = $self->SUPER::get_all_campaigns_for_adding(%opts, filter => ['AND' => \@filter]);

    if ($is_assistant) {
        my $cur_user       = $self->get_option('cur_user', {});
        my $user_id        = $cur_user->{'id'};
        my $cur_user_roles = $cur_user->{'roles'} // $self->rbac->get_cur_user_roles() // {};

        my %owners_pages;
        my @result;

        foreach my $page (@$pages) {
            if ($user_id == $page->{'owner_id'}) {
                push(@result, $page);
            } else {
                push(@{$owners_pages{$page->{'owner_id'}}}, $page);
            }
        }

        if (%owners_pages) {
            my $adblock_users = $self->partner_db->user_role->get_all(
                fields => [qw(user_id)],
                filter => {user_id => [sort keys(%owners_pages)], role_id => 36},
            );

            foreach my $user (@$adblock_users) {
                push(@result, @{$owners_pages{$user->{'user_id'}}});
            }
        }

        return \@result;
    } else {
        return $pages;
    }
}

sub get_bk_block_data {
    my ($self, $block) = @_;

    my %bk_data;

    my $direct_blocks = $self->direct_blocks();

    my $type_name = $block->{'direct_block'}     // '';
    my $type_obj  = $direct_blocks->{$type_name} // {};

    my %skip_fields = $type_obj->{'skip_fields'} ? (map {$_ => TRUE} @{$type_obj->{'skip_fields'}}) : ();

    $bk_data{'DSPType'} = 1;    #2**0 media

    # For video direct (PI-8892)
    $block->{'Sizes'} = [{Width => 0, Height => 0}];

    my $strategy = $block->{'strategy'};
    $bk_data{'OptimizeType'} = $strategy;

    my $direct_block = $block->{'direct_block'};
    my $limit        = $block->{'limit'};

    $bk_data{'BlockCaption'} = $block->{'caption'};
    $bk_data{'MultiState'}   = $block->{'multistate'};

    if (defined($block->{'limit'}) && !$skip_fields{'limit'}) {
        $bk_data{'DirectLimit'} = $limit;
    } else {
        my %direct_blocks = map {$_->{'id'} => $_} @{$self->get_types_direct_blocks()};

        throw gettext('direct_block = "%s" not found', $direct_block)
          unless $direct_blocks{$direct_block};
        $bk_data{'DirectLimit'} = $direct_blocks{$direct_block}->{'banners'};
    }

    $bk_data{'AdTypeSet'} = {
        'text'              => 1,
        'media'             => 0,
        'video'             => $block->{'show_video'} ? 1 : 0,
        'media-performance' => 0,
        'video-performance' => 0,
    };

    $bk_data{'AdBlockBlock'} = 1;

    return \%bk_data;
}

sub get_bk_block_data_key {'rtb_blocks'}

sub get_dsp_rules {
    my ($self) = @_;

    return [
        @{$self->SUPER::get_dsp_rules()},
        $self->get_show_video_dsp_rule(),
        Partner::DSP::Rule->new(
            available_dsps => Partner::DSP::Rule::Part->new(
                sub => sub {
                    limit => {id => [1, $self->get_video_dsp_list()]};
                },
            ),
            default_dsps => Partner::DSP::Rule::Part->new(
                sub => sub {
                    extend => {id => [1]};
                },
            ),
            turn_on_dsp => Partner::DSP::Rule::Part->new(
                sub => sub {
                    limit => Partner::DSP::Rule::Filter->zero(),
                      ;
                },
            ),
        ),
    ];
}

sub get_strategies {
    return {$MIN_CPM_STRATEGY_ID => {},};
}

sub _trim_opts {
    my ($self, $opts) = @_;

    foreach my $field_name (qw(caption)) {
        $opts->{$field_name} =~ s/^\s+|\s+$//g if $opts->{$field_name};
    }
}

TRUE;
