package Application::Model::Product::AN::ContextOnSite::Campaign;

use qbit;

use base qw(
  Application::Model::Page::Site::External
  RestApi::MultistateModel
  );

consume qw(
  Application::Model::Role::Has::Context
  Application::Model::Role::Has::ModerationReason::Page
  Application::Model::Role::Has::Tier
  Application::Model::Role::Page::Has::Assessor
  Application::Model::Role::Page::Has::BlocksLimit
  Application::Model::Role::Page::Has::MetricaCounters
  Application::Model::Role::Page::Has::MobileAppMode
  Application::Model::Role::Page::Has::Moderation::Site
  Application::Model::Role::Page::Has::MoneyMap::Web
  Application::Model::Role::Page::Has::Tags
  );

use URI;
use PiConstants qw(
  $CONTEXT_TARGET_TYPE
  $TECHNICAL_RTB_BLOCK_ID
  );

sub accessor      {'context_on_site_campaign'}
sub db_table_name {'context_on_site_campaign'}

sub get_product_name {gettext('context_on_site_campaign')}

sub get_page_id_field_name {'page_id'}

sub get_block_model_names {
    my ($self) = @_;
    return [map {"context_on_site_$_"} qw(content direct rtb stripe adblock natural)];
}

sub db_banner_lang {$_[0]->partner_db->context_on_site_campaign_banner_lang()}

sub block_seq_db_table {$_[0]->partner_db->context_on_site_block_seq}

sub get_block_models_for_available_levels {
    # Hack for context_on_site_market_api here, just to show statistics.
    # It is not a full-featured product, it does not have real blocks
    # and should not be turned on in BS from PI2.
    #
    return $_[0]->get_block_models(), $_[0]->context_on_site_market_api;
}

sub get_need_update_in_bk_block_models {
    qw(
      context_on_site_stripe
      context_on_site_direct
      context_on_site_rtb
      context_on_site_adblock
      context_on_site_market
      context_on_site_mcb
      context_on_site_content
      context_on_site_natural
      )
}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        context_on_site_stripe     => 'Application::Model::Product::AN::ContextOnSite::Stripe',
        context_on_site_direct     => 'Application::Model::Product::AN::ContextOnSite::Direct',
        context_on_site_rtb        => 'Application::Model::Product::AN::ContextOnSite::RTB',
        context_on_site_adblock    => 'Application::Model::Product::AN::ContextOnSite::AdBlock',
        context_on_site_market     => 'Application::Model::Product::AN::ContextOnSite::Market',
        context_on_site_mcb        => 'Application::Model::Product::AN::ContextOnSite::MCB',
        context_on_site_market_api => 'Application::Model::Product::AN::ContextOnSite::MarketAPI',
        context_on_site_content    => 'Application::Model::Product::AN::ContextOnSite::Content',
        context_on_site_natural    => 'Application::Model::Product::AN::ContextOnSite::Natural',

        context_on_site_ya_categories => 'Application::Model::Product::AN::ContextOnSite::Campaign::YaCategories',
        context_on_site_mirrors       => 'Application::Model::Product::AN::ContextOnSite::Campaign::Mirrors',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        map {$self->get_description_right($_)}
          qw(
          edit_field__adblock
          edit_field__allowed_amp
          view_field__adblock
          view_field__allowed_amp
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},

        adblock => {
            db         => TRUE,
            type       => 'boolean',
            api        => 1,
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
            adjust_type => 'str',
        },
        allowed_amp => {
            db          => TRUE,
            type        => 'boolean',
            need_check  => {type => 'boolean'},
            api         => 1,
            adjust_type => 'str',
        },

        # override. depends also on tag_id (needed for adblock)
        available_blocks => {
            depends_on => [qw(multistate owner_id owner assistants tag_id)],
            get        => sub {
                return $_[0]->model->get_available_block_model_names($_[1]);
            },
            type     => 'array',
            sub_type => 'string',
            api      => 1,
        },

        #additional products
        context_on_site_market_api_count => {
            depends_on => [qw(market_api)],
            get        => sub {
                return $_[1]->{'market_api'} ? 1 : 0;
            },
            type        => 'number',
            api         => 1,
            adjust_type => 'int',
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    my $filters = $self->SUPER::get_db_filter_simple_fields();

    push(@$filters, {name => 'market_api', label => gettext('Market API')});

    return $filters;
}

sub accessor_ya_categories {'context_on_site_ya_categories'}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $fields = $self->SUPER::get_available_fields($obj);

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(
        $fields,
        {
            $accessor . '_view_field__%s' => [
                qw(
                  fast_context
                  only_picture
                  block_title
                  adblock
                  allowed_amp
                  performance_tgo_disable
                  )
            ],
        }
    );

    $fields->{'allowed_amp'} = TRUE if $obj->{'allowed_amp'};

    return $fields;
}

sub get_available_fields_depends {
    my ($self) = @_;

    my $available_fields_depends = $self->SUPER::get_available_fields_depends();

    push(@$available_fields_depends, 'allowed_amp');

    return $available_fields_depends;
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my $res = {%{$self->SUPER::collect_editable_fields($data)}, %{$self->_get_common_add_edit_fields()}};

    $res->{$_} = TRUE foreach (
        qw(
        family_filter
        behavioral_targeting
        view_images
        mirrors
        excluded_domains
        excluded_phones
        page_lang
        )
    );

    foreach (
        qw(
        reload_timeout
        fast_context
        only_picture
        user_synchronization
        banner_lang
        adblock
        market_api
        )
      )
    {
        $res->{$_} = TRUE
          if ($self->check_short_rights('edit_field__' . $_));
    }

    # TODO удалить следующий блок в одном из следующих ПР, например по PI-15793
    unless (defined($data->{'page_id'})) {
        delete($res->{$_}) foreach (qw(excluded_domains excluded_phones assistants mirrors));
    }

    if (   $self->check_short_rights('edit_field__user_synchronization')
        || $data->{'user_synchronization'})
    {
        $res->{'tag_id'} = TRUE;
    }

    if ($data->{'is_read_only'}) {
        foreach (
            qw(
            block_title
            fast_context
            ),
          )
        {
            delete($res->{$_});
        }
    }

    if ($self->check_multistate_flag($data->{'multistate'}, 'protected')
        && !$self->check_rights('edit_protected_pages'))
    {
        foreach (qw(mirrors), keys(%{$self->get_need_update_in_bk_fields()})) {
            delete($res->{$_});
        }
    }
    return $res;
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        res_fields   => $self->SUPER::_get_common_add_edit_fields(),
        right_fields => {
            edit => [
                qw(
                  allowed_amp
                  performance_tgo_disable
                  )
            ]
        }
    );

    return $fields;
}

sub on_action_approve {
    my ($self, $obj) = @_;

    my $object = $self->_get_object_fields($obj, [qw(page_id owner_id caption)]);
    $object->{accessor} = $self->accessor;
    $self->app->mail_notification->add_when_approve_page($object);
}

sub can_action_reject {TRUE}

sub on_action_reject { }

sub on_action_reset_blocked { }

sub can_action_set_blocked {TRUE}

sub on_action_set_blocked {
    my $self = shift;

    return $self->SUPER::on_action_set_blocked(@_);
}

sub can_action_set_need_approve {TRUE}

sub on_action_set_need_approve { }

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my $fields = $self->SUPER::get_need_update_in_bk_fields();

    $fields->{$_} = TRUE foreach (
        qw(
        adblock
        behavioral_targeting
        fast_context
        only_picture
        view_images
        allowed_amp
        performance_tgo_disable
        )
    );

    return $fields;
}

sub get_bk_data {
    my ($self, $page) = @_;

    my %data = $self->SUPER::get_bk_data($page);

    my %options = (
        allowed_amp             => 'check-target-ref-only',
        adblock                 => 'adblock',
        performance_tgo_disable => 'performance-tgo-disable',
    );

    #PI-10512 set redir-to-page-handler
    my @enable  = ('redir-to-page-handler');
    my @disable = ();

    foreach (keys %options) {
        if ($page->{$_}) {
            push(@enable, $options{$_});
        } else {
            push(@disable, $options{$_});
        }
    }

    %data = (
        %data,
        target_type => $CONTEXT_TARGET_TYPE,
        options     => {
            dontshowbehavior => $page->{'behavioral_targeting'} ? 0 : 1,
            dontshowsex      => $page->{'family_filter'},
            ReloadTimeout    => $page->{'reload_timeout'},
            BlockTitle       => $page->{'block_title'},
        },
        page_options => $self->page_options->get_options($page->{'page_id'}, enable => \@enable, disable => \@disable),
    );

    return %data;
}

sub block_for_site {
    my ($self, $domain_id) = @_;

    my $campaigns = $self->get_all(
        fields => [qw(id domain client_id)],
        filter => {domain_id => $domain_id, multistate => 'not blocked'},
    );

    foreach (@$campaigns) {
        $self->do_action($_->{'id'}, 'set_blocked', 'suppress_mail_notification' => 1,);
    }
}

sub get_default_settings {
    return {
        fast_context            => 1,
        only_picture            => 0,
        mobile_app_mode         => 0,
        reload_timeout          => 30,
        user_synchronization    => 0,
        performance_tgo_disable => 0,
        allowed_amp             => 1,
        allowed_turbo           => 1,
    };
}

sub get_next_id {
    my ($self, $id) = @_;

    $id++ if $id == $TECHNICAL_RTB_BLOCK_ID;

    return $id;
}

=head2 get_vip

Возвращает список page_id. На самом деле, тут возвращается не список page_id, которые VIP,
а возвращается список page_id, на которых не нужно включать стандартный набор DSP.
на которых

=cut

sub get_vip {
    my ($self, %opts) = @_;

    return map {$_->{'page_id'}} @{
        $self->partner_db->vip->get_all(
            fields => ['page_id'],
            (defined($opts{'filter'}) ? (filter => $opts{'filter'}) : ())
        )
      };
}

sub api_can_add {TRUE}

sub api_available_actions {
    return qw(
      del_mirror
      approve
      delete
      edit
      reject
      reset_blocked
      reset_read_only
      restore
      set_blocked
      set_need_approve
      set_read_only
      start
      start_testing
      stop
      );
}

sub hook_fields_validation {
    my ($self, $opts) = @_;
    $self->SUPER::hook_fields_validation($opts);

    throw Exception::Validation::BadArguments gettext('Context campaigns are not allowed on this domain')
      if $self->hook_stash->get('site')->{'is_context_blocked'};
}

sub get_opts_schema_name {'context_on_site_campaign_opts'}

TRUE;
