package Application::Model::Product::AN::ContextOnSite::Direct;

use qbit;

use base qw(
  Application::Model::Block::External::Site
  Application::Model::ValidatableMixin
  RestApi::MultistateModel
  );

consume qw(
  Application::Model::Role::Has::CustomBkData
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::Block::DesignSettings
  Application::Model::Role::Block::Has::BlocksLimit
  );

use PiVariables qw(IGNORE_RTB_ON_DIRECT_SWITHER);

use Application::Model::Product::AN::ContextOnSite::BlockTypes::Direct qw(
  @FIELDS
  @COLOR_FIELDS
  @BOOLEAN_FIELDS
  direct_appearance_fields
  direct_block_types
  %DIRECT_BLOCKS
  );

use QBit::Validator;

use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Errors;
use Exception::Validator::Fields;

use PiConstants qw($FONT_SIZES $DIRECT_PLACE_ID);

sub accessor      {'context_on_site_direct'}
sub db_table_name {'context_on_site_direct'}

sub get_campaign_model_name {
    return 'context_on_site_campaign';
}

sub get_product_name {gettext('context_on_site_direct')}

sub get_page_id_field_name {'campaign_id'}

sub public_id_prefix {'D-A-'}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        @{$self->SUPER::get_structure_rights_to_register()},
        {
            name        => 'context_on_site_direct',
            description => d_gettext('Right to manage Direct blocks on context campaigns'),
            rights      => {
                context_on_site_direct_view_zero_block => d_gettext('Right to view zero context Direct blocks'),
                context_on_site_direct_view => d_gettext('Right to view Direct blocks on context camapigns in menu'),
                context_on_site_direct_view_field__comment =>
                  d_gettext('Right to view field "comment" of direct blocks on context campaign'),
                context_on_site_direct_add_field__comment =>
                  d_gettext('Right to add field "comment" of direct blocks on context campaign'),
                context_on_site_direct_edit_field__comment =>
                  d_gettext('Right to edit field "comment" of direct blocks on context campaign'),
                context_on_site_direct_edit_field__images_first =>
                  d_gettext('Right to edit field "images_first" of direct blocks on context campaign'),
                context_on_site_direct_view_filter__login =>
                  d_gettext('Right to view field "login" of direct blocks on context campaign'),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        type => {
            db         => TRUE,
            type       => 'string',
            label      => d_gettext('Block format'),
            need_check => {
                check => sub {
                    throw Exception::Validator::Fields gettext('Invalid type block')
                      unless $_[0]->app->check_type_direct_block($_[1]);
                },
            }
        },
        limit => {
            db         => TRUE,
            type       => 'number',
            label      => d_gettext('Count ad'),
            need_check => {
                type     => 'int_un',
                optional => TRUE,
                check    => sub {
                    my ($qv, $limit) = @_;

                    throw Exception::Validator::Fields gettext('Invalid limit block')
                      unless $qv->app->check_limit($qv->data->{'type'}, $limit);
                },
            }
        },
        alternative_code => {
            db         => TRUE,
            type       => 'string',
            label      => d_gettext('Alternative code'),
            need_check => {optional => TRUE},
        },
        adaptive_width => {
            db         => TRUE,
            type       => 'string',
            label      => d_gettext('Adaptive block. width'),
            need_check => {type => 'adaptive_width'}
        },
        adaptive_height => {
            db         => TRUE,
            type       => 'string',
            label      => d_gettext('Adaptive block. height'),
            need_check => {type => 'adaptive_height'}
        },
        pi_id => {db => TRUE, type => 'number', label => d_gettext('Old Interface ID')},
        (
            map {
                $_ => {db => TRUE, type => 'boolean', need_check => {type => 'boolean'}}
              } grep {
                $_ ne 'images_first'
              } @BOOLEAN_FIELDS
        ),
        images_first => {
            db           => TRUE,
            check_rights => 'context_on_site_direct_edit_field__images_first',
            type         => 'boolean',
            need_check   => {type => 'boolean'},
        },
        (
            map {
                $_ => {
                    db   => TRUE,
                    type => 'string',
                    need_check =>
                      {type => 'color', (($_ eq 'bg_color' or $_ eq 'header_bg_color') ? (optional => TRUE) : ())}
                  }
              } @COLOR_FIELDS
        ),
        border_type     => {db => TRUE, type => 'string', need_check => {in => [qw(none block ad collapse)]}},
        title_font_size => {db => TRUE, type => 'string', need_check => {in => [qw(1 2 3)],}},
        font_family     => {
            db         => TRUE,
            type       => 'string',
            need_check => {in => ['', 'arial', 'courier new', 'tahoma', 'times new roman', 'verdana',],}
        },
        font_size     => {db => TRUE, type => 'string', need_check => {in => $FONT_SIZES,}},
        block_caption => {
            depends_on => ['type', 'limit'],
            get        => sub {
                throw gettext('Block with type "%s" not found', $_[1]->{'type'})
                  unless $_[0]->{'BLOCK_TYPES_CAPTION'}{$_[1]->{'type'}};

                my $limit =
                  exists($DIRECT_BLOCKS{$_[1]->{'type'}}->{'banners'})
                  ? $DIRECT_BLOCKS{$_[1]->{'type'}}->{'banners'}
                  : $_[1]->{'limit'};

                $_[0]->{'BLOCK_TYPES_CAPTION'}{$_[1]->{'type'}}($limit);
            },
            type => 'string',
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $parent_filter = $self->SUPER::get_structure_model_filter();
    return {
        db_accessor => $parent_filter->{'db_accessor'},
        fields      => {%{$parent_filter->{'fields'}}, active => {type => 'number', label => d_gettext('Active')},}
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        (
            $self->check_rights('view_search_filters__login') ? {name => 'login', label => gettext('Login')}
            : ()
        ),
        (
            $self->check_rights('view_search_filters__user_type') ? {name => 'user_type', label => gettext('User type')}
            : ()
        ),
        {name => $self->get_page_id_field_name(), label => gettext('Page ID')},
        {name => 'id',                            label => gettext('Block ID')},
        {name => 'caption',                       label => gettext('Block\'s caption')},
        {name => 'campaign.all_domain',           label => gettext('Domain and mirror')},
        {name => 'multistate',                    label => gettext('Status')},
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_pgettext('Block multistate', 'New'),
        multistates => [
            [deleted           => d_pgettext('Block multistate', 'Archived')],
            [working           => d_pgettext('Block multistate', 'Working')],
            [check_statistics  => d_pgettext('Block multistate', 'Check statistics')],
            [deleted_with_page => d_pgettext('Block multistate', 'Archived with page'), private => TRUE],
            [need_update       => d_pgettext('Block multistate', 'Need update')],
            [updating          => d_pgettext('Block multistate', 'Updating')],
        ],
        actions => {
            delete                 => d_pgettext('Block action', 'Archive'),
            delete_with_page       => d_pgettext('Block action', 'Archive with page'),
            restore                => d_pgettext('Block action', 'Restore'),
            restore_with_page      => d_pgettext('Block action', 'Restore with page'),
            start                  => d_pgettext('Block action', 'Start'),
            stop                   => d_pgettext('Block action', 'Stop'),
            set_check_statistics   => d_pgettext('Block action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Block action', 'Reset "check_statistics"'),
        },
        right_name_prefix => $self->accessor . '_',
        right_group       => [$self->accessor => d_gettext('Right to manage context direct')],
        right_actions     => {
            add  => d_pgettext('Block action', 'Add'),
            edit => d_pgettext('Block action', 'Edit'),
            set_need_update =>
              {label => d_pgettext('Block action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Block action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Block action', 'Stop update'),  dont_write_to_action_log => TRUE},
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action    => 'delete_with_page',
                from      => 'not deleted',
                set_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page',
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action    => 'start',
                from      => 'not working and not deleted',
                set_flags => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
            },
            {
                action    => 'set_check_statistics',
                from      => 'not deleted',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('block_caption')) {
        $fields->{'BLOCK_TYPES_CAPTION'} =
          {map {$_->{'id'} => $_->{'constuctor_caption'}} @{$self->direct_block_types()}};
    }
}

sub get_available_fields_depends {
    [qw(multistate)];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor      = $self->accessor();
    my $page_accessor = $self->get_campaign_model_name();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s' => [qw( comment is_tutby)],
            $accessor . '_edit_field__%s' => 'images_first',

            $page_accessor . '_view_field__%s'    => [qw(login domain_id)],
            $page_accessor . '_view_field__owner' => [qw(owner_client_id)],
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    $fields->{'page_id'}          = TRUE;
    $fields->{'campaign_id'}      = TRUE;
    $fields->{'alternative_code'} = TRUE;

    $fields->{'comment'} = TRUE if $self->check_short_rights('add_field__comment');

    return $fields;
}

sub get_actions_depends {
    [qw(id campaign_id page multistate cur_user_is_read_assistant is_protected pi_id)];
}

sub get_editable_fields_depends {
    [qw(page_id campaign_id id multistate is_protected cur_user_is_read_assistant page)];
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my %res = %{$self->_get_common_add_edit_fields()};

    $res{'comment'} = TRUE
      if $self->check_rights('context_on_site_direct_edit_field__comment');

    return \%res;
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        no_right_fields => [
            qw(
              adaptive_width
              adaptive_height
              alternative_code
              caption
              ),
            $self->direct_block_fields,
            @{arrays_difference([direct_appearance_fields()], [qw(images_first)])},
        ],
        right_fields => {edit => ['images_first']}
    );

    $fields->{'type'} = $self->get_types_direct_blocks() if $fields->{'type'};

    return $fields;
}

sub hook_check_available_fields {
    my ($self, $opts) = @_;

    $opts->{'page_id'} //= delete($opts->{$self->get_page_id_field_name()});
    $self->SUPER::hook_check_available_fields($opts);
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    $opts->{alternative_code} = '' unless $opts->{alternative_code};
    $self->SUPER::hook_fields_processing_before_validation($opts);
}

sub hook_preparing_fields_to_save {
    my ($self, $opts) = @_;

    $opts->{$self->get_page_id_field_name()} = delete($opts->{'page_id'}) if exists $opts->{'page_id'};
    $self->SUPER::hook_preparing_fields_to_save($opts);
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $self->partner_db->transaction(
        sub {
            $opts->{id} = $self->page->get_next_block_id($opts->{'page_id'});
        }
    );
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    my $block_id = $self->hook_stash->get('id');

    $self->maybe_do_action($block_id, 'start');

    $self->update_in_bk($block_id) if $self->need_update_in_bk([keys(%$opts)]);
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    my $object = $self->hook_stash->get('id');

    $self->update_in_bk($object) if $self->need_update_in_bk([keys(%$opts)]);
}

sub duplicate {
    my ($self, %opts) = @_;

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {public_id => {type => 'publicid'}}
        },
        app   => $self,
        throw => TRUE
    );

    my $entity =
      $self->get($opts{'public_id'}, fields => [sort keys(%{$self->get_add_fields()}), 'is_deleted', 'is_protected'],);

    throw Exception::Validation::BadArguments gettext('You can not duplicate the deleted block')
      if delete($entity->{'is_deleted'});

    throw Exception::Validation::BadArguments gettext('You can not duplicate a block of protected page')
      if delete($entity->{'is_protected'}) && !$self->check_rights('edit_protected_pages');

    return $self->add(%$entity);
}

# Basic behaviour overridden. Cannot restore old Direct migrated from PI1.
sub can_action_restore {
    my ($self, $block) = @_;

    my $fields = $self->_get_object_fields($block, [qw(page pi_id)]);

    return FALSE if $fields->{'pi_id'};

    return $self->page->check_action($fields->{'page'}, 'restore_block');
}

sub fix_template {
    my ($self, $qv) = @_;

    my $template = $qv->template;

    my $images_first_skipped;
    my $block_type = $qv->data->{'type'};
    if (defined($block_type) && $DIRECT_BLOCKS{$block_type} && $DIRECT_BLOCKS{$block_type}{'skip_fields'}) {
        foreach (@{$DIRECT_BLOCKS{$block_type}{'skip_fields'}}) {
            $template->{'fields'}{$_} = {eq => undef};
            $images_first_skipped = TRUE if $_ eq 'images_first';
        }
    }
    $qv->data->{'images_first'} = $images_first_skipped ? undef : ($qv->data->{'images_first'} // 0);

    if ($qv->data->{'border_type'} && $qv->data->{'border_type'} eq 'none') {
        foreach (
            qw(
            border_color
            border_radius
            )
          )
        {
            $template->{'fields'}{$_} = {eq => undef};
        }
    }

    if ($qv->data->{'no_sitelinks'}) {
        $template->{'fields'}{'sitelinks_color'} = {eq => undef};
    }

    $qv->template($template);
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_campaign($filter);
    $filter = $self->limit_filter_view_zero_block($filter);
    $filter = $self->limit_filter_active($filter);

    return $filter;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = map {$_ => TRUE} qw(
      adaptive_width
      adaptive_height
      ), $self->direct_block_fields, $self->direct_appearance_fields;

    return \%fields_need_update;
}

sub api_available_actions {qw(duplicate)}

sub boolean_fields {
    return {map {$_ => TRUE} @BOOLEAN_FIELDS};
}

sub check_limit {
    my ($self, $type, $limit) = @_;

    return FALSE unless $DIRECT_BLOCKS{$type};

    if (exists($DIRECT_BLOCKS{$type}->{'banners'})) {
        return FALSE if defined($limit);
    } elsif (exists($DIRECT_BLOCKS{$type}->{'min_banners'}) && exists($DIRECT_BLOCKS{$type}->{'max_banners'})) {
        return FALSE
          if $limit < $DIRECT_BLOCKS{$type}->{'min_banners'} || $limit > $DIRECT_BLOCKS{$type}->{'max_banners'};
    } else {
        throw Exception::Validation::BadArguments gettext('Wrong format type');
    }

    return TRUE;
}

sub check_type_direct_block {
    my ($self, $type) = @_;

    return $DIRECT_BLOCKS{$type};
}

sub direct_block_fields {
    return qw(
      limit type
      );
}

sub direct_blocks {
    return \%DIRECT_BLOCKS;
}

sub get_bk_block_data {
    my ($self, $block) = @_;

    return {};
}

sub get_bk_block_data_key {'direct_blocks'}

sub get_bk_common_data {
    return (places => {$DIRECT_PLACE_ID => {}});
}

sub get_cnt {
    my ($self, $arr_campaign_id) = @_;

    my $data = $self->partner_db->query->select(
        table  => $self->partner_db_table(),
        fields => {
            $self->get_page_id_field_name() => '',
            cnt => {count => ['id']}
        },
        filter => [
            AND => [
                [$self->get_page_id_field_name() => 'IN'     => \$arr_campaign_id],
                [id                              => '<>'     => \0],
                [multistate                      => 'NOT IN' => \$self->get_multistates_by_filter('deleted')],
                IGNORE_RTB_ON_DIRECT_SWITHER()
                ? ()
                : [active => '=' => \1]
            ]
        ],
    )->group_by($self->get_page_id_field_name())->get_all();

    return {map {$_->{$self->get_page_id_field_name()} => $_->{'cnt'}} @$data};
}

sub get_types_direct_blocks {
    my ($self) = @_;

    my $blocks = $self->direct_block_types();

    foreach my $b (@$blocks) {
        delete($b->{'constuctor_caption'});
        $b->{'caption'} = $b->{'caption'}->();
    }

    return $blocks;
}

sub related_models {
    my ($self) = @_;

    return {
        pages => {
            accessor => 'page',
            filter   => sub {
                +{page_id => array_uniq(map {$_->{'page_id'}} @{$_[1]})};
            },
            key_fields => [qw(page_id)],
        }
    };
}

sub design_field {'Design'}

TRUE;
