package Application::Model::Product::AN::ContextOnSite::Stripe;

use qbit;

use base qw(
  Application::Model::Product::Stripe
  Application::Model::Block::External::Site
  Application::Model::ValidatableMixin
  RestApi::MultistateModel
  );

consume qw(
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Actions
  Application::Model::Role::Block::Has::BlocksLimit
  Application::Model::Role::Has::CreateDate
  );

use Exception::Denied;
use Exception::Validator::Fields;

use constant PLACE_ID => 956;

sub accessor      {'context_on_site_stripe'}
sub db_table_name {'context_on_site_stripe'}

sub get_campaign_model_name {
    return 'context_on_site_campaign';
}

sub get_product_name {gettext('context_on_site_stripe')}

sub get_page_id_field_name {'campaign_id'}

sub public_id_prefix {'S-A-'}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        %{$class->SUPER::get_structure_model_accessors()},
        context_on_site_campaign => 'Application::Model::Product::AN::ContextOnSite::Campaign',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            name        => 'context_on_site_stripe',
            description => d_gettext('Right to manage Stripe blocks on sites'),
            rights      => {
                context_on_site_stripe_view_all => d_gettext('Right to view all stripe blocks on sites'),
                context_on_site_stripe_view_field__comment =>
                  d_gettext('Right to view field "comment" of stripe blocks on sites'),
                context_on_site_stripe_edit_field__comment =>
                  d_gettext('Right to edit field "comment" of stripe blocks on sites'),
                context_on_site_stripe_view_action_log =>
                  d_gettext('Right to view action log of stripe blocks on sites'),
                context_on_site_stripe_view => d_gettext('Right to view stripe blocks on sites in menu'),
                context_on_site_stripe_edit => d_gettext('Right to edit stripe block to site'),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->Application::Model::Block::External::Site::get_structure_model_fields()},
        %{$self->Application::Model::Product::Stripe::get_structure_model_fields()},
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $parent_filter = $self->SUPER::get_structure_model_filter();
    return {
        db_accessor => $parent_filter->{'db_accessor'},
        fields      => {
            %{$parent_filter->{'fields'}},
            title   => {type => 'text', label => d_gettext('Title')},
            comment => {type => 'text', label => d_gettext('Comment')},
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        (
            $self->check_rights('view_search_filters__login')
            ? {name => 'campaign.owner.login', label => gettext('Login')}
            : ()
        ),
        (
            $self->check_rights('view_search_filters__user_type') ? {name => 'user_type', label => gettext('User type')}
            : ()
        ),
        {name => 'campaign_id',         label => gettext('Page ID')},
        {name => 'id',                  label => gettext('Block ID')},
        {name => 'title',               label => gettext('Block\'s caption')},
        {name => 'campaign.all_domain', label => gettext('Domain and mirror')},
        (
            $self->check_rights('context_on_site_stripe_view_field__comment')
            ? {name => 'comment', label => gettext('Comment')}
            : ()
        ),
        {name => 'multistate', label => gettext('Status')},
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_pgettext('Block multistate', 'New'),
        multistates => [
            [deleted           => d_pgettext('Block multistate', 'Archived')],
            [working           => d_pgettext('Block multistate', 'Working')],
            [check_statistics  => d_pgettext('Block multistate', 'Check statistics')],
            [deleted_with_page => d_pgettext('Block multistate', 'Archived with page'), private => TRUE],
            [need_update       => d_pgettext('Block multistate', 'Need update')],
            [updating          => d_pgettext('Block multistate', 'Updating')],
        ],
        actions => {
            start                  => d_pgettext('Block action', 'Start'),
            stop                   => d_pgettext('Block action', 'Stop'),
            edit                   => d_pgettext('Block action', 'Edit'),
            delete                 => d_pgettext('Block action', 'Archive'),
            delete_with_page       => d_pgettext('Block action', 'Archive with page'),
            restore                => d_pgettext('Block action', 'Restore'),
            restore_with_page      => d_pgettext('Block action', 'Restore with page'),
            set_check_statistics   => d_pgettext('Block action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Block action', 'Reset "check_statistics"'),
        },
        right_name_prefix => $self->accessor . '_',
        right_actions     => {
            add => d_pgettext('Block action', 'Add'),
            set_need_update =>
              {label => d_pgettext('Block action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Block action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Block action', 'Stop update'),  dont_write_to_action_log => TRUE},
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action    => 'delete_with_page',
                from      => 'not deleted',
                set_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page',
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action    => 'start',
                from      => 'not working and not deleted',
                set_flags => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action    => 'set_check_statistics',
                from      => 'not deleted',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('disposition_name') || $fields->need('editable_fields') || $fields->need('available_fields')) {
        $fields->{'disposition_ids'} = $self->get_disposition_ids();
    }
    if ($fields->need('interaction_name') || $fields->need('editable_fields') || $fields->need('available_fields')) {
        $fields->{'interaction_ids'} = $self->get_interaction_with_content_ids();
    }
    if ($fields->need('type_name') || $fields->need('editable_fields') || $fields->need('available_fields')) {
        $fields->{'type_ids'} = $self->get_type_ids();
    }
    if ($fields->need('animation_name') || $fields->need('editable_fields') || $fields->need('available_fields')) {
        $fields->{'animation_ids'} = $self->get_animation_ids();
    }
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    foreach (qw(interaction_id type_id animation interaction_name type_name animation_name)) {
        delete($fields{$_});
    }

    # lists of possible constant values
    $fields{'disposition_id'} = $self->get_disposition_ids();
    if ($obj->{'disposition_id'} == $Application::Model::Product::Stripe::DISPOSITION_TOP) {
        $fields{'interaction_id'}   = $self->get_interaction_with_content_ids();
        $fields{'interaction_name'} = TRUE;

        if ($obj->{'interaction_id'} == $Application::Model::Product::Stripe::INTERACTION_COVERS_CONTENT) {
            $fields{'type_name'}      = TRUE;
            $fields{'animation_name'} = TRUE;

            $fields{'type_id'}   = $self->get_type_ids();
            $fields{'animation'} = $self->get_animation_ids();
        }
    }

    my $accessor      = $self->accessor();
    my $page_accessor = $self->get_campaign_model_name();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s' => [qw( comment is_tutby)],

            $page_accessor . '_view_field__%s'    => [qw(login domain_id)],
            $page_accessor . '_view_field__owner' => [qw(owner_client_id)],
        }
    );

    return \%fields;
}

sub collect_editable_fields {
    my ($self, $data, $fields) = @_;

    my %res = ();

    $res{'comment'} = TRUE
      if $self->check_short_rights('edit_field__comment');

    $res{$_} = TRUE foreach (qw(title));

    if (defined($fields)) {
        $res{'disposition_id'} = $fields->{'disposition_ids'};
        $res{'interaction_id'} = $fields->{'interaction_ids'};
        $res{'type_id'}        = $fields->{'type_ids'};
        $res{'animation'}      = $fields->{'animation_ids'};
    } else {
        $res{'disposition_id'} = $self->get_disposition_ids();
        $res{'interaction_id'} = $self->get_interaction_with_content_ids();
        $res{'type_id'}        = $self->get_type_ids();
        $res{'animation'}      = $self->get_animation_ids();
    }

    return \%res;
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_campaign($filter);

    return $filter;
}

# NOTE! Т.к. need_update_in_bk() не используется, просто нужно реализловать абстрактный метод из (Application::Model::Block)
sub get_need_update_in_bk_fields {return {}}

sub get_animation_ids {
    return {0 => gettext('No animation'), 1 => gettext('With animation')};
}

sub get_bk_common_data {
    my ($self, $blocks) = @_;

    $self->normalize_appearance($_) foreach @$blocks;

    return (
        banners_count => 9,
        places        => {
            PLACE_ID() => {
                StripeAnimation => $blocks->[0]->{'animation'},
                StripeType      => $blocks->[0]->{'type_id'},
            },
        },
    );
}

sub get_type_ids {
    return {
        1 => gettext('Scroll off'),
        4 => gettext('Scroll on'),
    };
}

#Api methods

sub get_actions_depends {
    [
        qw(
          id
          campaign_id
          multistate
          page
          cur_user_is_read_assistant
          is_protected
          )
    ];
}

sub get_editable_fields_depends {
    [
        qw(
          page_id
          page
          campaign_id
          id
          multistate
          is_protected
          cur_user_is_read_assistant
          )
    ];
}

TRUE;
