package Application::Model::Product::AN::Distribution;

use qbit;
use base qw(
  Application::Model::PageMinimal
  RestApi::MultistateModel
  );

consume qw(
  Application::Model::Role::Has::Page::Patch
  );

sub accessor               {'distribution_campaign'}
sub db_table_name          {'distribution_campaign'}
sub get_product_name       {gettext('distribution_campaign')}
sub get_page_id_field_name {'page_id'}
sub get_block_model_names  {[]}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        # overwrite
        is_yandex_page => {
            db          => TRUE,
            db_expr     => \0,
            type        => 'boolean',
            adjust_type => 'int',
        },
        product_type => {
            db      => TRUE,
            db_expr => \'distribution',
            type    => 'string'
        },
    };
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    my $accessor = $self->accessor();
    my (undef, $right_group) = $accessor =~ /((.+)_)campaign$/;

    return {
        empty_name  => d_gettext('New'),
        multistates => [
            [working => d_pgettext('Campaign status', 'Working')],
            [testing => d_pgettext('Campaign status', 'Testing')],
            [stopped => d_pgettext('Campaign status', 'Stopped')],
            [deleted => d_pgettext('Campaign status', 'Archived'), private => TRUE],
            [
                balance_registered => d_pgettext('Campaign status', 'Added to Balance'),
                private            => TRUE,
                deprecated         => TRUE
            ],
            [read_only => d_pgettext('Campaign status', 'Read only'), private => TRUE],
            [
                check_statistics => d_pgettext('Campaign status', 'Check statistics'),
                private          => TRUE,
                deprecated       => TRUE
            ],
            [blocked      => d_pgettext('Campaign status', 'Blocked'),      private => TRUE, deprecated => TRUE],
            [need_approve => d_pgettext('Campaign status', 'Need approve'), private => TRUE, deprecated => TRUE],
            [rejected     => d_pgettext('Campaign status', 'Rejected'),     private => TRUE, deprecated => TRUE],
            [protected    => d_pgettext('Campaign status', 'Protected')],
            [need_update  => d_pgettext('Campaign status', 'Need update')],
            [updating     => d_pgettext('Campaign status', 'Updating')],
        ],
        actions => {
            can_update_in_bk => d_pgettext('Campaign action', 'Can update in bk'),
            restore_block    => d_pgettext('Campaign action', 'Restore block on campaign'),
        },
        right_actions => {
            add             => d_pgettext('Campaign action', 'Add'),
            edit            => d_pgettext('Campaign action', 'Edit'),
            delete          => d_pgettext('Campaign action', 'Archive'),
            restore         => d_pgettext('Campaign action', 'Restore'),
            start           => d_pgettext('Campaign action', 'Start'),
            stop            => d_pgettext('Campaign action', 'Stop'),
            start_testing   => d_pgettext('Campaign action', 'Start testing'),
            stop_testing    => d_pgettext('Campaign action', 'Stop testing'),
            set_protected   => d_pgettext('Campaign action', 'Set protected'),
            reset_protected => d_pgettext('Campaign action', 'reset protected'),
            set_read_only   => d_pgettext('Campaign action', 'Set read only'),
            reset_read_only => d_pgettext('Campaign action', 'Reset read only'),
            set_need_update =>
              {label => d_pgettext('Campaign action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Campaign action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Campaign action', 'Stop update'),  dont_write_to_action_log => TRUE},
        },
        right_group        => [$right_group => d_gettext('Right to manage distribution campaigns')],
        right_name_prefix  => $self->accessor . '_',
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
                # set_flags => ['stopped'],
            },
            {
                action => 'edit',
                from   => 'not deleted'
            },
            {
                action    => 'delete',
                from      => 'not (working or testing or deleted)',
                set_flags => ['deleted']
            },
            {
                action      => 'restore',
                from        => 'deleted',
                reset_flags => ['deleted'],
                set_flags   => ['stopped'],
            },
            {
                action      => 'start',
                from        => 'not (working or deleted or protected)',
                set_flags   => ['working'],
                reset_flags => ['testing', 'stopped']
            },
            {
                action      => 'stop',
                from        => 'working and not protected',
                set_flags   => ['stopped'],
                reset_flags => ['working'],
            },
            {
                action      => 'start_testing',
                from        => 'stopped and not protected',
                set_flags   => ['testing'],
                reset_flags => ['stopped'],
            },
            {
                action      => 'stop_testing',
                from        => 'testing and not protected',
                set_flags   => ['stopped'],
                reset_flags => ['testing'],
            },
            {
                action    => 'set_protected',
                from      => 'not protected',
                set_flags => ['protected'],
            },
            {
                action      => 'reset_protected',
                from        => 'protected',
                reset_flags => ['protected'],
            },
            {
                action    => 'set_read_only',
                from      => 'not read_only',
                set_flags => ['read_only'],
            },
            {
                action      => 'reset_read_only',
                from        => 'read_only',
                reset_flags => ['read_only'],
            },
            {
                action    => 'set_need_update',
                from      => 'not deleted',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update or updating',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ]
    };
}

# sub pre_process_fields {}

sub get_actions_depends {
    [qw(id multistate )];
}

sub get_available_fields_depends {[qw(id multistate)]}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my $fields = {map {$_ => TRUE} keys %$model_fields};

    my $accessor = $self->accessor();
    $self->delete_field_by_rights(
        $fields,
        {
            # <Право или маска под право>            <Поля котрые оно закрывает>
            $accessor . '_view_field__%s' => [qw(comment)],
        }
    );

    return $fields;
}

sub get_add_fields {
    my ($self) = @_;

    return $self->_get_common_add_edit_fields();
}

sub _get_common_add_edit_fields {
    my ($self, $add_fields, $edit_fields) = @_;

    my $fields = $self->get_fields_by_right(
        res_fields      => $self->SUPER::_get_common_add_edit_fields(),
        no_right_fields => [
            qw(
              caption
              )
        ],
        right_fields => {edit => ['patch']}
    );

    return $fields;
}

sub get_editable_fields_depends {
    [
        qw(
          id
          is_protected
          is_read_only
          multistate
          owner_id
          page_id
          )
    ];
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my $fields = {%{$self->SUPER::collect_editable_fields($data)}, %{$self->_get_common_add_edit_fields()}};

    if ($self->check_multistate_flag($data->{'multistate'}, 'protected')
        && !$self->check_rights('edit_protected_pages'))
    {
        foreach (keys(%{$self->get_need_update_in_bk_fields()})) {
            delete $fields->{$_};
        }
    }

    return $fields;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $opts->{'caption'} //= '';
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    my $id = $self->hook_stash->get('id');

    if (   $self->need_update_in_bk($self->hook_stash->get('fields'))
        && $self->check_action($id, 'set_need_update'))
    {
        $self->do_action($id, 'set_need_update');
    }
}

sub query_filter {
    my ($self, $filter, $opts) = @_;

    $filter = $self->limit_filter_by_robot_assistant_or_owner($filter, $opts);

    return $filter;
}

sub on_action_add {
    my ($self, $obj, %opts) = @_;

    # обычно это делает AutoStart, но не для этой модели
    $self->do_action($obj, 'start');
}

sub on_action_start {
    my ($self, $obj, %opts) = @_;
    $self->do_action($obj, 'set_need_update');
}

sub on_action_start_testing {
    my ($self, $obj, %opts) = @_;
    $self->do_action($obj, 'set_need_update');
}

sub on_action_stop {
    my ($self, $obj, %opts) = @_;
    $self->do_action($obj, 'set_need_update');
}

sub on_action_stop_testing {
    my ($self, $obj, %opts) = @_;
    $self->do_action($obj, 'set_need_update');
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my $fields = $self->SUPER::get_need_update_in_bk_fields();

    $fields->{$_} = TRUE foreach (qw(caption patch));

    return $fields;
}

sub api_can_add    {TRUE}
sub api_can_delete {TRUE}

sub check_action {
    my ($self, $object, $action) = @_;

    return FALSE
      unless $self->check_short_rights('edit');

    my $fields = $self->_get_object_fields($object, [qw(multistate)]);

    my $is_protected = $self->check_multistate_flag($fields->{'multistate'}, 'protected');
    if ($is_protected) {
        return FALSE unless $self->check_rights('edit_protected_pages');
    }

    $self->QBit::Application::Model::Multistate::DB::check_action($object, $action);
}

sub _data_for_bk {
    my ($self, $page, %opts) = @_;

    my %data = hash_transform(
        $page,
        [
            qw(
              page_id
              create_date
              is_yandex_page
              update_time
              ),
        ],
        {caption => 'page_caption',}
    );

    my $bk_data = {%data, %{$self->get_bk_data($page)}};

    if ($self->can('apply_bk_data_patch')) {
        $bk_data = $self->apply_bk_data_patch($page, $bk_data);
    }

    return $bk_data;
}

sub get_bk_data {
    my ($self, $page) = @_;

    # иначе логин недоступен users_view_field__login
    my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_all users_view_field__login));
    my $page_owner_login = $self->users->get($page->{'owner_id'}, fields => ['login'])->{'login'};
    undef $tmp_rights;

    my $bk_data = {
        banners_count => 9,
        login         => $page_owner_login,
        options       => "",
        places        => {},
        product_id    => $self->accessor,
        state         => 0,
        target_type   => 8,
    };

    return $bk_data;
}

sub api_available_actions {
    return qw(
      add
      edit
      delete
      restore
      start
      stop
      start_testing
      stop_testing
      set_protected
      reset_protected
      set_read_only
      reset_read_only
      );
}

sub api_can_edit {TRUE}

sub get_block_models {()}

sub is_external_page {
    # переопределение, чтобы не делалась проверка договора и не отрабаотывался replace_multi в global_excluded_phones
    return FALSE;
}
sub is_internal_page {return FALSE}

1;
