package Application::Model::Product::AN::Mirrors;

use qbit;

use base qw(Application::Model::Product::Mirrors Application::Model::Common);

consume qw(
  Application::Model::Role::Has::CreateDate
  );

use Exception::DB::DuplicateEntry;
use Exception::Multistate::BadAction;
use Exception::Validation::BadArguments;
use Exception::Validation::BadArguments::InvalidMirror;
use PiConstants qw($MYSQL_MIN_DATETIME);
use Utils::Logger qw(INFOF);

sub get_structure_model_accessors {
    return {
        api_http_gozora => 'QBit::Application::Model::API::Yandex::HTTPGoZora',
        moderation      => 'Application::Model::Moderation',
    };
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},

        campaign_id        => {default => TRUE, db          => TRUE,     adjust_type => 'str',},
        domain             => {default => TRUE, db          => TRUE},
        id                 => {db      => TRUE, pk          => TRUE},
        multistate         => {db      => TRUE, type        => 'number', adjust_type => 'str',},
        auto               => {db      => TRUE, adjust_type => 'str',},
        waiting_moderation => {db      => TRUE,},
        moderation_status  => {
            default    => TRUE,
            depends_on => ['multistate'],
            get        => sub {
                $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'approved') ? return 'approved'
                  : $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'rejected') ? return 'rejected'
                  : return 'new'
                  if (!$_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'deleted'));
            },
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_filter()},

        db_accessor => 'partner_db',
        fields      => {
            campaign_id        => {type => 'number',     label      => d_gettext('Campaign ID')},
            domain             => {type => 'domain',     value_type => 'text', label => d_gettext('Mirror')},
            id                 => {type => 'number',     label      => d_gettext('ID')},
            multistate         => {type => 'multistate', label      => d_gettext('Status')},
            auto               => {type => 'boolean',    label      => d_gettext('Status')},
            waiting_moderation => {type => 'date',       label      => d_gettext('Waiting moderation'),},
        }
    };
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_multistates_graph()},

        empty_name  => d_pgettext('Mirror multistate', 'New'),
        multistates => [
            [approved => d_pgettext('Mirror multistate', 'Approved')],
            [rejected => d_pgettext('Mirror multistate', 'Rejected')],
            [deleted  => d_pgettext('Mirror multistate', 'Deleted')],
        ],
        actions => {
            approve          => d_pgettext('Mirror action', 'Approve'),
            reject           => d_pgettext('Mirror action', 'Reject'),
            delete           => d_pgettext('Mirror action', 'Delete'),
            set_need_approve => d_pgettext('Mirror action', 'Set need approve'),
        },
        multistate_actions => [
            {
                action      => 'approve',
                from        => '__EMPTY__ or not __EMPTY__',
                set_flags   => ['approved'],
                reset_flags => ['rejected', 'deleted']
            },
            {
                action    => 'reject',
                from      => '__EMPTY__',
                set_flags => ['rejected'],
            },
            {
                action      => 'delete',
                from        => 'approved',
                set_flags   => ['deleted'],
                reset_flags => ['approved'],
            },
            {
                action      => 'set_need_approve',
                from        => 'rejected or deleted',
                reset_flags => ['rejected', 'deleted'],
            },
        ],
    };
}

sub replace {
    my ($self, $obj, $mirrors, $is_bad_domains) = @_;

    my %db_mirrors =
      map {$_->{domain} => $_->{id}}
      @{$self->get_all(fields => ['domain', 'id', 'multistate'], filter => {campaign_id => $obj->{id}})};

    # NOTE! Не создаем авто заявки на то что уже есть в базе, так как новые заявки пользователей затирают отклоненные (#PI-10919)
    $mirrors = [grep {!$db_mirrors{$_}} @$mirrors] if $is_bad_domains;

    my %incomming_mirrors = map {get_domain_or_throw($_) => TRUE} @$mirrors;

    #### ADD
    my @will_be_added = grep {!$db_mirrors{$_}} sort keys %incomming_mirrors;
    if (@will_be_added && $is_bad_domains) {

        # Фильтруем новые bad_domains через поддомены основного домена, а также через api_http_gozora
        my $failed_mirrors = {};
        foreach my $domain (@will_be_added) {
            my $is_fail = 0;
            try {
                $self->check_page_domain($obj->{id}, $domain, keys %db_mirrors);
            }
            catch {
                INFOF 'Check domain "%s" of id=%d failed: %s', $domain, $obj->{id}, shift->message();
                $is_fail = 1;
            };

            unless ($is_fail) {
                unless ($self->api_http_gozora->is_site_working($domain)) {
                    INFOF(q[Bad domain "%s" doesn't exist in zora/url_test2], $domain);
                    $is_fail = 1;
                }
            }
            $failed_mirrors->{$domain} = $is_fail;
        }

        if (%$failed_mirrors) {
            @will_be_added = grep {!$failed_mirrors->{$_}} @will_be_added;
            map {delete $incomming_mirrors{$_}} grep {$failed_mirrors->{$_}} sort keys %incomming_mirrors;
        }
    }

    $self->add($obj->{id}, \@will_be_added, undef, $is_bad_domains) if @will_be_added;

    #### DELETE
    my @will_be_deleted =
      $is_bad_domains
      ? ()
      : grep {!$incomming_mirrors{$_} and $self->check_action($db_mirrors{$_}, 'delete')} keys %db_mirrors;

    $self->delete($obj->{id}, \@will_be_deleted) if @will_be_deleted;

    #### APPROVE
    my @will_be_refreshed =
      grep {$db_mirrors{$_} and $self->check_action($db_mirrors{$_}, 'set_need_approve')} keys %incomming_mirrors;

    ### PI-10220 – Принимаем зеркала Дзена без модерации
    my @auto_approve_mirrors = map {$_->{id}} grep {$_->{domain} eq 'zen.yandex.ru'} @{
        $self->get_all(
            fields => ['domain', 'id'],
            filter => {
                campaign_id => $obj->{id},
                domain      => [@will_be_added, @will_be_refreshed]
            },
        )
      };
    $self->do_action($_, 'approve') foreach @auto_approve_mirrors;

    my @refreshed_mirrors = map {$_->{id}} grep {$self->check_action($_, 'set_need_approve')} @{
        $self->get_all(
            fields => ['id'],
            filter => {
                campaign_id => $obj->{id},
                domain      => \@will_be_refreshed
            },
        )
      };

    my @to_moderate = (
        @refreshed_mirrors,
        map {$_->{id}} @{
            $self->get_all(
                fields => ['id'],
                filter => {
                    campaign_id => $obj->{id},
                    domain      => \@will_be_added,
                    multistate  => 'not approved'
                },
            )
          }
    );

    if (!$is_bad_domains && $self->check_rights('add_mirrors_without_moderation')) {
        $self->do_action($_, 'approve') foreach @to_moderate;
    } else {
        $self->do_action($_, 'set_need_approve') foreach @refreshed_mirrors;

        $self->moderation->add_request($self->{'accessor'}, {campaign_id => $obj, moderation_elements => \@to_moderate})
          if @to_moderate;
    }
}

sub add {
    my ($self, $campaign_id, $domains, $is_approved, $is_auto) = @_;
    my @added_domains = ();

    $domains = [map {get_domain_or_throw($_)} @$domains];

    my %domain_id_pairs =
      map {$_->{domain} => $_->{id}}
      @{$self->get_all(fields => ['domain', 'id'], filter => {campaign_id => $campaign_id, domain => $domains})};

    foreach (@$domains) {
        if (!$domain_id_pairs{$_}) {
            try {
                $domain_id_pairs{$_} = $self->partner_db_table()->add(
                    {
                        campaign_id => $campaign_id,
                        domain      => $_,
                        auto        => $is_auto ? 1 : 0,
                        create_date        => curdate(oformat => 'db_time'),
                        waiting_moderation => $MYSQL_MIN_DATETIME,
                    }
                );
            }
            catch Exception::DB::DuplicateEntry with {
                throw Exception::Validation::BadArguments::InvalidMirror gettext('Mirror "%s" is exists', $_);
            };
            push @added_domains, $_;
        }
    }
    return \@added_domains;
}

sub delete {
    my ($self, $campaign_id, $domains) = @_;

    foreach (@$domains) {
        my $domain = get_domain($_)
          // throw Exception::Validation::BadArguments::InvalidMirror gettext('Invalid mirror: "%s"', $_);

        my $mirror_id =
          $self->get_all(fields => ['id'], filter => {campaign_id => $campaign_id, domain => $_})->[0]{id};

        $self->do_action($mirror_id, 'delete');
    }

    return $domains;
}

sub _set_need_update_page {
    my ($self, $obj, %opts) = @_;

    my $campaign_model = $self->page_model_name();

    $obj = $self->_get_object_fields($obj, ['campaign_id']);

    my $campaign = $self->$campaign_model->get_all(
        fields => ['id'],
        filter => [
            AND => [
                {id         => $obj->{campaign_id}},
                {multistate => $self->$campaign_model->get_multistate_by_action('set_need_update')},
                {multistate => 'not protected'},
            ]
        ]
    )->[0];

    if ($campaign) {
        my $tmp = $self->app->add_tmp_rights($self->$campaign_model->get_right('edit'));
        $self->$campaign_model->do_action($campaign->{id}, 'set_need_update');
    }
}

sub on_action_approve {
    my ($self, $obj, %opts) = @_;

    $self->_set_need_update_page($obj, %opts);
}

sub on_action_reject {
    my ($self, $obj, %opts) = @_;

    $self->_set_need_update_page($obj, %opts);
}

sub on_moderation_request_sent {
    my ($self, $data) = @_;

    my $pk_fields = $self->partner_db_table()->primary_key();
    my $pk_list = [map [@{$_}{@$pk_fields}], @$data];
    $self->partner_db_table()->edit($pk_list, {'waiting_moderation' => curdate('oformat' => 'db_time'),});

    return TRUE;
}

TRUE;
