package Application::Model::Product::AN::MobileApp;

use qbit;

use Application::Model::DAC;
use base qw(
  Application::Model::Source
  Application::Model::ValidatableMixin
  RestApi::MultistateModel
  Application::Model::DAC
  );

use PiConstants qw(
  $SUPPORT_MAIL
  $INTERNATIONAL_SUPPORT_MAIL
  $DOCS_MAIL
  $DOCS_UA_MAIL
  $SENIOR_MANAGER_FOR_DOCUMENT_PROCESSING_MAIL
  $APP_TYPES
  $BOOKMAKER_FLAG_ID
  $MOBILE_PARTNER_ROLE_ID
  $MYSQL_MIN_DATETIME
  );

use Exception::Denied;
use Exception::HTTPPI::NotFound;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;

consume qw(
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::IsMobileMediationBase
  Application::Model::Role::Has::Moderation::MobileApp
  Application::Model::Role::Has::ModerationReason::MobileApp
  Application::Model::Role::Page::Has::AppMarketFields
  );

sub accessor               {'mobile_app'}
sub db_table_name          {'mobile_app'}
sub get_product_name       {gettext('mobile_app')}
sub get_page_id_field_name { }
sub get_page_model_names   {qw(mobile_app_settings)}
sub get_opts_schema_name   {'mobile_app_opts'}

sub is_mobile_mediation_from_generated_column {TRUE}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        api_bk              => 'Application::Model::API::Yandex::BK',
        api_rmp             => 'Application::Model::API::Yandex::RMP',
        mobile_app_settings => 'Application::Model::Product::AN::MobileApp::Settings',
        agreement_checker   => 'Application::Model::AgreementChecker',
        mobile_app_owner    => 'Application::Model::Product::AN::MobileApp::MobileAppOwner',
        filters             => 'Application::Model::Filters',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        map {$self->get_description_right($_)}
          qw(
          view_field__comment
          edit_field__comment
          view_field__owners
          add_field__is_mobile_mediation
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields},
        type => {
            default    => TRUE,
            db         => TRUE,
            label      => d_gettext('Type ID'),
            need_check => {
                type => 'int_un',
                in   => [sort keys(%$APP_TYPES)],
            },
            type => 'number',
            api  => TRUE,
        },
        store_id => {
            default => TRUE,
            db      => TRUE,
            label   => d_gettext('Store ID'),
            # NOTE! если не боишься разгребать пол сотни конфликтов - раскоментируй
            #      (см https://st.yandex-team.ru/PI-10870#1511455363000 )
            #need_check => {
            #    type => 'bundle_id'
            #}
            need_check => {
                len_max => 255,
                check   => sub {
                    my ($qv, $value) = @_;
                    throw Exception::Validator::Fields gettext('Invalid Bundle ID "%s"', $value)
                      unless $qv->app->check_mobile_app_bundle_id($value, $qv->data->{type});
                },
            },
            type => 'string',
            api  => TRUE,
        },
        domain => {
            db      => TRUE,
            db_expr => 'store_id',
            label   => d_gettext('Domain'),
            type    => 'string',
            api     => TRUE,
        },
        type_name => {
            depends_on => ['type'],
            label      => d_gettext('Type'),
            get        => sub {
                $_[0]->model->get_app_type_caption($_[1]->{'type'});
            },
            type => 'string',
            api  => TRUE,
        },
        store => {
            depends_on => ['type'],
            label      => d_gettext('Store'),
            get        => sub {
                $_[0]->model->get_app_store($_[1]->{'type'});
            },
            type => 'string',
            api  => TRUE,
        },
        comment => {
            db           => TRUE,
            check_rights => 'mobile_app_view_field__comment',
            type         => 'string',
            need_check   => {
                optional => TRUE,
                len_max  => 255,
            },
            api => TRUE,
        },
        owners => {
            depends_on => ['id', 'mobile_app_owner.owner'],
            get        => sub {
                return [map {$_->{'owner'}} @{$_[0]->{'mobile_app_owner'}{$_[1]->{'id'}} // []}];
            },
            type     => 'array',
            sub_type => 'users',
            api      => TRUE,
        },
        fields_depends => {
            get => sub {
                our $FIELDS_DEPENDS;
                $FIELDS_DEPENDS //= $_[0]->model->get_fields_depends();

                return $FIELDS_DEPENDS;
            },
            type => 'complex',
            api  => TRUE,
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $filters = $self->SUPER::get_structure_model_filter();

    $filters->{'fields'} = {
        %{$filters->{'fields'}},
        type => {
            type   => 'dictionary',
            label  => d_gettext('Type'),
            values => [map {+{hash_transform($_, ['id'], {caption => 'label'})}} get_app_types()]
        },
        store_id => {type => 'text', label => d_gettext('Domain')},
        owner    => {
            type           => 'subfilter',
            model_accessor => 'mobile_app_owner',
            field          => 'id',
            fk_field       => 'app_id',
            label          => d_gettext('Owner'),
        },
        apple_store_id => {
            type       => 'json',
            value_type => 'text',
            label      => d_gettext('Apple Store ID'),
        },
        developer => {
            type       => 'json',
            value_type => 'text',
            label      => d_gettext('App developer'),
        },
    };

    return $filters;
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        (
            $self->check_short_rights('view_all')
            ? (
                [
                    {name => 'id',         label => gettext('ID')},
                    {name => 'store_id',   label => gettext('Store ID')},
                    {name => 'type',       label => gettext('Type')},
                    {name => 'multistate', label => gettext('Status')},
                    (
                        $self->check_rights('view_search_filters__login')
                        ? {name => 'owner.user.login', label => gettext('Login')}
                        : ()
                    )
                ]
              )
            : ()
        )
    ];
}

sub get_structure_multistates_graph {
    return {
        empty_name  => 'New',
        multistates => [
            [need_approve => d_pgettext('Application status', 'On moderation')],
            [rejected     => d_pgettext('Application status', 'Rejected')],
            [approved     => d_pgettext('Application status', 'Approved')],

        ],
        actions           => {set_need_approve => d_pgettext('Application action', 'Set "need approve"'),},
        right_group       => [mobile_app       => d_gettext('Right to manage mobile app')],
        right_name_prefix => 'mobile_app_',
        right_actions     => {
            add     => d_pgettext('Application action', 'Add'),
            reject  => d_pgettext('Application action', 'Reject'),
            approve => d_pgettext('Application action', 'Approve'),
            edit    => d_pgettext('Application action', 'Edit'),
        },
        multistate_actions => [
            {
                action    => 'add',
                from      => '__EMPTY__',
                set_flags => [],
            },
            {
                action      => 'set_need_approve',
                from        => 'not (approved or need_approve)',
                set_flags   => ['need_approve'],
                reset_flags => ['rejected'],
            },
            {
                action      => 'reject',
                from        => 'not rejected',
                reset_flags => ['need_approve', 'approved'],
                set_flags   => ['rejected'],
            },
            {
                action      => 'approve',
                from        => 'need_approve',
                reset_flags => ['need_approve'],
                set_flags   => ['approved'],
            },
            {
                action => 'edit',
                from   => '__EMPTY__ or need_approve or approved or rejected'
            },
        ]
    };
}

sub related_models {
    my ($self) = @_;

    return {
        mobile_app_owner => {
            accessor => 'mobile_app_owner',
            filter   => sub {
                return {app_id => array_uniq(map {$_->{'id'} // ()} @{$_[1]})};
            },
            key_fields => ['app_id'],
            value_type => 'array',
        }
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    $self->SUPER::pre_process_fields($fields, $result);
}

sub get_actions_depends {
    [qw(id multistate type)];
}

sub get_available_fields_depends {
    [qw(multistate)];
}

sub get_editable_fields_depends {
    [qw(id multistate store_url)];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(\%fields, {$accessor . '_view_field__%s' => [qw(owners comment)],});

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my %fields = map {$_ => TRUE} qw(type store_id store_url);

    $fields{'comment'} = TRUE if $self->check_short_rights('add_field__comment');

    $fields{'login'} = TRUE if $self->check_short_rights('add_other');

    return \%fields;
}

sub get_editable_fields {
    my ($self, $object) = @_;

    return {}
      unless $self->check_action($object, 'edit');

    return $self->collect_editable_fields($object);
}

sub collect_editable_fields {
    my ($self, $object) = @_;

    my %res;

    $res{'comment'} = TRUE if $self->check_short_rights('edit_field__comment');

    return \%res;
}

sub hook_fields_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_validation($opts);

    my $roles = $self->rbac->get_roles_by_user_id($self->hook_stash->get('user')->{'id'});

    throw Exception::Validation::BadArguments gettext("Username must belong to the partner")
      unless $roles->{$MOBILE_PARTNER_ROLE_ID};

    my $tmp_rights = $self->app->add_tmp_rights('mobile_app_view_all');
    my $apps       = $self->get_all(
        fields => [qw(id multistate is_mobile_mediation)],
        filter => {store_id => $opts->{'store_id'}, type => $opts->{'type'}}
    );
    undef $tmp_rights;

    if (@$apps) {
        my $app = $apps->[0];
        if ($self->check_multistate_flag($app->{'multistate'}, 'rejected')) {
            if ($self->can_action_set_need_approve($app)) {
                $self->do_action($app, 'set_need_approve');
            } else {
                throw Exception::Validation::BadArguments gettext('Application with Bundle ID "%s" is rejected',
                    $opts->{'store_id'});
            }
        }
        my $owners = $self->mobile_app_owner->get_all(
            fields => ['user_id'],
            filter => {app_id => $app->{id}, user_id => $self->hook_stash->get('user')->{'id'}}
        );
        if (@$owners) {
            if ($self->app->user_features->has_feature_simple_inapp()) {
                $self->hook_stash->set('app_already_exists', TRUE);
            } else {
                throw Exception::Validation::BadArguments gettext(
                    'You already have application with this Bundle ID "%s"', $opts->{'store_id'});
            }
        }
        $self->hook_stash->set('id', {id => $app->{'id'}});
        $self->hook_stash->set('app', $app);
    }
}

sub hook_preparing_fields_to_save {
    my ($self, $opts, @rest) = @_;

    return if $self->hook_stash->mode('add') && $self->hook_stash->check('app_already_exists');

    return $self->SUPER::hook_preparing_fields_to_save($opts);
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    return if $self->hook_stash->check('app_already_exists');

    $self->SUPER::hook_processing_after_insert($opts);

    if (    $self->hook_stash->check('app')
        and $opts->{is_mobile_mediation}
        and !$self->hook_stash->get('app')->{is_mobile_mediation})
    {
        $self->partner_db_table->edit($self->hook_stash->get('id'),
            {opts => {json_set => ['opts', \'$.is_mobile_mediation', \1]},});
    }

    unless ($self->app->user_features->has_feature_simple_inapp()) {
        $self->do_action($self->hook_stash->get('id'), 'set_need_approve')
          if $self->check_action($self->hook_stash->get('id'), 'set_need_approve');
    } else {
        if (exists $opts->{store_url} && $opts->{store_url}) {
            $self->do_action($self->hook_stash->get('id'), 'set_need_approve')
              if $self->check_action($self->hook_stash->get('id'), 'set_need_approve');
        }
    }
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    $self->SUPER::hook_processing_after_update($opts);

    if ($self->app->user_features->has_feature_simple_inapp()) {
        if (exists $opts->{store_url} && $opts->{store_url}) {
            $self->do_action($self->hook_stash->get('id'), 'set_need_approve')
              if $self->check_action($self->hook_stash->get('id'), 'set_need_approve');
        }
    }
}

sub hook_writing_to_database {
    my ($self, $opts) = @_;

    return if $self->hook_stash->check('app_already_exists');

    unless ($self->hook_stash->check('app')) {
        $self->SUPER::hook_writing_to_database($opts);
    } elsif ($self->hook_stash->check('fields_from_related_models')) {
        $self->hook_save_fields_from_related_models($opts);
    }
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    return if $self->hook_stash->check('app_already_exists');

    $self->SUPER::hook_save_fields_from_related_models($opts);

    my $tmp_rights = $self->app->add_tmp_rights('mobile_app_owner_add');
    $self->mobile_app_owner->add(
        user_id => $self->hook_stash->get('user')->{id},
        app_id  => $self->hook_stash->get('id')->{id},
    );
}

sub get_fields_to_trim {qw(store_id store_url)}

sub on_action_approve {
    my ($self, $obj, %opts) = @_;

    my $tmp_rights = $self->app->add_tmp_rights(
        qw(
          mobile_app_settings_edit
          mobile_app_settings_view_all
          do_mobile_app_settings_approve
          mobile_app_view_field__owners
          users_edit_all
          users_edit_field__has_approved
          users_edit_field__has_approved_app
          )
    );

    $obj = $self->get($obj, fields => [qw(id store_id owners is_mobile_mediation)]);

    $self->mobile_app_settings->approve_for_app($obj->{'id'});

    $self->app->queue->add(
        method_name => 'update_mobile_app_status',
        params      => {app_id => $obj->{'id'}, active => 1},
        group_id    => $obj->{'id'},
    ) if $obj->{is_mobile_mediation};

    foreach my $owner (@{$obj->{'owners'}}) {
        my $owners_data = $self->users->get_all(
            fields => [qw(multistate has_approved has_approved_app)],
            filter => {id => $owner->{'id'}, multistate => 'not blocked'},
        );

        # Если нашли НЕзаблокированного владельца то шлем ему письма
        if (@$owners_data) {
            my $owner_data = $owners_data->[0];

            if (!$owner_data->{has_approved_app}) {

                if ($owner_data->{has_approved} != 1) {
                    $self->app->mail_notification->add_when_approved(
                        {
                            user_id      => $owner->{'id'},
                            full_name    => $owner->{'full_name'},
                            has_approved => $owner_data->{'has_approved'},
                            resource     => $obj->{'store_id'},
                        }
                    );
                }
                $self->app->mail_notification->add_when_first_approved_mobile_app(
                    {user_id => $owner->{'id'}, resource => $obj->{'store_id'}});
                $self->users->do_action($owner->{'id'}, 'edit', has_approved => 1, has_approved_app => 1);
            } else {
                $self->app->mail_notification->add_when_approved_mobile_app(
                    {user_id => $owner->{'id'}, resource => $obj->{'store_id'}});
            }
        }
    }
}

sub can_action_reject {TRUE}

sub on_action_reject {
    my ($self, $obj, %opts) = @_;

    my $tmp_rights = $self->app->add_tmp_rights(
        qw(
          mobile_app_settings_edit
          mobile_app_settings_view_all
          do_mobile_app_settings_stop
          do_mobile_app_settings_reject
          )
    );

    $obj = $self->get($obj, fields => [qw(id store_id owners is_mobile_mediation)]);

    $self->mobile_app_settings->reject_for_app($obj->{'id'}, %opts);

    $self->app->queue->add(
        method_name => 'update_mobile_app_status',
        params      => {app_id => $obj->{'id'}, active => 0},
        group_id    => $obj->{'id'},
    ) if $obj->{is_mobile_mediation};

    foreach my $owner (@{$obj->{'owners'}}) {
        $self->send_letter_after_reject(
            user_id   => $owner->{'id'},
            full_name => $owner->{'full_name'},
            resource  => $obj->{'store_id'},
            reason    => $opts{'moderation_reason'},
        );
    }
}

sub _get_reject_mail_notification {return 'add_when_rejected_mobile_app'}

sub can_action_set_need_approve {TRUE}

sub on_action_set_need_approve {
    my ($self, $obj, %opts) = @_;

    my $tmp_rights = $self->app->add_tmp_rights(
        qw(
          mobile_app_settings_edit
          mobile_app_settings_view_all
          do_mobile_app_settings_set_need_approve
          )
    );

    $self->mobile_app_settings->set_need_approve_for_app($obj->{'id'});
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_assistant_mobile_app($filter);

    return $filter;
}

sub get_app_for_adding {
    my ($self, $user_id) = @_;

    throw Exception::Validation::BadArguments gettext('Expected "user_id"') unless $user_id;

    return $self->get_all(
        fields => [qw(id store_id multistate type is_approved)],
        filter => ['AND', [[multistate => '=' => 'not rejected'], [owner => 'MATCH' => [user_id => '=' => $user_id]]]]
    );
}

sub get_app_for_adding_by_login {
    my ($self, $login) = @_;

    throw Exception::Validation::BadArguments gettext('Expected "login"') unless $login;

    my $apps = $self->get_all(
        fields => [qw(id store_id type)],
        filter =>
          ['AND', [{multistate => 'not rejected'}, ['owner', 'MATCH', ['user', 'MATCH', ['login', '=', $login]]]]]
    );

    return $apps;
}

sub get_app_store {
    my ($self, $id) = @_;

    return undef unless $id && exists($APP_TYPES->{$id});

    return $APP_TYPES->{$id}{'store'};
}

sub get_app_type_caption {
    my ($self, $id) = @_;

    return undef unless $id && exists($APP_TYPES->{$id});

    return $APP_TYPES->{$id}{'caption'}->();
}

sub get_app_types {
    return map {{id => $_->{'id'}, store => $_->{'store'}, caption => $_->{'caption'}->()}}
      sort {$a->{'id'} <=> $b->{'id'}} values(%$APP_TYPES);
}

sub api_available_actions {qw(set_need_approve reject approve edit)}

sub api_can_add {TRUE}

sub api_can_edit {TRUE}

sub get_fields_depends {{}}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = ();

    if ($need_fields->{'type'}) {
        $result{'type'} = [$self->get_app_types()];
    }

    return \%result;
}

sub hook_stash_add_fields {
    my ($self) = @_;

    my %fields = $self->SUPER::hook_stash_add_fields;
    push @{$fields{once}}, 'app';
    push @{$fields{once}}, 'app_already_exists';
    return %fields;
}

sub fix_template {
    my ($self, $qv) = @_;

    my $fields = $qv->template->{'fields'};
    if ($self->app->user_features->has_feature_simple_inapp()) {
        $fields->{store_id}{optional}  = TRUE;
        $fields->{store_url}{optional} = TRUE;
    }
}

sub check_simple_inapp_feature {
    my ($self) = @_;
    return $self->app->user_features->has_feature_simple_inapp();
}

TRUE;
