package Application::Model::Product::AN::MobileApp::MobileAppOwner;

use qbit;

use Application::Model::DAC;
use base qw(
  Application::Model::DBManager::Base
  Application::Model::DAC
  );

use Exception::Denied;
use Exception::Validation::BadArguments;

sub accessor      {'mobile_app_owner'}
sub db_table_name {'mobile_app_owner'}

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    $self->register_rights($self->get_structure_rights_to_register());
    $self->model_fields($self->get_structure_model_fields());
    $self->model_filter($self->get_structure_model_filter());
}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        partner_db => 'Application::Model::PartnerDB',
        users      => 'Application::Model::Users',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            name        => $self->accessor,
            description => d_gettext('Right to manage %s', $self->accessor()),
            rights      => {
                mobile_app_owner_view_all => 'Rigth to view all mobile app owners',
                mobile_app_owner_add      => 'Rigth to add mobile app owners'
            }
        }
    ];
}

sub get_structure_model_fields {
    return {
        id      => {default => TRUE, db => TRUE, pk   => TRUE, type => 'number'},
        app_id  => {default => TRUE, db => TRUE, type => 'number'},
        user_id => {default => TRUE, db => TRUE, type => 'number'},
        owner   => {
            label      => d_gettext('Owners'),
            depends_on => [
                'user_id',            'users.id',
                'users.login',        'users.client_id',
                'users.name',         'users.lastname',
                'users.midname',      'users.full_name',
                'users.email',        'users.phone',
                'users.has_approved', 'users.has_approved_app',
                'users.has_approved_site',
            ],
            get => sub {
                $_[0]->{'users'}->{$_[1]->{'user_id'}} // {};
            },
            type => 'users',
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        db_accessor => 'partner_db',
        fields      => {
            app_id  => {type => 'number', label => d_gettext('Application ID')},
            user_id => {type => 'number', label => d_gettext('Owner ID')},
            user    => {
                type           => 'subfilter',
                model_accessor => 'users',
                field          => 'user_id',
                label          => d_gettext('User'),
            },
        }
    };
}

sub related_models {
    my ($self) = @_;

    return {
        users => {
            accessor => 'users',
            filter   => sub {
                return {id => array_uniq(map {$_->{'user_id'} // ()} @{$_[1]})};
            },
            key_fields => ['id'],
        }
    };
}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied unless $self->check_short_rights('add');

    my @bad_fields = grep {!$opts{$_}} qw(app_id user_id);
    throw Exception::Validation::BadArguments gettext('Expected fileds: %s', join(', ', @bad_fields))
      if @bad_fields;

    $self->partner_db_table()->add({map {$_ => $opts{$_}} qw(app_id user_id)});
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_tutby($filter);

    return $filter;
}

TRUE;
