package Application::Model::Product::AN::MobileApp::RTB;

use qbit;

use base qw(
  Application::Model::Block::External
  RestApi::MultistateModel
  Application::Model::ValidatableMixin
  );

use Partner::Filters
  qw(get_filters_not_rewarded_mobile_blocks get_filters_not_adaptive_banner_mobile_blocks get_filters_and);

use PiConstants qw(
  $DSP_DIRECT_ID
  $DSP_MOBILE_TYPE_ID
  $FONT_SIZES
  $INTERNAL_DSP_TAGS
  $MAX_REVENUE_STRATEGY_ID
  $MIN_CPM_STRATEGY_ID
  $SEPARATE_CPM_STRATEGY_ID
  $VIDEO_INTERSTITIAL_MOBILE_ID
  @MOBILE_DEFAULT_DSPS
  @VIDEO_BANNER_DSPS
  @VIDEO_INTERSTITIAL_DSPS
  );

use Application::Model::Product::BlockTypes::MobileAppRTB qw(
  direct_block_types
  @FIELDS
  @COLOR_FIELDS
  @BOOLEAN_FIELDS
  direct_appearance_fields
  %DIRECT_BLOCKS
  @MEDIA_BLOCKS
  %MEDIA_BLOCKS
  @DSP_BLOCKS
  %DSP_BLOCKS
  );

use Application::Model::Product::AN::MobileApp::BlockTypes::Settings qw(
  %BLOCK_TYPES
  );

use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Errors;
use Exception::Validator::Fields;

consume qw(
  Application::Model::Role::Block::AdFoxNotifier
  Application::Model::Role::Block::Has::AdfoxBlock
  Application::Model::Role::Block::Has::BlocksLimit
  Application::Model::Role::Block::Has::CloseButtonDelay
  Application::Model::Role::Block::Has::CPMCurrency
  Application::Model::Role::Block::Has::InitUniqueId
  Application::Model::Role::Block::Has::IsBidding
  Application::Model::Role::Block::Has::IsExternalMediation
  Application::Model::Role::Block::Has::IsMobileMediation
  Application::Model::Role::Block::Has::MediaBlock
  Application::Model::Role::Block::Has::MobileAppMode
  Application::Model::Role::Block::Has::ShowSlider
  Application::Model::Role::Block::Has::Tags
  Application::Model::Role::Block::Has::UnmoderatedDSPs
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Block::DesignSettings
  Application::Model::Role::Has::Categories
  Application::Model::Role::Has::CreateDate
  Application::Model::Role::Has::CustomBKOptions
  Application::Model::Role::Has::DSPS::MobileAdaptiveBannerRule
  Application::Model::Role::Has::DSPS::MobileBannerRule
  Application::Model::Role::Has::DSPS::MobileNativeRule
  Application::Model::Role::Has::DSPS::MobileRewardedRule
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::PlaceID
  Application::Model::Role::Has::Readonly
  Application::Model::Role::Has::RTB
  Application::Model::Role::Has::RTBDesignFields
  Application::Model::Role::Has::ShowVideo
  Application::Model::Role::Has::Strategies::ShiftingFields
  Application::Model::Role::JavaJsonApiProxy
  );

sub accessor      {'mobile_app_rtb'}
sub db_table_name {'mobile_app_rtb'}

sub get_campaign_model_name {
    return 'mobile_app_settings';
}

sub get_product_name {gettext('mobile_app_rtb')}

sub get_page_id_field_name {'campaign_id'}

sub public_id_prefix {'R-M-'}

sub design_fields_from_opts {TRUE}

sub get_opts_schema_name {'mobile_app_rtb_opts'}

sub get_dsp_type {[$DSP_MOBILE_TYPE_ID]}

sub is_block_table_with_multiple_models {TRUE}

sub is_mobile_mediation_from_generated_column {FALSE}

sub is_show_video_applicable {
    my ($self, $block) = @_;

    return unless defined($block->{block_type});

    # не можем проверить право на show_video, потому что оно открыто для партнёра для баннера
    # не можем удалить по праву, потому что оно докидывается ПОСЛЕ формирования ролью
    my ($applicable_for_native, $applicable_for_interstitial, $default_for_banner) = (1, 1, 0);
    if ($self->app->user_features->page_owner_has_feature_simple_inapp($block)) {
        $applicable_for_native       = $self->check_short_rights("edit_field__rich_media");
        $applicable_for_interstitial = 0;
        $default_for_banner          = 1;
    }

    my %show_video_by_block_type = (
        native          => {applicable => $applicable_for_native,       default_value => 0},
        interstitial    => {applicable => $applicable_for_interstitial, default_value => 1},
        banner          => {applicable => 1,                            default_value => $default_for_banner},
        rewarded        => {applicable => 0,                            default_value => 1},
        adaptive_banner => {applicable => 0},
    );

    return $show_video_by_block_type{$block->{block_type}};
}

# TODO: Block::Mobile ???
sub get_video_dsp_list {
    my ($self, $block) = @_;

    return () unless defined($block->{block_type});

    my %video_dsp_list_by_block_type = (
        native       => [],
        interstitial => \@VIDEO_INTERSTITIAL_DSPS,
        banner       => \@VIDEO_BANNER_DSPS,
        rewarded     => \@VIDEO_BANNER_DSPS,
    );

    return @{$video_dsp_list_by_block_type{$block->{block_type}}};
}

sub get_video_dsp_list_depends {qw(block_type)}

# TODO: [show_video => '=' => 1] filter should be added in Has::ShowVideo
# It's a common part of a filter for all implementors
sub get_video_dsp_turn_on_block_filter {
    my ($self, $dsp) = @_;

    my %block_type_by_video_dsp;
    push @{$block_type_by_video_dsp{$_}}, ('interstitial') for @VIDEO_INTERSTITIAL_DSPS;
    push @{$block_type_by_video_dsp{$_}}, ('banner', 'rewarded', 'native') for @VIDEO_BANNER_DSPS;

    if (my $block_types = $block_type_by_video_dsp{$dsp->{id}}) {
        return [AND => [[show_video => '=' => 1], [OR => [map {[block_type => '=' => $_]} @$block_types]]]];
    } else {
        return;
    }
}

sub role_show_video_get_fields_depends {[qw(show_video block_type)]}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        %{$class->SUPER::get_structure_model_accessors()},
        mobile_app_settings => 'Application::Model::Product::AN::MobileApp::Settings',
        tns_dict_brand      => 'Application::Model::TNSDict::Brand',
        dsp                 => 'Application::Model::DSP',
        kv_store            => 'QBit::Application::Model::KvStore',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        map {$self->get_description_right($_)}
          qw(
          edit_field__comment
          edit_field__is_custom_format_direct
          view_field__comment
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        campaign_id => {
            default    => TRUE,
            db         => TRUE,
            pk         => TRUE,
            need_check => {type => 'int_un',},
            type       => 'string',
        },
        context_page_id => {
            default => TRUE,
            db      => TRUE,
            db_expr => 'campaign_id',
            label   => d_gettext('Application ID'),
            hint    => d_gettext('The unique application identifier (the same in BK)'),
            api     => 1,
            type    => 'number',
        },
        block_type => {
            db         => TRUE,
            label      => d_gettext('Mobile block type'),
            api        => 1,
            type       => 'string',
            need_check => {
                len_max => 18,     # as in database
                check   => sub {
                    my ($qv, $block_type, $template) = @_;

                    my $apps = $qv->app->mobile_app_settings->get_all(
                        fields => ['block_types'],
                        filter => {context_page_id => $qv->data->{'page_id'}}
                    );

                    $qv->_add_error($template, gettext('Mobile app with page id %s not found', $qv->data->{'page_id'}),
                        [])
                      unless $apps && @$apps;

                    my $block_types = $apps->[0]{'block_types'};

                    throw Exception::Validator::Fields gettext('Invalid block_type value')
                      unless grep {$_->{'id'} eq $block_type} @{$block_types};
                  }
            },
        },
        block_type_label => {
            depends_on => ['block_type'],
            get        => sub {
                return $Application::Model::Product::AN::MobileApp::Settings::BLOCK_TYPES{$_[1]->{'block_type'}}
                  {'label'}();
            },
            api  => 1,
            type => 'string',
        },
        is_custom_format_direct => {
            from_opts => 'from_hash',
            get       => sub {
                $_[1]->{'is_custom_format_direct'} // 0;
            },
            label      => d_gettext('Custom Direct format enabled'),
            need_check => {type => 'boolean', optional => TRUE},
        },
        domain => {
            depends_on => ['store_id'],
            get        => sub {
                $_[1]->{store_id};
            },
            api  => 1,
            type => 'string',
        },
        border_type => {
            from_opts  => 'from_hash',
            type       => 'string',
            need_check => {in => [qw(none block ad collapse)],},
            api        => 1,
        },
        title_font_size => {from_opts => 'from_hash', type => 'string', need_check => {in => [qw(1 2 3)],}, api => 1,},
        font_family     => {
            from_opts  => 'from_hash',
            type       => 'string',
            need_check => {in => ['', 'arial', 'courier new', 'tahoma', 'times new roman', 'verdana',],},
            api        => 1,
        },
        font_size     => {from_opts => 'from_hash', type => 'string', need_check => {in   => $FONT_SIZES,}, api => 1,},
        site_bg_color => {from_opts => 'from_hash', type => 'string', need_check => {type => 'color'},      api => 1,},
        bg_color =>
          {from_opts => 'from_hash', type => 'string', need_check => {type => 'color', optional => TRUE,}, api => 1,},
        border_color => {from_opts => 'from_hash', type => 'string', need_check => {type => 'color',}, api => 1,},
        header_bg_color =>
          {from_opts => 'from_hash', type => 'string', need_check => {type => 'color', optional => TRUE,}, api => 1,},
        title_color => {from_opts => 'from_hash', type => 'string', need_check => {type => 'color'}, api => 1,},
        text_color  => {from_opts => 'from_hash', type => 'string', need_check => {type => 'color'}, api => 1,},
        url_background_color =>
          {from_opts => 'from_hash', type => 'string', need_check => {type => 'color'}, api => 1,},
        url_color       => {from_opts => 'from_hash', type => 'string',  need_check => {type => 'color'},    api => 1,},
        hover_color     => {from_opts => 'from_hash', type => 'string',  need_check => {type => 'color'},    api => 1,},
        sitelinks_color => {from_opts => 'from_hash', type => 'string',  need_check => {type => 'color'},    api => 1,},
        border_radius   => {from_opts => 'from_hash', type => 'boolean', need_check => {type => 'boolean',}, api => 1,},
        no_sitelinks    => {
            from_opts  => 'from_hash',
            type       => 'boolean',
            need_check => {type => 'boolean'},
            api        => 1,
        },
        favicon => {
            from_opts  => 'from_hash',
            type       => 'boolean',
            need_check => {type => 'boolean'},
            api        => 1,
        },
        links_underline => {
            from_opts  => 'from_hash',
            type       => 'boolean',
            need_check => {type => 'boolean'},
            api        => 1,
        },
        store_id => {
            depends_on => ['page_id', 'pages.store_id'],
            label      => d_gettext('store_id'),
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'store_id'};
            },
            api  => 1,
            type => 'string',
        },
        application_id => {
            depends_on => ['page_id', 'pages.application_id'],
            label      => d_gettext('App ID'),
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'application_id'};
            },
            api  => 1,
            type => 'number',
        },
        page_public_id => {
            depends_on => ['page_id', 'pages.public_id'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'public_id'};
            },
        },
        application => {
            depends_on => ['page_id', 'pages.id', 'pages.multistate', 'pages.owner_id'],
            get        => sub {
                $_[0]->{'pages'}->{$_[1]->{'page_id'}};
            },
        },
        application_caption => {
            depends_on => ['page_id', 'pages.caption'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'caption'};
            },
            api  => 1,
            type => 'string',
        },
        application_platform => {
            depends_on => ['page_id', 'pages.type_name'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'type_name'};
            },
            api  => 1,
            type => 'string',
        },
        is_application_deleted => {
            depends_on => ['is_page_deleted'],
            get        => sub {
                $_[1]->{'is_page_deleted'};
            },
            api  => 1,
            type => 'boolean',
        },
        bk_state_name => {
            depends_on => ['page_id', 'pages.bk_state_name'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'bk_state_name'};
            },
        },
        currency_type => {
            from_opts  => 'from_hash',
            need_check => {
                len_min => 1,
                len_max => 64,
            },
            api  => TRUE,
            type => 'string',
        },
        currency_value => {
            from_opts  => 'from_hash',
            need_check => {type => 'int_un',},
            api        => TRUE,
            type       => 'number',
        },
        callback => {
            from_opts  => 'from_hash',
            need_check => {
                optional => TRUE,
                len_min  => 1,
                len_max  => 255,
            },
            api  => TRUE,
            type => 'string',
        },
        sign => {
            from_opts  => 'from_hash',
            need_check => {
                optional => TRUE,
                len_min  => 1,
                len_max  => 64,
            },
            api  => TRUE,
            type => 'string',
        },
        #redefine
        blind => {
            db      => TRUE,
            db_expr => \0,
        },
        rich_media => {
            from_opts  => 'from_hash',
            need_check => {
                optional => TRUE,
                type     => 'boolean',
            },
            api  => 1,
            type => 'boolean',
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $parent_filter = $self->SUPER::get_structure_model_filter();
    return {
        db_accessor => $parent_filter->{'db_accessor'},
        fields      => {
            %{$parent_filter->{'fields'}},
            campaign_id     => {type => 'number', label => d_gettext('Application ID')},
            context_page_id => {type => 'alias',  path  => [qw(campaign_id)], label => d_gettext('Application ID')},
            page_id => {
                type      => 'number',
                label     => d_gettext('Page ID'),
                db_filter => sub {
                    return ['campaign_id', $_[1]->[1], \$_[1]->[2]];
                  }
            },
            page => {
                type           => 'subfilter',
                model_accessor => 'mobile_app_settings',
                field          => 'campaign_id',
                fk_field       => 'context_page_id',
                label          => d_gettext('application'),
            },
            block_type => {
                type   => 'dictionary',
                label  => d_gettext('Ad format'),
                values => sub {
                    my ($self) = @_;
                    my $has_feature_simple_inapp = $self->app->user_features->has_feature_simple_inapp();
                    my %block_types = %Application::Model::Product::AN::MobileApp::BlockTypes::Settings::BLOCK_TYPES;
                    my @BLOCK_TYPES = keys %block_types;
                    @BLOCK_TYPES = grep {$_ ne 'adaptive_banner'} @BLOCK_TYPES if $has_feature_simple_inapp;
                    my $block_type_values = [
                        map {+{id => $_, label => $block_types{$_}{'label'}()}}
                        sort @BLOCK_TYPES
                    ];
                    return $block_type_values;
                  }
            },
            application    => {type => 'alias', path => [qw(page)]},
            login          => {type => 'alias', path => [qw(page owner login)]},
            client_id      => {type => 'alias', path => [qw(page owner client_id)]},
            is_tutby       => {type => 'alias', path => [qw(page owner is_tutby)]},
            application_id => {type => 'alias', path => [qw(page application_id)]},
            page_caption   => {type => 'alias', path => [qw(page caption)]},
            platform       => {type => 'alias', path => [qw(page application type)]},
            store_id       => {type => 'alias', path => [qw(page application store_id)]},
        }
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        (
            $self->check_rights('view_search_filters__login')
            ? {name => 'login', label => gettext('Login'), login_filter => 'mobile_an_partner=1'}
            : ()
        ),
        (
            $self->check_rights('view_search_filters__client_id')
            ? {name => 'application.owner.client_id', label => gettext('Client ID')}
            : ()
        ),
        (
            $self->check_rights('view_search_filters__user_type') ? {name => 'user_type', label => gettext('User type')}
            : ()
        ),
        {name => 'page_id',                          label => gettext('Page ID')},
        {name => 'application.application.store_id', label => gettext('Store ID')},
        {name => 'application.application.type',     label => gettext('Platform')},
        (
            $self->check_rights('view_search_filters__app_id')
            ? {name => 'application.application_id', label => gettext('App ID')}
            : ()
        ),
        {name => 'block_type', label => gettext('Block type')},
        {name => 'id',         label => gettext('Block ID')},
        {name => 'caption',    label => gettext('Block caption')},
        {name => 'multistate', label => gettext('Status')},
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_pgettext('Block multistate', 'New'),
        multistates => [
            [deleted           => d_pgettext('Block multistate', 'Archived')],
            [working           => d_pgettext('Block multistate', 'Working')],
            [check_statistics  => d_pgettext('Block multistate', 'Check statistics')],
            [deleted_with_page => d_pgettext('Block multistate', 'Archived with page'), private => TRUE],
            [need_update       => d_pgettext('Block multistate', 'Need update')],
            [updating          => d_pgettext('Block multistate', 'Updating')],
        ],
        actions => {
            edit                   => d_pgettext('Block action', 'Edit'),
            delete                 => d_pgettext('Block action', 'Archive'),
            delete_with_page       => d_pgettext('Block action', 'Archive with page'),
            restore                => d_pgettext('Block action', 'Restore'),
            restore_with_page      => d_pgettext('Block action', 'Restore with page'),
            start                  => d_pgettext('Block action', 'Start'),
            stop                   => d_pgettext('Block action', 'Stop'),
            set_check_statistics   => d_pgettext('Block action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Block action', 'Reset "check_statistics"'),
            duplicate              => d_pgettext('Block action', 'Duplicate'),
        },
        right_name_prefix => $self->accessor . '_',
        right_actions     => {
            add => d_pgettext('Block action', 'Add'),
            set_need_update =>
              {label => d_pgettext('Block action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Block action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Block action', 'Stop update'),  dont_write_to_action_log => TRUE},
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action    => 'delete_with_page',
                from      => 'not deleted',
                set_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page',
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action    => 'start',
                from      => 'not working and not deleted',
                set_flags => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
            },
            {
                action    => 'set_check_statistics',
                from      => 'not deleted',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action => 'duplicate',
                from   => 'not (deleted or deleted_with_page)',
            },
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('dsp_blocks')) {
        $fields->{'dsp_blocks'} = {};

        my $page_ids = $fields->stash->{'page_ids'} //= array_uniq(map {$_->{'page_id'}} @$result);

        foreach (
            @{
                $self->partner_db->media_sizes->get_all(
                    fields => [qw(page_id block_id type)],
                    filter => {page_id => $page_ids},
                )
            }
          )
        {
            push(@{$fields->{'dsp_blocks'}{$_->{'page_id'}, $_->{'block_id'}}}, $_->{'type'});
        }
    }
}

sub get_available_fields_depends {
    [qw(multistate block_type)];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);
    delete($fields{'campaign_id'});

    my $accessor      = $self->accessor();
    my $page_accessor = $self->get_campaign_model_name();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            # <Право или маска под право>            <Поля которые оно закрывает>
            $accessor . '_view_field__%s' => [qw( comment is_tutby)],

            # campaign's right here!
            $page_accessor . '_view_field__owner' => [qw(owner_client_id)],
            $page_accessor . '_view_field__%s'    => 'login',
        }
    );

    my $block_type = $obj->{'block_type'};

    if ($block_type eq 'rewarded') {
        delete @fields{qw(callback sign)};
        delete @fields{qw(media_block direct_block)};
    } else {
        delete @fields{qw(currency_type currency_value callback sign)};
    }

    if ($block_type eq 'native') {
        delete @fields{qw(media_block direct_block)};
        delete $fields{dsp_blocks} unless $self->check_short_rights("view_field__dsp_blocks");
    } elsif ($block_type eq 'interstitial') {
        delete @fields{qw(media_block direct_block)};
    }

    my $has_feature_simple_inapp = $self->app->user_features->has_feature_simple_inapp();
    if ($has_feature_simple_inapp) {
        delete $fields{$_} for (grep {$_ ne 'limit'} (direct_block_fields(), direct_appearance_fields()));
    }

    delete @fields{qw(rich_media)}
      unless ($self->app->user_features->page_owner_has_feature_simple_inapp($obj)
        && $block_type eq 'native'
        && $self->check_short_rights("edit_field__rich_media"));

    return \%fields;
}

sub get_add_fields {
    my ($self, $obj) = @_;

    if (defined $obj->{'block_type'} && !exists $BLOCK_TYPES{$obj->{'block_type'}}) {
        throw Exception::IncorrectParams gettext('Invalid block_type value');
    }

    my $fields = $self->_get_common_add_edit_fields($obj);

    $fields->{'context_page_id'} = TRUE;
    $fields->{'block_type'}      = TRUE;
    $fields->{'page_id'}         = TRUE;

    if (defined($obj->{'block_type'}) && $obj->{'block_type'} eq 'interstitial') {
        delete($fields->{'dsp_blocks'});
    }

    $self->set_show_video_add_fields(
        {
            block_type => $obj->{block_type},
            page_id    => $obj->{page_id}
        },
        $fields
    );

    return $fields;
}

sub get_editable_fields {
    my ($self, $data) = @_;
    my $res = $self->SUPER::get_editable_fields($data);
    if ($data->{is_bidding}) {
        delete $res->{strategy};
    }
    return $res;
}

sub _get_common_add_edit_fields {
    my ($self, $obj) = @_;

    my $fields = $self->get_fields_by_right(
        no_right_fields => [
            qw(
              caption
              currency_type
              currency_value
              mincpm
              )
        ],
        right_fields => {edit => ['comment', 'is_custom_format_direct']}
    );

    $self->get_fields_by_right(
        res_fields      => $fields,
        no_right_fields => [
            $self->app->user_features->page_owner_has_feature_simple_inapp($obj)
            ? qw(limit)
            : (
                qw(
                  adaptive_width
                  adaptive_height
                  dsp_blocks
                  ),
                direct_block_fields(),
                direct_appearance_fields()
              )
        ],
    );

    delete($fields->{'caption'}) unless $self->check_right_for_do_action_mobile_mediation($obj);

    my $block_type = $obj->{'block_type'} // '';
    if (defined($block_type) && exists $BLOCK_TYPES{$block_type}) {
        delete(@$fields{@{$BLOCK_TYPES{$block_type}->{'skip_fields'}}});
        delete(@$fields{@{$BLOCK_TYPES{$block_type}->{'read_only_fields'}}});

        if ($block_type eq 'native') {
            delete $fields->{dsp_blocks} unless $self->check_short_rights("edit_field__dsp_blocks");
            $fields->{'rich_media'} = TRUE;
        }
    }

    if (  !$self->app->user_features->page_owner_has_feature_simple_inapp($obj)
        || $block_type ne 'native'
        || !$self->check_short_rights("edit_field__rich_media"))
    {
        delete $fields->{'rich_media'};
    }

    return $fields;
}

sub can_action_duplicate {
    my ($self, $obj, %opts) = @_;

    return FALSE unless $self->check_rights($self->get_rights_by_actions('add'));

    my $object = $self->_get_object_fields($obj, [qw(is_protected)]);

    return !$object->{'is_protected'} || $self->check_rights('edit_protected_pages') ? TRUE : FALSE;
}

sub can_action_restore {
    my ($self, $block) = @_;

    my $fields = $self->_get_object_fields($block, [qw(page)]);

    return $self->mobile_app_settings->check_action($fields->{'page'}, 'restore_block');
}

sub on_action_restore {
    my ($self, $block, %opts) = @_;

    my $fields = $self->_get_object_fields($block, [qw(page owner_client_id direct_block media_block dsp_blocks)]);

    # Since old blocks were archived the agreement might have been changed
    #
    if ($self->mobile_app_settings->check_multistate_flag($fields->{'page'}{'multistate'}, 'working')) {
        my $can_start = $self->agreement_checker->has_agreement_for_any_product_for_today(
            client_id => $fields->{'owner_client_id'},
            products  => ['mobile_app_rtb'],
        );

        unless ($can_start) {
            throw Exception::Denied $self->get_agreement_error_message();
        }
    }

    $self->maybe_do_action($block, 'start');

    $self->update_in_bk($block) unless $opts{'do_not_update_in_bk'};

    return TRUE;
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_campaign($filter);

    return $filter;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = map {$_ => TRUE} qw(
      adaptive_height
      adaptive_width
      block_type
      callback
      caption
      currency_type
      currency_value
      dsp_blocks
      dsps
      rich_media
      show_video
      sign
      ), direct_block_fields(), direct_appearance_fields();

    return \%fields_need_update;
}

sub update_in_bk {
    my ($self, $block) = @_;

    my $page_id = ref($block) eq 'ARRAY' ? $block->[0] : $block->{$self->get_page_id_field_name};

    my $page = $self->page->get_all(
        fields => $self->page->get_depends_for_field('actions'),
        filter => {page_id => $page_id}
    )->[0];

    my $tmp_rights = $self->app->add_tmp_rights($self->page->get_rights_by_actions('set_need_update'));
    return FALSE unless $self->page->check_action($page, 'set_need_update');

    return $self->SUPER::update_in_bk($block);
}

# API

sub api_available_actions {
    my ($self) = @_;
    my @actions = qw(duplicate edit delete restore);
    push @actions, qw(start stop) unless $self->app->user_features->has_feature_simple_inapp();
    return @actions;
}

sub api_can_edit {TRUE}

sub api_can_add {TRUE}

sub boolean_fields {
    return {map {$_ => TRUE} @BOOLEAN_FIELDS};
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my %res = %{$self->_get_common_add_edit_fields($data)};

    if ($data->{'is_custom_format_direct'}
        && !$self->check_short_rights('edit_field__is_custom_format_direct'))
    {
        delete($res{'direct_block'});
        delete($res{'limit'});
    }

    my $block_type = $data->{'block_type'};

    if (defined($block_type) && $block_type eq 'native') {
        delete(@res{@{$DIRECT_BLOCKS{$block_type}->{'skip_fields'}}});
    }

    return \%res;
}

sub direct_block_fields {
    return qw(
      direct_block media_block limit
      );
}

sub direct_blocks {
    return \%DIRECT_BLOCKS;
}

sub dsp_blocks {
    return \%DSP_BLOCKS;
}

sub fix_template {
    my ($self, $qv) = @_;

    $self->SUPER::fix_template($qv);

    my $template   = $qv->template;
    my $block_type = $qv->data->{'block_type'};

    my $fields = $template->{'fields'};

    if ($qv->data->{'is_custom_format_direct'}) {
        $fields->{'callouts'} = {eq => undef};
    } else {
        my $direct_block = (defined $block_type && $block_type eq 'native') ? 'native' : $qv->data->{'direct_block'};

        if (   defined($direct_block)
            && $DIRECT_BLOCKS{$direct_block}
            && $DIRECT_BLOCKS{$direct_block}{'skip_fields'})
        {
            foreach (@{$DIRECT_BLOCKS{$direct_block}{'skip_fields'}}) {
                $fields->{$_} = {eq => undef};
            }
        }
    }

    if (exists $BLOCK_TYPES{$block_type}) {
        foreach (@{$BLOCK_TYPES{$block_type}->{'skip_fields'}}) {
            $fields->{$_} = {eq => undef};
        }

        $BLOCK_TYPES{$block_type}->{'fix_template'}->($qv);
    }

    if ($qv->data->{'border_type'} && $qv->data->{'border_type'} eq 'none') {
        foreach (
            qw(
            border_color
            border_radius
            )
          )
        {
            $fields->{$_} = {eq => undef};
        }
    }

    if ($qv->data->{'no_sitelinks'}) {
        $fields->{'sitelinks_color'} = {eq => undef};
    }

    $qv->template($template);
}

# API

sub media_blocks {
    return \%MEDIA_BLOCKS;
}

sub get_actions_depends {
    [
        qw(
          campaign_id
          cur_user_is_read_assistant
          id
          is_custom_format_direct
          is_protected
          multistate
          page
          )
    ];
}

sub get_all_campaigns_for_adding {
    my ($self, %opts) = @_;

    return $self->SUPER::get_all_campaigns_for_adding(
        %opts,
        (
            fields => [
                qw(application_id store_id type_name caption context_page_id app_is_approved owner_id),
                @{$opts{'fields'} // []}
            ],
            exclude_multistate => $opts{exclude_multistate} // [qw(deleted protected)],
        )
    );
}

sub get_bk_block_data {
    my ($self, $block) = @_;

    my $bk_data = $self->get_bk_mobile_rtb_settings($block);

    if ($block->{'block_type'} eq 'rewarded') {
        push(@{$bk_data->{'PageImpOptions'}{'Enable'}}, 'rewarded');

        $bk_data->{'CustomBlockData'} = {
            Reward => {
                CurrencyType  => $block->{'currency_type'},
                CurrencyValue => $block->{'currency_value'} + 0,
                Callback      => $block->{'callback'},
                Sign          => $block->{'sign'},
            },
        };
    } else {
        push(@{$bk_data->{'PageImpOptions'}{'Disable'}}, 'rewarded');
    }

    return $bk_data;
}

sub get_bk_block_data_key {'rtb_blocks'}

sub get_bk_common_data {
    return ();
}

sub get_dsp_rules {
    my ($self) = @_;

    return [
        @{$self->SUPER::get_dsp_rules()},             $self->get_show_video_dsp_rule(),
        $self->get_mobile_banner_dsp_rule(),          $self->get_mobile_native_dsp_rule(),
        $self->get_mobile_adaptive_banner_dsp_rule(), $self->get_mobile_rewarded_dsp_rule(),
        $self->get_special_show_video_dsp_rule(),
    ];
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    #TODO: удалить когда фронт перейдет на использование page_id,
    # во всех сабах так же нужно поддержать page_id
    $opts->{attributes}{'context_page_id'} //= $opts->{attributes}{'page_id'};

    my $result = $self->_rtb_make_fields_defaults($opts, $need_fields);

    if ($need_fields->{'direct_block'}) {
        $result->{'direct_block'} = $self->get_types_direct_blocks($opts->{attributes});
    }

    if ($need_fields->{'block_type'}) {
        my $page = $self->mobile_app_settings->get_all(
            fields => ['block_types'],
            filter => {page_id => $opts->{attributes}{'page_id'}}
          )->[0]
          or throw Exception::Validation::BadArguments gettext('Mobile app with page id %s not found',
            $opts->{attributes}{'page_id'});

        $result->{'block_type'} = [sort {$a->{'id'} cmp $b->{'id'}} @{$page->{'block_types'}}];
    }

    if ($need_fields->{'rich_media'} && $self->check_short_rights("edit_field__rich_media")) {
        my $block_type = $opts->{attributes}{'block_type'} // '';
        if ($block_type eq 'native') {
            $result->{'rich_media'} = [0, 1];
            $result->{'default_values'} //= {};
            $result->{'default_values'}{'rich_media'} = 0;
        }
    }

    return $result;
}

sub get_fields_depends {
    return {
        #если поменялось поле из ключа, то
        #нужно перезапросить поля из значения
        depends => {
            page_id    => [qw(dsps block_type)],
            show_video => [qw(dsps)],
            block_type => [qw(dsps direct_block media_block show_video rich_media)],
            id         => [qw(strategy)],
        },
        #для поля из ключа обязятельно нужно передать поля из значения
        required => {
            dsps         => [qw(page_id block_type show_video)],
            block_type   => [qw(page_id)],
            direct_block => [qw(block_type)],
            strategy     => [qw(id block_type)],
            media_block  => [qw(block_type)],
            show_video   => [qw(block_type)],
            rich_media   => [qw(block_type)],
        },
    };
}

sub get_editable_fields_depends {
    [
        qw(
          block_type
          campaign_id
          cur_user_is_read_assistant
          id
          is_bidding
          is_custom_format_direct
          is_protected
          multistate
          page_id
          page
          place_id
          )
    ];
}

sub get_filter_applicable_to_by_strategy {
    my ($self, $strategy_id) = @_;

    if ($SEPARATE_CPM_STRATEGY_ID eq $strategy_id) {
        return (
            applicable_to => get_filters_and(
                [get_filters_not_rewarded_mobile_blocks(), get_filters_not_adaptive_banner_mobile_blocks()]
            ),
        );
    } else {
        return ();
    }
}

sub get_ad_type_list {
    return [qw(media text)];
}

sub get_types_direct_blocks {
    my ($self, $opts) = @_;

    my $block_type = $opts->{block_type};

    my $direct_blocks = $self->direct_block_types();
    my @result        = ();
    my $block_types;

    foreach (@$direct_blocks) {
        $_->{'short_caption'} = $_->{'short_caption'}();
        $_->{'caption'}       = $_->{'caption'}();
        $block_types          = $_->{'block_types'};

        $_ = {hash_transform($_, [qw(id short_caption caption min_banners max_banners skip_fields banners)])};

        push(@result, $_) if in_array($block_type, $block_types);
    }

    return \@result;
}

sub get_types_dsp_blocks {
    my ($self, $opts) = @_;

    my $block_type = $opts->{block_type};

    my $dsp_blocks = clone(\@DSP_BLOCKS);
    my @result     = ();

    foreach (@$dsp_blocks) {
        $_->{'short_caption'} = $_->{'short_caption'}();

        push(@result, $_) if !$block_type || in_array($block_type, $_->{'block_types'});
    }

    return \@result;
}

sub get_types_media_blocks {
    my ($self, $opts) = @_;

    my $block_type = $opts->{block_type};

    my $media_blocks = clone(\@MEDIA_BLOCKS);
    my @result       = ();

    foreach (@$media_blocks) {
        $_->{'short_caption'} = $_->{'short_caption'}();

        push(@result, $_) if in_array($block_type, $_->{'block_types'});
    }

    return \@result;
}

sub set_block_default_values {
    my ($self, $opts, $old_settings) = @_;

    my $new_settings;
    if (defined $old_settings) {
        $new_settings = {%$old_settings, %$opts};

        my $rules = $self->get_dsp_rule_set();
        $rules->edit_dsp_list($old_settings, $opts);
        $new_settings->{dsps} = $opts->{dsps} if exists($opts->{dsps});
    } else {
        $new_settings = $opts;
    }

    my $block_type = $new_settings->{'block_type'};
    if (exists $BLOCK_TYPES{$block_type}) {
        foreach (@{$BLOCK_TYPES{$block_type}->{'skip_fields'}}) {
            $new_settings->{$_} = undef;
            $opts->{$_}         = undef;
        }
    }

    return $new_settings;
}

sub _rtb_direct_block_has_limit {
    my ($self, $direct_block) = @_;

    return in_array($direct_block, [qw(adaptive)]);
}

sub _set_block_default_values {
    my ($self, $opts) = @_;

    my $block_type = $opts->{'block_type'};
    if (exists $BLOCK_TYPES{$block_type}) {
        foreach (keys(%{$BLOCK_TYPES{$block_type}->{'defaults'}})) {
            $opts->{$_} //= $BLOCK_TYPES{$block_type}->{'defaults'}{$_};
        }
    }

    if ($self->app->user_features->page_owner_has_feature_simple_inapp($opts)) {
        my $defaults = {
            common => {
                caption  => gettext('Mobile RTB block'),
                strategy => '1',
            },
            adaptive_banner => {
                adaptive_height      => '',
                adaptive_width       => '',
                border_color         => 'DDDCDA',
                border_radius        => 0,
                border_type          => 'block',
                callouts             => undef,
                direct_block         => 'adaptive0418',
                dsp_blocks           => ['adaptive0418'],
                favicon              => 1,
                font_family          => undef,
                font_size            => undef,
                horizontal_align     => 1,
                hover_color          => 'DD0000',
                limit                => undef,
                links_underline      => 1,
                media_block          => 'adaptive0418',
                mincpm               => undef,
                no_sitelinks         => 0,
                site_bg_color        => 'FFFFFF',
                sitelinks_color      => '0000CC',
                text_color           => '000000',
                title_color          => '0000CC',
                title_font_size      => undef,
                url_background_color => undef,
                url_color            => '006600',
            },
            banner => {
                adaptive_height      => '',
                adaptive_width       => '',
                border_color         => 'DDDCDA',
                border_radius        => 0,
                border_type          => 'block',
                callouts             => undef,
                direct_block         => '300x250',
                dsp_blocks           => ['300x250'],
                favicon              => 1,
                font_family          => undef,
                font_size            => undef,
                horizontal_align     => 1,
                hover_color          => 'DD0000',
                limit                => undef,
                links_underline      => 1,
                media_block          => '300x250',
                no_sitelinks         => 0,
                site_bg_color        => 'FFFFFF',
                sitelinks_color      => '0000CC',
                text_active          => undef,
                text_blocked         => undef,
                text_color           => '000000',
                text_cpm             => undef,
                title_color          => '0000CC',
                title_font_size      => undef,
                url_color            => '006600',
                url_background_color => undef,
            },
            interstitial => {
                border_color     => 'DDDCDA',
                hover_color      => 'DD0000',
                horizontal_align => 1,
                links_underline  => 1,
                media_active     => undef,
                media_blocked    => undef,
                media_cpm        => undef,
                mincpm           => undef,
                show_video       => 1,
                site_bg_color    => 'FFFFFF',
                text_active      => undef,
                text_blocked     => undef,
                text_color       => '000000',
                title_color      => '0000CC',
                url_color        => '006600',
            },
            native => {
                bg_color             => undef,
                border_color         => undef,
                border_radius        => undef,
                border_type          => undef,
                font_family          => undef,
                font_size            => undef,
                header_bg_color      => undef,
                horizontal_align     => undef,
                hover_color          => undef,
                links_underline      => undef,
                media_active         => undef,
                media_blocked        => undef,
                media_cpm            => undef,
                mincpm               => undef,
                rich_media           => 0,
                show_video           => 0,
                site_bg_color        => undef,
                sitelinks_color      => undef,
                text_active          => undef,
                text_blocked         => undef,
                text_color           => undef,
                text_cpm             => undef,
                title_color          => undef,
                title_font_size      => undef,
                url_background_color => undef,
                url_color            => undef,
            },
            rewarded => {
                border_color     => 'DDDCDA',
                currency_type    => 'Reward',
                currency_value   => '1',
                horizontal_align => 1,
                hover_color      => 'DD0000',
                links_underline  => 1,
                mincpm           => undef,
                show_video       => 1,
                site_bg_color    => 'FFFFFF',
                text_color       => '000000',
                title_color      => '0000CC',
                url_color        => '006600',
            }
        };
        for ('common', $block_type) {
            for my $key (keys %{$defaults->{$_}}) {
                $opts->{$key} //= $defaults->{$_}{$key};
            }
        }
    }
}

sub hook_owner_processing { }

sub hook_fields_processing_before_validation {
    my ($self) = @_;

    my $opts     = $self->hook_stash->get('opts');
    my $current  = $self->hook_stash->get('current');
    my $settings = $self->hook_stash->get('settings');

    $opts->{'page_id'} //= $settings->{'campaign_id'} // $settings->{'context_page_id'};

    $self->SUPER::hook_fields_processing_before_validation($opts);

    if ($self->hook_stash->mode('add')) {
        unless ($self->check_short_rights('edit_field__show_video') && defined($opts->{show_video})) {
            my $block_type = $opts->{block_type};
            if ($block_type eq 'native') {
                $opts->{'show_video'} = 0;
            } elsif ($block_type eq 'rewarded') {
                $opts->{'show_video'} = 1;
            } elsif ($block_type eq 'interstitial') {
                $opts->{'show_video'} //= 1;
            } elsif ($block_type eq 'banner') {
                $opts->{'show_video'} //= 1;
            }
        }
    }

    my $block_type = $current->{'block_type'};
    if (exists $BLOCK_TYPES{$block_type}) {
        foreach (@{$BLOCK_TYPES{$block_type}->{'skip_fields'}}) {
            $opts->{$_} = undef;
        }
    }

    if ($self->hook_stash->mode('add')) {
        $self->_set_block_default_values($opts);
    }
    if ($opts->{is_bidding} || (!exists($opts->{is_bidding}) && $current->{is_bidding})) {
        $opts->{strategy} = $MAX_REVENUE_STRATEGY_ID;
        $opts->{mincpm}   = undef;
    }
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $opts->{'campaign_id'} //= $opts->{'page_id'};
    delete($opts->{'page_id'});
    delete($opts->{'context_page_id'});

    $self->partner_db->transaction(
        sub {
            $opts->{id} = $self->mobile_app_settings->get_next_block_id($opts->{'campaign_id'});
        }
    );
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    my $id      = $self->hook_stash->get('id');
    my $related = $self->hook_stash->get('fields_from_related_models');
    my $mode    = $self->hook_stash->mode('add') ? 'add' : 'edit';

    if (my $dsp_blocks = $related->{'dsp_blocks'}) {
        $self->partner_db->media_sizes->delete(
            $self->partner_db->filter(
                {
                    page_id  => $id->{'campaign_id'},
                    block_id => $id->{'id'},
                }
            )
        ) if $mode eq 'edit';

        $self->partner_db->media_sizes->add_multi(
            [
                map +{
                    page_id  => $id->{'campaign_id'},
                    block_id => $id->{'id'},
                    type     => $_,
                },
                @$dsp_blocks,
            ]
        );
    }
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    my $id = $self->hook_stash->get('id');

    $self->maybe_do_action($id, 'start');

    $self->update_in_bk($id);
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    my $id = $self->hook_stash->get('id');

    $self->update_in_bk($id) if $self->need_update_in_bk($self->hook_stash->get('fields'));
}

sub get_dsp_working_on_all_pages_multistate_name {
    'working_on_all_platforms'

}

sub get_strategies_available {
    my ($self, %opts) = @_;
    my @strategies = ($MAX_REVENUE_STRATEGY_ID, $MIN_CPM_STRATEGY_ID);
    if (!!exists $opts{page_id}
        || (exists $opts{page_id} && !$self->app->user_features->page_owner_has_feature_simple_inapp(\%opts)))
    {
        push @strategies, $SEPARATE_CPM_STRATEGY_ID;
    }
    return {map {$_ => 1} @strategies};
}

TRUE;
