package Application::Model::Product::AN::MobileApp::Settings;

use qbit;

use base qw(
  Application::Model::Page::Mobile
  Application::Model::Page::MIXIN::External
  RestApi::MultistateModel
  );

use PiConstants qw(
  $CONTEXT_TARGET_TYPE
  :ROLES
  @BK_LANGUAGES_BY_DEFAULT
  );

use Utils::PhoneBK;
use Utils::DomainBK;

use Application::Model::Product::AN::MobileApp::BlockTypes::Settings qw(
  @BLOCK_TYPES
  %BLOCK_TYPES
  );

use Exception::BK::Protected;
use Exception::Denied;
use Exception::Validation::BadArguments;
use Utils::Logger qw(WARN);

consume qw(
  Application::Model::Role::Has::IsMobileMediation
  Application::Model::Role::Has::ModerationReason::Page
  Application::Model::Role::Page::AdFoxNotifier
  Application::Model::Role::Page::Has::Assessor
  Application::Model::Role::Page::Has::BlocksLimit
  Application::Model::Role::Page::Has::HasMediationBlock
  Application::Model::Role::Page::Has::Moderation::MobileApp
  Application::Model::Role::Page::Has::MoneyMap::SDK
  Application::Model::Role::Page::Has::PICategories
  Application::Model::Role::Page::Has::IgnoreDomainCheck
  );

sub accessor      {'mobile_app_settings'}
sub db_table_name {'mobile_app_settings'}

sub get_product_name {gettext('mobile_app_settings')}

sub get_page_id_field_name {'context_page_id'}

sub get_block_model_names {
    return [map {"mobile_app_$_"} qw(rtb)];
}
sub block_seq_db_table {$_[0]->partner_db->mobile_app_block_seq}

sub get_opts_schema_name {'mobile_app_settings_opts'}

sub is_mobile_mediation_from_generated_column {TRUE}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        mobile_app     => 'Application::Model::Product::AN::MobileApp',
        mobile_app_rtb => 'Application::Model::Product::AN::MobileApp::RTB',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        %{$self->get_external_structure_rights_to_register()},

        view_search_filters__app_id => d_gettext('Right to view search filter "app_id"'),

        $self->get_description_right('add_adfox_block', d_gettext('Right to add adfox block in mobile applications'),),

        map {$self->get_description_right($_)} qw(
          add
          view_field__statistics_exists
          reject
          approve
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        %{$self->get_external_structure_model_fields()},

        application_id => {
            default     => TRUE,
            db          => TRUE,
            type        => 'number',
            need_check  => {len_min => 1,},
            api         => 1,
            adjust_type => 'str',
        },
        source_id => {
            db          => TRUE,
            type        => 'number',
            db_expr     => 'application_id',
            api         => 1,
            adjust_type => 'str',
        },
        context_page_id => {
            default     => TRUE,
            db          => TRUE,
            label       => d_gettext('Page id'),
            need_check  => {type => 'int_un', optional => TRUE,},
            adjust_type => 'str',
        },
        cpa => {
            db           => TRUE,
            check_rights => 'mobile_app_settings_view_field__cpa',
        },
        domain => {
            depends_on => ['store_id'],
            label      => d_gettext('Domain'),
            get        => sub {
                $_[1]->{'store_id'};
            },
            type => 'string',
            api  => 1,
        },
        app_is_approved => {
            forced_depends_on => ['application_id', 'mobile_app.is_approved'],
            get               => sub {
                $_[0]->{'mobile_app'}{$_[1]->{'application_id'}}{'is_approved'};
            },
            type        => 'boolean',
            api         => 1,
            adjust_type => 'str',
        },
        store_id => {
            forced_depends_on => ['application_id', 'mobile_app.store_id'],
            label             => d_gettext('Store ID'),
            get               => sub {
                $_[0]->{'mobile_app'}->{$_[1]->{'application_id'}}{'store_id'};
            },
            type => 'string',
            api  => 1,
        },
        store_url => {
            forced_depends_on => ['application_id', 'mobile_app.store_url'],
            label             => d_gettext('Store URL'),
            get               => sub {
                $_[0]->{'mobile_app'}->{$_[1]->{'application_id'}}{'store_url'};
            },
            need_check => {
                optional => TRUE,
                len_max  => 255,
            },
            api  => 1,
            type => 'string',
        },
        type => {
            forced_depends_on => ['application_id', 'mobile_app.type'],
            label             => d_gettext('Type ID'),
            get               => sub {
                $_[0]->{'mobile_app'}->{$_[1]->{'application_id'}}{'type'};
            },
            adjust_type => 'str',
        },
        type_name => {
            forced_depends_on => ['application_id', 'mobile_app.type_name'],
            label             => d_gettext('Type'),
            get               => sub {
                $_[0]->{'mobile_app'}->{$_[1]->{'application_id'}}{'type_name'};
            },
            api  => 1,
            type => 'string',
        },
        store => {
            forced_depends_on => ['application_id', 'mobile_app.store'],
            label             => d_gettext('Store'),
            get               => sub {
                $_[0]->{'mobile_app'}->{$_[1]->{'application_id'}}{'store'};
            },
            api  => 1,
            type => 'string',
        },
        apple_store_id => {
            forced_depends_on => ['application_id', 'mobile_app.apple_store_id'],
            label             => d_gettext('Apple Store ID'),
            get               => sub {
                $_[0]->{'mobile_app'}->{$_[1]->{'application_id'}}{'apple_store_id'};
            },
            type => 'string',
            api  => TRUE,
        },
        icon => {
            forced_depends_on => ['application_id', 'mobile_app.icon'],
            label             => d_gettext('App icon'),
            get               => sub {
                $_[0]->{'mobile_app'}->{$_[1]->{'application_id'}}{'icon'};
            },
            type => 'string',
            api  => TRUE,
        },
        developer => {
            forced_depends_on => ['application_id', 'mobile_app.developer'],
            label             => d_gettext('App developer'),
            get               => sub {
                $_[0]->{'mobile_app'}->{$_[1]->{'application_id'}}{'developer'};
            },
            type => 'string',
            api  => TRUE,
        },
    };
}

sub get_single_states {
    #flags: at least one of flags should be set, all flags from prev single states are cleared
    #flags=[] - all flags from prev single state are cleared
    return (
        {
            id    => 'archived',
            label => d_gettext('single_state archived'),
            flags => ['deleted'],
        },
        {
            id    => 'active',
            label => d_gettext('single_state active'),
            flags => [qw'working testing'],
        },
        {
            id    => 'blocked',
            label => d_gettext('single_state blocked'),
            flags => ['blocked'],
        },
        {
            id    => 'rejected',
            label => d_gettext('single_state rejected'),
            flags => ['rejected'],
        },
        {
            id    => 'stopped',
            label => d_gettext('single_state stopped'),
            flags => ['stopped'],
        },
        {
            id    => 'pending_review',
            label => d_gettext('single_state pending_review'),
            flags => ['need_approve'],
        },
        {
            id    => 'testing',
            flags => [],
            label => d_gettext('single_state testing'),
        },
    );
}

sub init {
    my ($self) = @_;
    $self->SUPER::init();
    $self->_init_single_state_multistates();
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $filter = $self->SUPER::get_structure_model_filter();

    $filter->{'fields'} = {
        %{$filter->{'fields'}},
        %{$self->get_external_structure_model_filter()},

        application_id => {type => 'number', label => d_gettext('App ID')},
        source_id      => {
            type      => 'number',
            label     => d_gettext('Source ID'),
            db_filter => sub {
                return ['application_id' => $_[1]->[1] => \$_[1]->[2]];
              }
        },
        domain_text => {type => 'alias', path => [qw(application store_id)]},
        application => {
            type           => 'subfilter',
            model_accessor => 'mobile_app',
            field          => 'application_id',
            label          => d_gettext('Application'),
        },
        context_page_id => {type => 'number', label => d_gettext('Page ID')},
        page_id         => {
            type      => 'number',
            label     => d_gettext('Page ID'),
            db_filter => sub {
                return ['context_page_id' => $_[1]->[1] => \$_[1]->[2]];
            },
        },
        type           => {type => 'alias', path => [qw(application type)]},
        apple_store_id => {type => 'alias', path => [qw(application apple_store_id)]},
        developer      => {type => 'alias', path => [qw(application developer)]},
    };

    return $filter;
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        (
            $self->check_rights('view_search_filters__login') ? {name => 'owner.login', label => gettext('Login')}
            : ()
        ),
        (
            $self->check_rights('view_search_filters__assistants_login')
            ? {name => 'assistants.user.login', label => gettext('Assistant\'s login')}
            : ()
        ),
        (
            $self->check_rights('view_search_filters__client_id')
            ? {name => 'owner.client_id', label => gettext('Client ID')}
            : ()
        ),
        (
            $self->check_rights('view_search_filters__user_type') ? {name => 'user_type', label => gettext('User type')}
            : ()
        ),
        {name => 'page_id',              label => gettext('Page ID')},
        {name => 'application.store_id', label => gettext('Store ID')},
        (
            $self->check_rights('view_search_filters__app_id') ? {name => 'application_id', label => gettext('App ID')}
            : ()
        ),
        {name => 'caption',      label => gettext('Caption')},
        {name => 'multistate',   label => gettext('Status')},
        {name => 'single_state', label => gettext('Single_state')},
        ($self->app->user_features->has_feature_simple_inapp() ? {name => 'type', label => gettext('Platform')} : ())
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_gettext('New'),
        multistates => [
            [working => d_pgettext('Settings status', 'Working')],
            [testing => d_pgettext('Settings status', 'Testing')],
            [stopped => d_pgettext('Settings status', 'Stopped')],
            [deleted => d_pgettext('Settings status', 'Archived')],
            [
                check_statistics => d_pgettext('Settings status', 'Check statistics'),
                private          => TRUE,
                deprecated       => TRUE
            ],
            [
                balance_registered => d_pgettext('Settings status', 'Added to Balance'),
                private            => TRUE,
            ],
            [
                created_in_mediaproducts => d_pgettext('Settings status', 'Created in "mediaproducts"'),
                private                  => TRUE,
                deprecated               => TRUE
            ],
            [blocked      => d_pgettext('Settings status', 'Blocked')],
            [protected    => d_pgettext('Settings status', 'Protected')],
            [need_update  => d_pgettext('Settings status', 'Need update')],
            [updating     => d_pgettext('Settings status', 'Updating')],
            [need_approve => d_pgettext('Settings status', 'Need approve')],
            [rejected     => d_pgettext('Settings status', 'Rejected')],
        ],
        actions => {
            delete                 => d_pgettext('Settings action', 'Archive'),
            restore                => d_pgettext('Settings action', 'Restore'),
            edit                   => d_pgettext('Settings action', 'Edit'),
            start_testing          => d_pgettext('Settings action', 'Start testing'),
            stop_testing           => d_pgettext('Settings action', 'Stop testing'),
            set_check_statistics   => d_pgettext('Settings action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Settings action', 'Reset "check_statistics"'),
            start_block            => d_pgettext('Settings action', 'Start block on campaign'),
            restore_block          => d_pgettext('Settings action', 'Restore block on campaign'),
            can_update_in_bk       => d_pgettext('Settings action', 'Can update in bk'),
        },
        right_group       => [mobile_app_settings => d_gettext('Right to manage mobile app settings')],
        right_name_prefix => $self->accessor() . '_',
        right_actions     => {
            add              => d_pgettext('Settings action', 'Add'),
            start            => d_pgettext('Settings action', 'Start'),
            stop             => d_pgettext('Settings action', 'Stop'),
            set_need_approve => d_pgettext('Settings action', 'Set need approve'),
            approve          => d_pgettext('Settings action', 'Approve'),
            reject           => d_pgettext('Settings action', 'Reject'),
            set_protected    => d_pgettext('Settings action', 'Set protected'),
            reset_protected  => d_pgettext('Settings action', 'Reset protected'),
            set_need_update =>
              {label => d_pgettext('Settings action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Settings action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Settings action', 'Stop update'),  dont_write_to_action_log => TRUE},
            register_in_balance => d_pgettext('Settings action', 'Register in Balance'),
            reset_blocked       => d_pgettext('Settings action', 'Reset blocked'),
            set_blocked         => d_pgettext('Settings action', 'Set blocked'),
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action => 'set_need_approve',
                from =>
'(rejected and not blocked) or __EMPTY__ or (not working and not testing and not stopped and not deleted and not blocked and not protected and not need_approve)',
                set_flags   => ['need_approve'],
                reset_flags => ['rejected'],
            },
            {
                action      => 'reject',
                from        => 'not rejected',
                reset_flags => ['need_approve'],
                set_flags   => ['rejected'],
            },
            {
                action      => 'approve',
                from        => 'need_approve',
                reset_flags => ['need_approve'],
            },
            {
                action => 'start_testing',
                from =>
'balance_registered and not (testing or working or deleted or protected or need_approve or rejected or blocked)',
                set_flags   => ['testing'],
                reset_flags => ['stopped'],
            },
            {
                action      => 'stop_testing',
                from        => 'testing and not protected',
                set_flags   => ['stopped'],
                reset_flags => ['testing'],
            },
            {
                action => 'start',
                from =>
                  'balance_registered and not (working or deleted or protected or need_approve or rejected or blocked)',
                set_flags   => ['working'],
                reset_flags => ['testing', 'stopped']
            },
            {
                action      => 'stop',
                from        => 'working and not protected',
                set_flags   => ['stopped'],
                reset_flags => ['working', 'testing'],
            },
            {
                action    => 'delete',
                from      => 'not deleted and (need_approve or stopped or rejected or __EMPTY__)',
                set_flags => ['deleted']
            },
            {
                action      => 'restore',
                from        => 'deleted and not blocked',
                reset_flags => ['deleted'],
            },
            {
                action => 'edit',
                from   => 'not (deleted or blocked or rejected)'
            },
            {
                action => 'start_block',
                from   => '(working or testing) and not protected',
            },
            {
                action => 'restore_block',
                from   => 'not deleted',
            },
            {
                action => 'can_update_in_bk',
                from   => 'not (deleted or protected)',
            },
            {
                action    => 'set_protected',
                from      => 'not protected',
                set_flags => ['protected'],
            },
            {
                action      => 'reset_protected',
                from        => 'protected',
                reset_flags => ['protected'],
            },
            {
                action    => 'set_need_update',
                from      => 'not (deleted)',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update or updating',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
            {
                action    => 'register_in_balance',
                from      => 'not (balance_registered or deleted or need_approve or rejected or blocked)',
                set_flags => ['balance_registered'],
            },
            {
                action      => 'set_blocked',
                from        => 'not blocked',
                set_flags   => ['blocked'],
                reset_flags => ['need_approve'],
            },
            {
                action      => 'reset_blocked',
                from        => 'blocked',
                reset_flags => ['blocked'],
            },
        ]
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    my $ids = array_uniq(map {$_->{'application_id'} // ()} @$result);
    my $page_ids = array_uniq(map {$_->{'context_page_id'} // $_->{'page_id'} // ()} @$result);

    $self->SUPER::pre_process_fields($fields, $result, ids => $ids, page_ids => $page_ids);
    $self->Application::Model::Page::MIXIN::External::pre_process_fields($fields, $result);

    if ($fields->need('bk_languages') || $fields->need('bk_languages_name')) {
        my %bk_languages =
          map {$_->{'id'} => $_->{'language'}} @{$self->bk_language->get_all(fields => [qw(id language)])};

        my @bk_languages_by_context_page_id = @{
            $self->partner_db->mobile_app_bk_language->get_all(
                fields => [qw(context_page_id language_id)],
                filter => {context_page_id => $page_ids},
            )
          };

        $fields->{'bk_languages'}{$_->{'context_page_id'}}{$_->{'language_id'}} = $bk_languages{$_->{'language_id'}}
          foreach @bk_languages_by_context_page_id;

        %bk_languages = reverse(%bk_languages);

        foreach my $context_page_id (@$page_ids) {
            $fields->{'bk_languages'}{$context_page_id} ||=
              {map {$bk_languages{$_} => $_} grep {$bk_languages{$_}} @BK_LANGUAGES_BY_DEFAULT};
        }
    }

}

sub get_actions_depends {
    [qw(id multistate app_is_approved owner_id cur_user_is_read_assistant)];
}

sub get_editable_fields_depends {
    [qw(id multistate app_is_approved owner_id context_page_id cur_user_is_read_assistant)];
}

sub get_available_fields_depends {[qw(owner_id)]}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->SUPER::get_available_fields($obj);
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s' => [
                qw( comment statistics_exists login
                  fast_context view_images only_picture
                  block_title false_click is_tutby performance_tgo_disable)
            ],
            $accessor . '_view_field__owner'          => [qw(owner)],
            $accessor . '_view_field__application_id' => [qw(application_id source_id)],
            $accessor . '_view_field__bk_languages'   => [qw(bk_languages bk_languages_name)],
        }
    );

    $self->handle_field_assistants($obj, \%fields, 'view_field__assistants');

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->SUPER::get_add_fields();

    $fields->{'app_id'} = TRUE;
    $fields->{'login'} = TRUE if $self->check_short_rights('add_other');

    return $fields;
}

sub collect_editable_fields {
    my ($self, $object) = @_;

    my $res = {%{$self->SUPER::collect_editable_fields($object)}, %{$self->_get_common_add_edit_fields()},};

    delete($res->{'caption'}) unless $self->check_right_for_do_action_mobile_mediation($object);

    map {$res->{$_} = TRUE} qw(
      family_filter
      behavioral_targeting
      excluded_domains
      excluded_phones
      store_url
      );

    foreach (
        qw(
        bk_languages
        block_title
        false_click
        fast_context
        view_images
        only_picture
        )
      )
    {
        $res->{$_} = TRUE if $self->check_short_rights("edit_field__$_");
    }

    $self->handle_field_assistants($object, $res, 'edit_field__assistants');

    # TODO удалить следующий блок в одном из следующих ПР, например по PI-15793
    unless (defined($object->{'context_page_id'})) {
        delete($res->{$_}) foreach (qw(assistants excluded_domains excluded_phones));
    }

    if ($self->check_multistate_flag($object->{'multistate'}, 'protected')
        && !$self->check_rights('edit_protected_pages'))
    {
        foreach (keys(%{$self->get_need_update_in_bk_fields()})) {
            delete($res->{$_});
        }
    }

    return $res;
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        res_fields      => $self->SUPER::_get_common_add_edit_fields(),
        no_right_fields => ['caption'],
        right_fields    => {edit => ['performance_tgo_disable']}
    );

    return $fields;
}

sub can_action_approve {
    my ($self, $obj) = @_;

    my $object = $self->_get_object_fields($obj, [qw(app_is_approved)]);

    return $object->{'app_is_approved'};
}

sub on_action_approve {
    my ($self, $obj, %opts) = @_;

    my $object = $self->_get_object_fields($obj, ['id']);

    my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions('register_in_balance'));
    $self->do_action($object, 'register_in_balance') if $self->check_action($object->{'id'}, 'register_in_balance');

    if ($self->app->user_features->has_feature_simple_inapp()) {
        # skip missing agreement exception
        if ($self->has_active_agreement($obj)) {
            $self->maybe_do_action($obj, 'start');
        }
    } else {
        $self->maybe_do_action($obj, 'start_testing');
    }
}

sub on_action_register_in_balance {
    my ($self, $obj, %opts) = @_;

    my $object = $self->_get_object_fields(
        $obj,
        [
            qw(
              page_id
              client_id
              store_id
              owner
              create_date
              )
        ]
    );

    $self->api_balance->create_or_update_place(
        operator_uid  => $self->get_option(cur_user => {})->{'id'},
        client_id     => $object->{'client_id'},
        page_id       => $object->{'page_id'},
        domain        => $object->{'store_id'},
        campaign_type => 3,
        is_tutby      => $object->{'owner'}{'is_tutby'},
        create_date   => $object->{'create_date'},
    );
}

sub can_action_reject {TRUE}

sub on_action_reject {
    my ($self, $obj, %opts) = @_;

    $self->maybe_do_action($obj, 'stop_testing');
    $self->maybe_do_action($obj, 'stop');
}

sub on_action_reset_blocked { }

sub can_action_set_blocked {TRUE}

sub on_action_set_blocked {
    my ($self, $obj, %opts) = @_;

    $self->do_action($obj, 'stop') if $self->check_action($obj, 'stop');
}

sub can_action_set_need_approve {
    my ($self, $obj) = @_;
    return !!$self->_get_object_fields($obj, [qw(store_url)])->{'store_url'};
}

sub on_action_set_need_approve { }

sub query_filter {
    my ($self, $filter, $opts) = @_;

    $filter = $self->limit_filter_by_tutby_and_assistant_and_robot_assistant($filter, $opts);

    return $filter;
}

sub get_bk_data {
    my ($self, $page) = @_;

    my %data = $self->SUPER::get_bk_data($page);

    my $user_id = $page->{'owner_id'};

    my @user_global_excluded_phones =
      map {$_->{'phone'}} @{$self->user_global_excluded_phones->get_all(filter => {user_id => $user_id})};

    my @user_global_excluded_domains =
      map {$_->{'domain'}} @{$self->user_global_excluded_domains->get_all(filter => {user_id => $user_id})};

    my @domains = check_domains_for_absorption([@{$page->{'excluded_domains'}}, @user_global_excluded_domains]);

    my %hash_excluded_domains = map {$_ => TRUE} @domains,
      bs_format_phone([@{$page->{'excluded_phones'}}, @user_global_excluded_phones]);

    %data = (%data, excluded_domains => [sort keys(%hash_excluded_domains)],);

    unless ($self->check_multistate_flag($page->{multistate}, 'balance_registered')) {
        # flag for BK/SDK to not show real ads and instead show mocks
        $data{MockMode} = 1;
        # https://st.yandex-team.ru/BSSERVER-19902#61a9c6883e3d400053772b83
        unless ($page->{store_id}) {
            $data{domain} = sprintf('com.test.%d.%s', $page->{page_id}, lc($page->{type_name} // 'undef'));
        }
    }

    return %data;
}

# API

sub api_available_actions {
    return qw(
      edit
      delete
      reject
      reset_blocked
      restore
      set_blocked
      set_need_approve
      start
      stop
      );
}

sub api_can_add {TRUE}

sub api_can_edit {TRUE}

sub approve_for_app {
    my ($self, $app_id) = @_;

    my $settings = $self->get_all(
        fields => [
            qw(id
              caption
              owner_id
              multistate
              page_id
              client_id
              store_id
              owner_id
              is_mobile_mediation)
        ],
        filter => {application_id => $app_id, multistate => 'need_approve and not protected'},
    );

    $self->do_action($_, 'approve', %$_) for (@$settings);

    if ($settings->[0]->{is_mobile_mediation}) {
        $self->app->mail_notification->add_when_first_app_added_to_mediation(
            {
                caption  => $settings->[0]->{caption},
                owner_id => $settings->[0]->{owner_id}
            }
        );
    }
}

sub check_action {&Application::Model::Page::MIXIN::External::check_action}

sub get_block_types {
    my ($self, %opts) = @_;

    return [map {{id => $_, label => $BLOCK_TYPES{$_}{'label'}()}} keys %BLOCK_TYPES];
}

sub reject_for_app {
    my ($self, $app_id, %opts) = @_;

    my $protected_pages =
      $self->get_all(fields => [qw(id)], filter => {application_id => $app_id, multistate => 'protected'});
    WARN {
        message => sprintf("App with protected pages is rejected. Protected pages: %s",
            join(', ', (map {$_->{id}} @$protected_pages))),
        fingerprint => [$self->accessor(), 'App with protected pages is rejected']
      }
      if @$protected_pages;

    my $settings = $self->get_all(
        fields => $self->get_depends_for_field('actions'),
        filter => {application_id => $app_id, multistate => 'not rejected and not protected'},
    );

    foreach (@$settings) {
        $self->do_action($_, 'reject', %opts)
          if $self->check_action($_->{'id'}, 'reject');
    }
}

sub set_need_approve_for_app {
    my ($self, $app_id) = @_;

    my $settings = $self->get_all(
        fields => $self->get_depends_for_field('actions'),
        filter => {application_id => $app_id, multistate => $self->get_multistate_by_action('set_need_approve'),},
    );

    foreach (@$settings) {
        $self->do_action($_->{'id'}, 'set_need_approve')
          if $self->check_action($_->{'id'}, 'set_need_approve');
    }
}

sub related_models {
    my ($self) = @_;

    return {
        %{$self->SUPER::related_models()},
        %{$self->external_related_models()},

        mobile_app => {
            accessor => 'mobile_app',
            filter   => sub {
                return {id => array_uniq(map {$_->{'application_id'} // ()} @{$_[1]})};
            },
            key_fields => ['id'],
        }
    };
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_processing_before_validation($opts);

    $opts->{application_id} = delete $opts->{app_id} if ($self->hook_stash->mode('add'));
}

sub hook_fields_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_validation($opts);

    my $owner_id = $self->hook_stash->get('user')->{'id'};
    my $app_id   = $opts->{application_id};

    my @apps = grep {$_->{'id'} == $app_id} @{$self->mobile_app->get_app_for_adding($owner_id)};
    throw Exception::Validation::BadArguments gettext("You can't add a mobile campaigns for this APP ID: %s", $app_id)
      unless @apps;

    #add store_id for IgnoreDomainCheck role
    $self->hook_stash->get('current')->{store_id} = $apps[0]->{store_id};
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    my $app = $self->mobile_app->get($opts->{'application_id'}, 'fields' => 'multistate',);
    if ($self->mobile_app->check_multistate_flag($app->{'multistate'}, 'approved')) {
        my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions(qw(set_need_approve approve)));
        $self->do_action($self->hook_stash->get('id'), 'set_need_approve');
        $self->do_action($self->hook_stash->get('id'), 'approve');
    } elsif ($self->mobile_app->check_multistate_flag($app->{'multistate'}, 'rejected')) {
        my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions(qw(set_need_approve reject)));
        $self->do_action($self->hook_stash->get('id'), 'set_need_approve');
        $self->do_action($self->hook_stash->get('id'), 'reject');
    } elsif ($self->mobile_app->check_multistate_flag($app->{'multistate'}, 'need_approve')) {
        my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions(qw(set_need_approve)));
        $self->do_action($self->hook_stash->get('id'), 'set_need_approve');
    }

    if ($self->app->user_features->has_feature_simple_inapp()) {
        my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions('set_need_update'));
        if ($self->check_action($self->hook_stash->get('id'), 'set_need_update')) {
            $self->do_action($self->hook_stash->get('id'), 'set_need_update');
        }
    }
}

sub may_send_not_balance_registered {TRUE}

sub is_can_per_page_invite {TRUE}

sub has_feature_simple_inapp {
    my ($self) = @_;

    return ($self->app->user_features->has_feature_simple_inapp());
}

sub page_owner_has_feature_simple_inapp {
    my ($self, $obj) = @_;

    return ($self->app->user_features->page_owner_has_feature_simple_inapp($obj));
}

TRUE;
