package Application::Model::Product::AN::SearchOnSite::Campaign;

use qbit;

use base qw(
  Application::Model::Page::Site::External
  RestApi::MultistateModel
  );

use URI;
use PiConstants qw(
  $BIG_SEARCH_TARGET_TYPE
  $SEARCH_TARGET_TYPE
  @BK_LANGUAGES_BY_DEFAULT
  );

consume qw(
  Application::Model::Role::Has::ModerationReason::Page
  Application::Model::Role::Has::Search
  Application::Model::Role::Has::Tier
  Application::Model::Role::Page::Has::MobileAppMode
  Application::Model::Role::Page::Has::MetricaCounters
  Application::Model::Role::Page::Has::BlocksLimit
  Application::Model::Role::Page::Has::Assessor
  Application::Model::Role::Page::Has::Moderation::Site
  Application::Model::Role::Page::Has::MoneyMap::Web
  );

sub accessor             {'search_on_site_campaign'}
sub db_table_name        {'search_on_site_campaign'}
sub get_opts_schema_name {'search_on_site_campaign_opts'}
sub get_product_name     {gettext('search_on_site_campaign')}

sub get_page_id_field_name {'page_id'}

sub get_block_model_names {
    return [map {"search_on_site_$_"} qw(premium direct)];
}

sub db_banner_lang {$_[0]->partner_db->search_on_site_campaign_banner_lang()}

sub block_seq_db_table {$_[0]->partner_db->search_on_site_block_seq}

sub get_block_models_for_available_levels {
    # Hack for context_on_site_market_api here, just to show statistics.
    # It is not a full-featured product, it does not have real blocks
    # and should not be turned on in BS from PI2.
    #
    return $_[0]->get_block_models(), $_[0]->search_on_site_market_api;
}

sub get_need_update_in_bk_block_models {
    qw(
      search_on_site_premium
      search_on_site_direct
      search_on_site_market
      search_on_site_mcb
      )
}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        search_on_site_direct     => 'Application::Model::Product::AN::SearchOnSite::Direct',
        search_on_site_premium    => 'Application::Model::Product::AN::SearchOnSite::Premium',
        search_on_site_market     => 'Application::Model::Product::AN::SearchOnSite::Market',
        search_on_site_mcb        => 'Application::Model::Product::AN::SearchOnSite::MCB',
        search_on_site_market_api => 'Application::Model::Product::AN::SearchOnSite::MarketAPI',

        search_on_site_mirrors => 'Application::Model::Product::AN::SearchOnSite::Campaign::Mirrors',
    };
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},

        yandex_xml => {
            db         => TRUE,
            type       => 'boolean',
            api        => 1,
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
            adjust_type => 'str',
        },

        #additional products
        search_on_site_market_api_count => {
            depends_on => [qw(market_api)],
            get        => sub {
                return $_[1]->{'market_api'} ? 1 : 0;
            },
            type        => 'number',
            api         => 1,
            adjust_type => 'int',
        },
        search_on_site_yandex_xml_count => {
            depends_on => [qw(yandex_xml)],
            get        => sub {
                return $_[1]->{'yandex_xml'} ? 1 : 0;
            },
            type        => 'number',
            adjust_type => 'int',
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $filter = $self->SUPER::get_structure_model_filter();

    $filter->{'fields'} = {
        %{$filter->{'fields'}},
        yandex_xml => {type => 'boolean', label => d_gettext('XML')},
        product    => {
            type   => 'dictionary',
            label  => d_gettext('Product'),
            values => sub {
                [
                    {id => 'market_api', label => gettext('market_api')},
                    {id => 'yandex_xml', label => gettext('yandex_xml')},
                ];
            },
            db_filter => sub {
                my ($model, $data, $filter_dif) = @_;

                return [($data->[1] eq '=' ? 'OR' : 'AND'), [map {[$_, $data->[1], \1]} @{$data->[2]}]];
            },
        },
    };

    return $filter;
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    my $filters = $self->SUPER::get_db_filter_simple_fields();

    push(@$filters, {name => 'product', label => gettext('Product')});

    return $filters;
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my $res = {%{$self->SUPER::collect_editable_fields($data)}, %{$self->_get_common_add_edit_fields()}};

    $res->{$_} = TRUE foreach (
        qw(
        family_filter
        mirrors
        excluded_domains
        excluded_phones
        yandex_xml
        page_lang
        )
    );

    foreach (
        qw(
        reload_timeout
        block_title
        banner_lang
        comment
        market_api
        user_synchronization
        )
      )
    {
        $res->{$_} = TRUE
          if $self->check_short_rights('edit_field__' . $_);
    }

    # TODO удалить следующий блок в одном из следующих ПР, например по PI-15793
    unless (defined($data->{'page_id'})) {
        delete($res->{$_}) foreach (qw(excluded_domains excluded_phones assistants mirrors));
    }

    if (   $self->check_short_rights('edit_field__user_synchronization')
        || $data->{'user_synchronization'})
    {
        $res->{'tag_id'} = TRUE;
    }

    if ($data->{'is_read_only'}) {
        foreach (
            qw(
            block_title
            ),
          )
        {
            delete($res->{$_});
        }
    }

    if ($self->check_multistate_flag($data->{'multistate'}, 'protected')
        && !$self->check_rights('edit_protected_pages'))
    {
        foreach (qw(mirrors), keys(%{$self->get_need_update_in_bk_fields()})) {
            delete($res->{$_});
        }
    }

    return $res;
}

sub on_action_approve { }

sub can_action_reject {TRUE}

sub on_action_reject { }

sub on_action_reset_blocked { }

sub can_action_set_blocked {TRUE}

sub on_action_set_blocked {
    my $self = shift;

    return $self->SUPER::on_action_set_blocked(@_);
}

sub can_action_set_need_approve {TRUE}

sub on_action_set_need_approve { }

sub get_bk_data {
    my ($self, $page) = @_;

    my %data = $self->SUPER::get_bk_data($page);

    # Хак для того чтобы можно было работать с площадками у которых в БК TargetType = 0
    my $target_type = in_array($page->{'page_id'}, [qw(114 131 136)]) ? $BIG_SEARCH_TARGET_TYPE : $SEARCH_TARGET_TYPE;

    %data = (
        %data,
        target_type => $target_type,
        options     => {
            dontshowsex   => $page->{'family_filter'},
            ReloadTimeout => $page->{'reload_timeout'},
            BlockTitle    => $page->{'block_title'},
        },
        dsp_info     => [],
        page_options => $self->page_options->get_options($page->{'page_id'}),
    );

    return %data;
}

sub block_for_site {
    my ($self, $domain_id) = @_;

    my $campaigns = $self->get_all(
        fields => [qw(id domain client_id)],
        filter => {domain_id => $domain_id, multistate => 'not blocked'},
    );

    foreach (@$campaigns) {
        $self->do_action($_->{'id'}, 'set_blocked', 'suppress_mail_notification' => 1,);
    }
}

sub get_default_settings {
    return {
        mobile_app_mode      => 0,
        user_synchronization => 0,
    };
}

sub api_available_actions {
    return qw(
      del_mirror
      approve
      delete
      edit
      reject
      reset_blocked
      reset_read_only
      restore
      set_blocked
      set_need_approve
      set_read_only
      start
      start_testing
      stop
      );
}

sub api_can_add {TRUE}

sub api_can_edit {TRUE}

sub hook_fields_validation {
    my ($self, $opts) = @_;
    $self->SUPER::hook_fields_validation($opts);

    throw Exception::Validation::BadArguments gettext('Search campaigns are not allowed on this domain')
      if $self->hook_stash->get('site')->{'is_search_blocked'};
}

TRUE;
