package Application::Model::Product::AN::SearchOnSite::Premium;

use qbit;

use base qw(
  Application::Model::Block::External::Site
  );

consume qw(
  Application::Model::Role::Has::Block::DesignSettings
  );

use QBit::Validator;
use PiConstants qw($DIRECT_PLACE_ID);

use Application::Model::Product::AN::SearchOnSite::BlockTypes::Premium qw(
  direct_block_types
  @FIELDS
  @COLOR_FIELDS
  @BOOLEAN_FIELDS
  direct_appearance_fields
  %DIRECT_BLOCKS
  );

use Exception::Denied;
use Exception::Validation::BadArguments;

consume qw(
  Application::Model::Role::Has::CustomBkData
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Actions
  Application::Model::Role::Block::Has::BlocksLimit
  );

sub accessor      {'search_on_site_premium'}
sub db_table_name {'search_on_site_premium'}

sub get_campaign_model_name {
    return 'search_on_site_campaign';
}

sub get_product_name {gettext('search_on_site_premium')}

sub get_page_id_field_name {'campaign_id'}

sub public_id_prefix {'P-A-'}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            name        => 'search_on_site_premium',
            description => d_gettext('Right to manage Premium blocks on search sites'),
            rights      => {
                search_on_site_premium_view_all => d_gettext('Right to view all Premium blocks on search sites'),
                search_on_site_premium_view_action_log =>
                  d_gettext('Right to view Premium blocks action logs on search sites'),
                search_on_site_premium_view_zero_block => d_gettext('Right to view zero search Premium blocks'),
                search_on_site_premium_view => d_gettext('Right to view Premium blocks on search sites in menu'),
                search_on_site_premium_view_field__comment =>
                  d_gettext('Right to view field "comment" of premium blocks on search campaigns'),
                search_on_site_premium_edit => d_gettext('Right to edit Premium block to search site'),
                search_on_site_premium_view_filter__login =>
                  d_gettext('Right to view filter "login" of premium blocks on search campaigns'),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        type  => {db => TRUE, label => d_gettext('Block format')},
        limit => {db => TRUE, label => d_gettext('Count ad'), adjust_type => 'str',},
        (map {$_ => {db => TRUE, adjust_type => 'str',}} direct_appearance_fields()),
        block_caption => {
            depends_on => ['type', 'limit'],
            get        => sub {
                throw gettext('Block with type "%s" not found', $_[1]->{'type'})
                  unless $_[0]->{'BLOCK_TYPES_CAPTION'}{$_[1]->{'type'}};

                $_[0]->{'BLOCK_TYPES_CAPTION'}{$_[1]->{'type'}}($_[1]->{'limit'});
              }
        },
        #redefine
        fields_depends => {
            get => sub {{}}
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        (
            $self->check_rights('view_search_filters__login')
            ? {name => 'campaign.owner.login', label => gettext('Login')}
            : ()
        ),
        (
            $self->check_rights('view_search_filters__user_type') ? {name => 'user_type', label => gettext('User type')}
            : ()
        ),
        {name => 'campaign_id',         label => gettext('Page ID')},
        {name => 'id',                  label => gettext('Block ID')},
        {name => 'caption',             label => gettext('Block\'s caption')},
        {name => 'campaign.all_domain', label => gettext('Domain and mirror')},
        {name => 'multistate',          label => gettext('Status')},
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_pgettext('Block multistate', 'New'),
        multistates => [
            [deleted           => d_pgettext('Block multistate', 'Archived')],
            [working           => d_pgettext('Block multistate', 'Working')],
            [check_statistics  => d_pgettext('Block multistate', 'Check statistics')],
            [deleted_with_page => d_pgettext('Block multistate', 'Archived with page'), private => TRUE],
            [need_update       => d_pgettext('Block multistate', 'Need update')],
            [updating          => d_pgettext('Block multistate', 'Updating')],
        ],
        actions => {
            edit                   => d_pgettext('Block action', 'Edit'),
            delete                 => d_pgettext('Block action', 'Archive'),
            delete_with_page       => d_pgettext('Block action', 'Archive with page'),
            restore                => d_pgettext('Block action', 'Restore'),
            restore_with_page      => d_pgettext('Block action', 'Restore with page'),
            start                  => d_pgettext('Block action', 'Start'),
            stop                   => d_pgettext('Block action', 'Stop'),
            set_check_statistics   => d_pgettext('Block action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Block action', 'Reset "check_statistics"'),
        },
        right_name_prefix => $self->accessor . '_',
        right_actions     => {
            add => d_pgettext('Block action', 'Add'),
            set_need_update =>
              {label => d_pgettext('Block action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Block action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Block action', 'Stop update'),  dont_write_to_action_log => TRUE},
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action    => 'delete_with_page',
                from      => 'not deleted',
                set_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page',
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action    => 'start',
                from      => 'not working and not deleted',
                set_flags => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
            },
            {
                action    => 'set_check_statistics',
                from      => 'not deleted',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('block_caption')) {
        $fields->{'BLOCK_TYPES_CAPTION'} =
          {map {$_->{'id'} => $_->{'constuctor_caption'}} @{$self->get_types_premium_blocks}};
    }
}

sub get_available_fields_depends {
    [qw(multistate)];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor      = $self->accessor();
    my $page_accessor = $self->get_campaign_model_name();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s' => [qw( comment is_tutby)],

            $page_accessor . '_view_field__%s'    => [qw(login domain_id)],
            $page_accessor . '_view_field__owner' => [qw(owner_client_id)],
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my %fields =
      map {$_ => TRUE} qw(campaign_id caption), $self->direct_block_fields(),
      $self->direct_appearance_fields();

    $fields{'comment'} = TRUE if $self->check_short_rights('view_field__comment');

    return \%fields;
}

sub get_actions_depends {
    [qw(id campaign_id page multistate is_protected cur_user_is_read_assistant)];
}

sub get_editable_fields_depends {
    [qw(page_id page campaign_id id multistate is_protected cur_user_is_read_assistant)];
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my %res;

    $res{$_} = TRUE foreach (qw(caption type limit), direct_appearance_fields());

    if ($res{'type'}) {
        $res{'type'} = $self->get_types_premium_blocks();
        foreach (@{$res{'type'}}) {
            delete($_->{'constuctor_caption'});
            $_->{'caption'} = $_->{'caption'}->();
        }
    }

    $res{'comment'} = TRUE
      if $self->check_short_rights('view_field__comment');

    return \%res;
}

#Не убирать пока не переведено на хуки
#TODO: После перехода на хуки убрать эти две вирутальные сабы. Сделано для ограничения блоков.
sub hook_fields_validation { }

sub hook_processing_after_insert { }

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied gettext('Access denied')
      unless $self->check_rights($self->get_rights_by_actions('add'));

    $self->_trim_opts(\%opts);
    $self->_check_opts(\%opts);

    throw Exception::Validation::BadArguments gettext('Invalid type block')
      unless $self->check_type_premium_block($opts{'type'});

    throw Exception::Validation::BadArguments gettext('Cannot find campaign')
      unless @{$self->get_all_campaigns_for_adding(filter => [page_id => '=' => $opts{$self->get_page_id_field_name()}])
          };

    $self->hook_fields_validation(\%opts);

    my $block_id;

    my $model_fields = $self->get_model_fields();

    my @model_opts_fields;
    if ($self->has_opts_storage()) {
        @model_opts_fields = grep {$model_fields->{$_}{'from_opts'}} keys(%$model_fields);

        my $db_opts = {($opts{'opts'} ? %{$opts{'opts'}} : ()), hash_transform(\%opts, \@model_opts_fields)};

        $self->force_to_number($db_opts, $model_fields);

        $opts{'opts'} = to_json($db_opts);

        my $qv = $self->app->validator->check(
            data     => $opts{'opts'},
            app      => $self,
            template => {
                type   => 'json',
                schema => $self->get_opts_schema_name(),
            }
        );

        throw Exception $qv->get_all_errors if $qv->has_errors;
        delete $opts{$_} foreach @model_opts_fields;
    }

    $self->partner_db->transaction(
        sub {
            my $id = $self->search_on_site_campaign->get_next_block_id($opts{$self->get_page_id_field_name()});

            $block_id = $self->partner_db_table()->add(
                {
                    id => $id,
                    map {$_ => $opts{$_}}
                      @{arrays_difference([sort keys %{$self->get_add_fields()}], \@model_opts_fields)}
                }
            );

            $self->do_action($block_id, 'add', %opts);
            $self->maybe_do_action($block_id, 'start');

            $self->update_in_bk($block_id);
        }
    );

    $self->hook_processing_after_insert($block_id);

    return $block_id;
}

sub on_action_edit {
    my ($self, $block, %data) = @_;

    $self->check_bad_fields($block, \%data);

    $self->_trim_opts(\%data);
    $self->_check_opts(\%data);

    if (%data) {
        $self->partner_db_table()->edit($block, \%data);

        $self->update_in_bk($block) if $self->need_update_in_bk([keys(%data)]);
    }

    return TRUE;
}

sub duplicate {
    my ($self, %opts) = @_;

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {public_id => {type => 'publicid'}}
        },
        app   => $self,
        throw => TRUE
    );

    my $entity = $self->get($opts{'public_id'}, fields => [sort keys(%{$self->get_add_fields()}), 'is_deleted'],);

    throw Exception::Validation::BadArguments gettext('You can not duplicate the deleted block')
      if delete($entity->{'is_deleted'});

    my $result = $self->add(%$entity);

    $self->hook_processing_after_insert({campaign_id => $result->[0], id => $result->[1]});

    return $self->public_id({campaign_id => $result->[0], id => $result->[1]});
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_campaign($filter);
    $filter = $self->limit_filter_view_zero_block($filter);

    return $filter;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = map {$_ => TRUE} direct_block_fields(), direct_appearance_fields();

    return \%fields_need_update;
}

sub boolean_fields {
    return {map {$_ => TRUE} @BOOLEAN_FIELDS};
}

sub check_type_premium_block {
    my ($self, $type) = @_;

    return $DIRECT_BLOCKS{$type};
}

sub direct_block_fields {
    return qw(type limit);
}

sub direct_blocks {
    return \%DIRECT_BLOCKS;
}

sub get_bk_block_data {
    my ($self, $block) = @_;

    return {};
}

sub get_bk_block_data_key {'direct_blocks'}

sub get_bk_common_data {
    return (places => {436 => {}, $DIRECT_PLACE_ID => {}});
}

sub get_types_premium_blocks {
    my ($self) = @_;

    return $self->direct_block_types();
}

#TODO: implement standard validation (ValidatableMixin) and remove this
sub _check_opts {
    my ($self, $data) = @_;

    if (exists($data->{'bk_data'})) {
        try {
            from_json($data->{'bk_data'});
        }
        catch {
            throw Exception::Validation::BadArguments gettext('Incorrect custom bk data');
        };
    }
}

sub _trim_opts {
    my ($self, $opts) = @_;

    foreach my $field_name (qw(caption)) {
        $opts->{$field_name} =~ s/^\s+|\s+$//g if $opts->{$field_name};
    }
}

sub design_field {'Design'}

TRUE;
