package Application::Model::Product::AN::Site;

use qbit;

use base qw(
  Application::Model::Source
  Application::Model::ValidatableMixin
  RestApi::MultistateModel
  Application::Model::DAC
  );

consume qw(
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::Moderation::Site
  Application::Model::Role::Has::ModerationReason::Site
  );

use Utils::JSON qw(fix_type_for_complex);

use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;
use PiConstants qw($MYSQL_MIN_DATETIME);

sub accessor             {'site'}
sub db_table_name        {'site'}
sub get_opts_schema_name {'site_opts'}
sub get_product_name     {gettext('site')}
sub get_page_model_names {qw(context_on_site_campaign search_on_site_campaign)}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        users                    => 'Application::Model::Users',
        api_http_gozora          => 'QBit::Application::Model::API::Yandex::HTTPGoZora',
        owner_site               => 'Application::Model::Product::AN::Site::OwnerSite',
        context_on_site_campaign => 'Application::Model::Product::AN::ContextOnSite::Campaign',
        search_on_site_campaign  => 'Application::Model::Product::AN::SearchOnSite::Campaign',
        mail_notification        => 'Application::Model::MailNotification',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        map {$self->get_description_right($_)}
          qw(
          view_field__data_for_site_stat
          view_filter__login
          view_field__is_graysite
          ),
    };

    $rights->[0]{'rights'}{'site_add_yandex_subdomains'} = d_gettext('Right to add sites on yandex subdomains');

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields},
        domain => {
            default    => TRUE,
            db         => TRUE,
            label      => d_gettext('Site'),
            type       => 'string',
            need_check => {
                len_min => 4,
                len_max => 255,
                check   => sub {
                    my ($qv, $domain) = @_;

                    throw Exception::Validator::Fields gettext('Domain doesn\'t exist')
                      unless get_domain($domain);

                    if (  !$qv->app->check_rights('site_add_yandex_subdomains')
                        && $domain =~ m/yandex\.(?:(?:ru)|(?:com(?:\.tr)?))$/)
                    {
                        throw Exception::Validator::Fields gettext('Don\'t add domain "%s"', $domain);
                    }

                    throw Exception::Validator::Fields gettext('Domain doesn\'t exist')
                      unless $qv->app->api_http_gozora->is_site_working($domain);
                },
            },
            api => TRUE
        },
        is_graysite =>
          {db => TRUE, label => d_gettext('Gray site'), type => 'boolean', api => TRUE, adjust_type => 'str',},
        data_for_site_stat => {
            depends_on   => ['id'],
            check_rights => 'site_view_field__data_for_site_stat',
            label        => d_gettext('Data for site statistics'),
            get          => sub {
                $_[0]->{'__OWNER_SITE__'}->{$_[1]->{'id'}} // [];
            },
            type     => 'array',
            sub_type => 'owner_site',
            api      => TRUE,
        },
        public_id => {
            depends_on => ['id'],
            get        => sub {
                return $_[1]->{'id'};
            },
            adjust_type => 'str',
        },
        editable_fields => {
            forced_depends_on => [qw(id multistate)],
            get               => sub {
                return $_[0]->model->get_editable_fields($_[1]);
            },
            type        => 'complex',
            fix_type    => \&fix_type_for_complex,
            api         => 1,
            adjust_type => 'hash_int',
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $filters = $self->SUPER::get_structure_model_filter();

    $filters->{'fields'} = {
        %{$filters->{'fields'}},
        domain      => {type => 'domain',  value_type => 'text', label => d_gettext('Site')},
        is_graysite => {type => 'boolean', label      => d_gettext('Gray site')},
        login       => {
            type           => 'subfilter',
            model_accessor => 'owner_site',
            field          => 'id',
            fk_field       => 'domain_id',
            label          => d_gettext('Login'),
        },
        user_type => {type => 'alias', path => [qw(login user user_type)]},
        owner_id  => {type => 'alias', path => [qw( login user_id)]},
    };

    return $filters;
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    my @filter_data = ();

    if ($self->check_rights('view_search_filters__login')) {
        push @filter_data, {name => 'login.user.login', label => gettext('Login')};
    }

    if ($self->check_rights('view_search_filters__user_type')) {
        push @filter_data, {name => 'user_type', label => gettext('User type')};
    }

    {
        my @filter_fields = qw( id domain multistate );

        # Галка 2-й сорт  (#PI-7888)
        push @filter_fields, 'is_graysite' if $self->check_short_rights('view_field__is_graysite');

        my $model_fields = $self->get_db_filter_fields();
        push(
            @filter_data,
            map +{
                name  => $_,
                label => $model_fields->{$_}->{label}
            },
            @filter_fields
        );
    }

    return \@filter_data;
}

sub get_structure_multistates_graph {
    return {
        empty_name  => 'New',
        multistates => [
            [need_approve                     => d_pgettext('Site status', 'On moderation')],
            [rejected                         => d_pgettext('Site status', 'Rejected')],
            [approved                         => d_pgettext('Site status', 'Approved')],
            [context_on_site_campaign_blocked => d_pgettext('Site status', 'Context campaigns blocked')],
            [search_on_site_campaign_blocked  => d_pgettext('Site status', 'Search campaigns blocked')],
            [blocked                          => d_pgettext('Site status', 'Blocked')],
        ],
        actions           => {set_need_approve => d_pgettext('Site action', 'Set "need approve"'),},
        right_group       => [site             => d_gettext('Right to manage site')],
        right_name_prefix => 'site_',
        right_actions     => {
            add                                  => d_pgettext('Site action', 'Add'),
            reject                               => d_pgettext('Site action', 'Reject'),
            approve                              => d_pgettext('Site action', 'Approve'),
            return_to_moderation                 => d_pgettext('Site action', 'Return to moderation'),
            set_is_graysite                      => d_pgettext('Site action', 'Set "is_graysite"'),
            unset_is_graysite                    => d_pgettext('Site action', 'Unset "is_graysite"'),
            set_context_on_site_campaign_blocked => d_pgettext('Site action', 'Set "context blocked"'),
            set_search_on_site_campaign_blocked  => d_pgettext('Site action', 'Set "search blocked"'),
            block                                => d_pgettext('Site action', 'Block'),
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'set_need_approve',
                from      => '__EMPTY__',
                set_flags => ['need_approve'],
            },
            {
                action      => 'approve',
                from        => 'need_approve',
                set_flags   => ['approved'],
                reset_flags => ['need_approve'],
            },
            {
                action => 'set_is_graysite',    # Галка 2-й сорт  (#PI-7888)
                from   => 'not rejected',
            },
            {
                action => 'unset_is_graysite',    # Галка 2-й сорт  (#PI-7888)
                from   => 'not rejected',
            },
            {
                action      => 'return_to_moderation',
                from        => 'rejected',
                set_flags   => ['need_approve'],
                reset_flags => ['rejected'],
            },

            # Следущие 2 определения сделаны для того чтобы не было статуса
            # "Утвержден. Контекстные площадки заблокированы. Поисковые площадки заблокированы.",
            # а вместо него был бы статус "Отклоненный."
            {
                action      => 'block',
                from        => 'need_approve',
                set_flags   => ['blocked'],
                reset_flags => ['need_approve'],
            },
            {
                action      => 'block',
                from        => 'approved',
                set_flags   => ['blocked'],
                reset_flags => ['approved', 'context_on_site_campaign_blocked', 'search_on_site_campaign_blocked'],
            },
            {
                action      => 'reject',
                from        => 'need_approve',
                set_flags   => ['rejected'],
                reset_flags => ['need_approve'],
            },
            {
                action      => 'reject',
                from        => 'approved',
                set_flags   => ['rejected'],
                reset_flags => ['approved', 'context_on_site_campaign_blocked', 'search_on_site_campaign_blocked'],
            },
            {
                action => 'set_context_on_site_campaign_blocked',
                from   => 'approved and not search_on_site_campaign_blocked and not context_on_site_campaign_blocked',
                set_flags => ['context_on_site_campaign_blocked'],
            },
            {
                action      => 'set_context_on_site_campaign_blocked',
                from        => 'approved and search_on_site_campaign_blocked and not context_on_site_campaign_blocked',
                set_flags   => ['rejected'],
                reset_flags => ['approved', 'search_on_site_campaign_blocked'],
            },
            {
                action => 'set_search_on_site_campaign_blocked',
                from   => 'approved and not context_on_site_campaign_blocked and not search_on_site_campaign_blocked',
                set_flags => ['search_on_site_campaign_blocked'],
            },
            {
                action      => 'set_search_on_site_campaign_blocked',
                from        => 'approved and context_on_site_campaign_blocked and not search_on_site_campaign_blocked',
                set_flags   => ['rejected'],
                reset_flags => ['approved', 'context_on_site_campaign_blocked'],
            },
        ]
    };
}

sub related_models {
    my ($self) = @_;

    return {
        owner_site => {
            accessor => 'owner_site',
            filter   => sub {
                return {domain_id => array_uniq(map {$_->{'id'} // ()} @{$_[1]})};
            },
            key_fields => ['domain_id'],
            value_type => 'array',
        }
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    $self->SUPER::pre_process_fields($fields, $result);

    if ($fields->need('data_for_site_stat')) {
        foreach (
            @{
                $self->owner_site->get_all(
                    fields => [qw(domain_id user_id owner)],
                    filter => {domain_id => array_uniq(map {$_->{'id'}} @$result)},
                )
            }
          )
        {
            $fields->{'__OWNER_SITE__'}{$_->{'domain_id'}} //= [];
            push(@{$fields->{'__OWNER_SITE__'}{$_->{'domain_id'}}}, $_);
        }
    }
}

sub get_actions_depends {
    [qw(id multistate is_graysite)];
}

sub get_available_fields_depends {
    [qw(multistate)];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields       = map {$_ => TRUE} keys(%$model_fields);
    my $accessor     = $self->accessor();

    $self->delete_field_by_rights(
        \%fields,
        {
            # <Право или маска под право>            <Поля котрые оно закрывает>
            $accessor . '_view_field__%s' => [qw(data_for_site_stat is_graysite)],
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my %fields = map {$_ => TRUE} qw(domain);

    # TODO: нужно использовать $self->delete_field_by_rights()
    $fields{'login_owner'} = TRUE if $self->check_short_rights('add_other');
    $fields{'is_graysite'} = TRUE if $self->check_short_rights('view_field__is_graysite');

    return \%fields;
}

sub get_editable_fields {
    my ($self, $object) = @_;

    return {}
      unless $self->check_action($object, 'edit');

    return $self->collect_editable_fields($object);
}

sub collect_editable_fields {
    my ($self, $object) = @_;

    my %fields = ();

    return \%fields;
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_processing_before_validation($opts);
    $opts->{'domain'} = get_domain($opts->{'domain'});
}

sub hook_owner_processing {
    my ($self, $opts) = @_;

    my $user;
    if ($self->check_short_rights('add_other') && $opts->{'login_owner'}) {
        $user = $self->users->get_by_login($opts->{'login_owner'}, fields => ['id', 'client_id'])
          // throw Exception::Validation::BadArguments gettext('Unknown user');
    } else {
        $user = $self->get_option('cur_user', {});
    }
    $self->hook_stash->set('user', $user);
    # to make Common to call hook_save_fields_from_related_models
    # add a field from related table to $opts
    $opts->{user_id} = $user->{id};
}

sub hook_fields_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_validation($opts);

    my $tmp_rights = $self->app->add_tmp_rights('site_view_all');
    my $site       = $self->get_all(
        fields => [qw(id multistate is_graysite)],
        filter => {domain => $opts->{'domain'}}
    )->[0];
    undef($tmp_rights);

    # We can "create" a site many times and link it to different partners and pages.
    # But it's fields must have the same values, otherwise it's confusing for managers.
    # And the site (domain) should not be rejected or blocked.
    #
    if ($site) {
        if ($self->check_multistate_flag($site->{'multistate'}, 'blocked')) {
            throw Exception::Validation::BadArguments gettext('Site with domain "%s" is blocked', $opts->{'domain'});
        }

        if ($self->check_multistate_flag($site->{'multistate'}, 'rejected')) {
            if ($self->can_action_return_to_moderation($site)) {
                my $tmp = $self->app->add_tmp_rights('do_site_return_to_moderation');
                $self->do_action($site, 'return_to_moderation');
            } else {
                throw Exception::Validation::BadArguments gettext('Site with domain "%s" is blocked',
                    $opts->{'domain'});
            }
        }

        if (defined($opts->{'is_graysite'}) && $opts->{'is_graysite'} != $site->{'is_graysite'}) {
            throw Exception::Validation::BadArguments gettext(
                'Site with domain "%s" already exists with a different is_graysite value', $opts->{'domain'});
        }
        # If domain has subdomain, need check it.
        # If subdomain is rejected, but can be re-moderated
        # domain can be created
    } elsif ($opts->{domain} =~ /^.+?\.(.+\..+)$/) {
        my $parent_domains = get_parent_domains($opts->{domain});
        my $tmp_rights     = $self->app->add_tmp_rights('site_view_all');
        my $parent_sites   = $self->get_all(
            fields => [qw(domain moderation_reason multistate)],
            filter => [
                AND => [
                    ['domain'     => 'IN' => $parent_domains],
                    ['multistate' => '='  => $self->get_multistates_by_filter('rejected')],
                ],
            ],
        );
        undef($tmp_rights);

        if (scalar @$parent_sites) {
            $parent_sites = {map {$_->{domain} => $_} @$parent_sites};
            my $subdomain = (grep {exists($parent_sites->{$_})} @$parent_domains)[0];

            throw Exception::Validation::BadArguments gettext('Domain %s is rejected by moderation', $opts->{domain})
              unless ref($parent_sites->{$subdomain}->{moderation_reason}) eq 'ARRAY'
                  && $self->get_moderation_timeout($parent_sites->{$subdomain}->{moderation_reason}) > -1;
        }
    }

    $self->hook_stash->set('id', $site) if $site;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $opts->{is_graysite} //= 0;
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    # Если связки еще не сущетсвует, то создаем
    my $tmp_rights = $self->app->add_tmp_rights('owner_site_add');
    $self->owner_site->add(
        user_id   => $self->hook_stash->get('user')->{id},
        domain_id => $self->hook_stash->get('id')->{id}
    );
    undef($tmp_rights);
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    my $id = $self->hook_stash->get('id');
    $self->do_action($id, 'set_need_approve') if $self->check_action($id, 'set_need_approve');
}

sub on_action_approve {
    my ($self, $obj, %opts) = @_;

    my $tmp_rights = $self->app->add_tmp_rights(
        qw(
          context_on_site_campaign_view_all
          do_context_on_site_approve
          search_on_site_campaign_view_all
          do_search_on_site_approve
          context_on_site_campaign_edit
          search_on_site_campaign_edit
          site_view_field__data_for_site_stat
          users_edit_field__has_approved
          users_edit_field__has_approved_site
          )
    );

    $obj = $self->get($obj, fields => [qw(id multistate domain data_for_site_stat)],);

    my $context_owners = $self->context_on_site_campaign->approve_for_site($obj->{'id'});

    my $search_owners = $self->search_on_site_campaign->approve_for_site($obj->{'id'});

    foreach my $owner_site (@{$obj->{'data_for_site_stat'}}) {
        my $owner       = $owner_site->{'owner'};
        my $owners_data = $self->users->get_all(
            fields => [qw(multistate has_approved has_approved_site)],
            filter => {id => $owner->{'id'}, multistate => 'not blocked'},
        );

        # Если нашли НЕзаблокированного владельца то шлем ему письма
        if (@$owners_data) {
            my $owner_data = $owners_data->[0];

            if ($context_owners->{$owner->{'id'}} || $search_owners->{$owner->{'id'}}) {
                # Отправляем письма, если есть не удаленные пейджи

                $self->app->mail_notification->add_when_approved(
                    {
                        user_id      => $owner->{'id'},
                        full_name    => $owner->{'full_name'},
                        has_approved => $owner_data->{'has_approved'},
                        resource     => $obj->{'domain'},
                    }
                );

                if (!$owner_data->{has_approved_site}) {
                    $self->mail_notification->add_when_first_approved_site(
                        {user_id => $owner->{'id'}, resource => $obj->{'domain'}});
                } else {
                    $self->mail_notification->add_when_approved_site(
                        {user_id => $owner->{'id'}, resource => $obj->{'domain'}});
                }
            }

            if (!$owner_data->{has_approved_site}) {
                my $tmp_rights = $self->app->add_tmp_rights(
                    qw(users_edit_all users_edit_field__has_approved users_edit_field__has_approved_site));
                $self->users->do_action($owner->{'id'}, 'edit', has_approved => 1, has_approved_site => 1);
            }
        }
    }

    undef($tmp_rights);
}

sub on_action_block {
    my ($self, $obj, %opts) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights();

    $self->context_on_site_campaign->block_for_site($obj->{'id'});

    $self->search_on_site_campaign->block_for_site($obj->{'id'});

    my $object = $self->_get_object_fields($obj, [qw(domain id)]);
    $self->mail_notification->add_when_blocking_site($object);

    undef($tmp_rights);
}

sub can_action_reject {TRUE}

sub _set_reject {
    my ($self, $obj, %opts) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights();

    $obj = $self->get($obj, fields => [qw(id domain data_for_site_stat)]);

    my $context_owners = $self->context_on_site_campaign->reject_for_site($obj->{'id'}, %opts);

    my $search_owners = $self->search_on_site_campaign->reject_for_site($obj->{'id'}, %opts);

    foreach my $owner_site (@{$obj->{'data_for_site_stat'}}) {
        my $owner = $owner_site->{'owner'};
        if ($context_owners->{$owner->{'id'}} || $search_owners->{$owner->{'id'}}) {
            # Отправляем письмо, если есть не удаленные пейджи
            $self->send_letter_after_reject(
                user_id   => $owner->{'id'},
                full_name => $owner->{'full_name'},
                resource  => $obj->{'domain'},
                reason    => $opts{'moderation_reason'},
            );
        }
    }
    undef($tmp_rights);
}

sub _get_reject_mail_notification {return 'add_when_rejected_site'}

sub on_action_reject {
    _set_reject(@_);
}

sub on_action_return_to_moderation {
    my ($self, $obj, %opts) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights();

    $self->context_on_site_campaign->return_to_moderation_for_site($obj->{'id'});

    $self->search_on_site_campaign->return_to_moderation_for_site($obj->{'id'});

    undef($tmp_rights);
}

sub on_action_set_context_on_site_campaign_blocked {
    my ($self, $obj, %opts) = @_;

    _set_reject(@_) if ($self->check_multistate_flag($obj->{multistate}, 'search_on_site_campaign_blocked'));
}

sub can_action_set_is_graysite {
    my ($self, $obj) = @_;
    $obj = $self->_get_object_fields($obj, [qw(is_graysite)]);
    return !$obj->{is_graysite};
}

sub on_action_set_is_graysite {
    my ($self, $obj) = @_;
    $self->partner_db_table()->edit($obj, {is_graysite => 1});

    my $tmp_rights = $self->app->add_tmp_rights(qw(context_on_site_campaign_edit));

    $self->_set_need_update__context_on_site_campaign($obj->{'id'});

    return TRUE;
}

sub on_action_set_search_on_site_campaign_blocked {
    my ($self, $obj, %opts) = @_;

    _set_reject(@_) if ($self->check_multistate_flag($obj->{multistate}, 'context_on_site_campaign_blocked'));
}

sub can_action_unset_is_graysite {
    my ($self, $obj) = @_;
    return !$self->can_action_set_is_graysite($obj);
}

sub on_action_unset_is_graysite {
    my ($self, $obj, $val) = @_;
    $self->partner_db_table()->edit($obj, {is_graysite => 0});

    my $tmp_rights = $self->app->add_tmp_rights(qw(context_on_site_campaign_edit));

    $self->_set_need_update__context_on_site_campaign($obj->{'id'});

    return TRUE;
}

sub throw_error_by_action {
    my ($self, $object, $action) = @_;

    if ($action eq 'approve' && $self->check_multistate_flag($object->{'multistate'}, 'approved')) {
        throw Exception::Validation::BadArguments gettext('You cannot approve this site because the site approved');
    } elsif ($action eq 'block' && $self->check_multistate_flag($object->{'multistate'}, 'blocked')) {
        throw Exception::Validation::BadArguments gettext('You cannot block this site because it is already blocked');
    } elsif ($action eq 'reject' && $self->check_multistate_flag($object->{'multistate'}, 'rejected')) {
        throw Exception::Validation::BadArguments gettext('You cannot reject this site because it is already rejected');
    }

    return FALSE;
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_special($filter, 'id', 'owner_site', 'domain_id');

    return $filter;
}

# API

sub api_available_actions {
    return qw(return_to_moderation reject approve);
}

sub api_can_add {TRUE}

sub get_domain_for_adding {
    my ($self, $user_id) = @_;

    throw gettext('Expected "user_id"') unless $user_id;

    my $sites = $self->get_all(
        fields => [qw(id domain multistate moderation_reason)],
        filter => {owner_id => $user_id}
    );

    my %flags = (
        is_approved        => 'approved',
        is_context_blocked => 'context_on_site_campaign_blocked',
        is_search_blocked  => 'search_on_site_campaign_blocked',
        is_blocked         => 'blocked',
        is_rejected        => 'rejected',
    );

    foreach (@$sites) {
        for my $flag (keys %flags) {
            $_->{$flag} = $self->check_multistate_flag($_->{'multistate'}, $flags{$flag});
        }
    }

    return $sites;
}

sub get_domain_for_adding_by_login {
    my ($self, $login, $campaign_type) = @_;

    throw gettext('Expected "login"') unless $login;

    my $check_types_expr;

    if (defined($campaign_type)) {
        throw gettext('Expected "campaign_type" equal "context_on_site_campaign" or "search_on_site_campaign"')
          unless $campaign_type eq 'context_on_site_campaign' || $campaign_type eq 'search_on_site_campaign';
        $check_types_expr = " and not ${campaign_type}_blocked";
    }

    my $sites = $self->get_all(
        fields => [qw(id domain multistate)],
        filter => [
            'AND',
            [
                {multistate => 'not blocked' . (defined($check_types_expr) ? $check_types_expr : '')},
                ['login', 'MATCH', ['user', 'MATCH', ['login', '=', $login]]]
            ]
        ]
    );

    return $sites;
}

sub _set_need_update__context_on_site_campaign {
    my ($self, $domain_id) = @_;

    my $model = $self->context_on_site_campaign;

    my $data = $model->get_all(
        fields => [qw(id)],
        filter =>
          ['AND' => [{domain_id => $domain_id}, {multistate => $model->get_multistate_by_action('set_need_update')},]],
    );

    foreach my $obj (@$data) {
        $model->do_action($obj, 'set_need_update');
    }

    return TRUE;
}

sub is_rejected {
    my ($self, $domains) = @_;

    return scalar @{
        $self->partner_db->query->select(
            table  => $self->partner_db->site,
            fields => [qw(id)],
            filter => [
                AND => [
                    ['domain', 'IN', \$domains],
                    ['multistate' => 'IN' => \$self->app->site->get_multistates_by_filter('rejected')],
                ],
            ],
            limit => 1,
        )->get_all();
      };
}

TRUE;
