package Application::Model::Product::AN::Site::OwnerSite;

use qbit;
use Crypt::CBC;

use base qw(
  Application::Model::DBManager::Base
  Application::Model::ValidatableMixin
  Application::Model::DAC
  );

use Exception::Denied;
use Exception::Validator::Fields;
use Exception::Validation::BadArguments;

use PiConstants qw($SITE_PARTNER_ROLE_ID);

sub accessor      {'owner_site'}
sub db_table_name {'owner_site'}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        partner_db => 'Application::Model::PartnerDB',
        users      => 'Application::Model::Users',
        site       => 'Application::Model::Product::AN::Site',
        all_pages  => 'Application::Model::AllPages',
        assistants => 'Application::Model::Assistants',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;
    return [
        {
            name        => 'owner_site',
            description => d_gettext('Right to manage site'),
            rights      => {
                owner_site_view_all => d_gettext('Right to view all owners'),
                owner_site_add      => d_gettext('Right to add site to owner'),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        domain_id => {default => TRUE, db => TRUE, pk => TRUE, type => 'number'},
        site      => {
            depends_on => ['domain_id', 'sites.id', 'sites.domain'],
            get        => sub {
                $_[0]->{'sites'}{$_[1]->{'domain_id'}};
            },
        },
        user_id => {
            default    => TRUE,
            db         => TRUE,
            pk         => TRUE,
            type       => 'number',
            need_check => {
                check => sub {
                    my ($qv, $user_id) = @_;

                    my $roles = $qv->app->app->rbac->get_roles_by_user_id($user_id);
                    unless ($roles->{$SITE_PARTNER_ROLE_ID}) {
                        throw Exception::Validator::Fields gettext("Username must belong to the partner");
                    }

                    my $owner_site = $qv->app->get({domain_id => $qv->data->{domain_id}, user_id => $user_id},
                        fields => ['user_id', 'site'],);
                    if ($owner_site) {
                        throw Exception::Validator::Fields gettext('You have site with this domain "%s"',
                            $owner_site->{'site'}->{'domain'});
                    }
                },
            },
        },
        owner => {
            label      => d_gettext('Owners'),
            depends_on => ['user_id'],
            get        => sub {
                $_[0]->{'__OWNERS__'}->{$_[1]->{'user_id'}} // {};
            },
            type => 'users',
        },
        editable_fields => {
            depends_on => ['domain_id'],
            get        => sub {
                my %res = ();

                $res{'pass'} = TRUE
                  if $_[0]->model->check_rights('owner_site_edit_field__pass');

                return \%res;
              }
        },
    };
}

sub related_models {
    my ($self) = @_;

    return {
        sites => {
            accessor => 'site',
            filter   => sub {
                +{id => array_uniq(map {$_->{'domain_id'} // ()} @{$_[1]})};
            },
            key_fields => ['id'],
        }
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        db_accessor => 'partner_db',
        fields      => {
            domain_id => {type => 'number', label => d_gettext('Domain ID')},
            user_id   => {type => 'number', label => d_gettext('Owner ID')},
            user      => {
                type           => 'subfilter',
                model_accessor => 'users',
                field          => 'user_id',
                label          => d_gettext('User'),
            },
        },
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('owner')) {
        $fields->{'__OWNERS__'} = {
            map {$_->{'id'} => $_} @{
                $self->users->get_all(
                    fields => [qw(login id full_name client_id)],
                    filter => {id => array_uniq(map {$_->{'user_id'}} @$result)}
                )
              }
        };
    }
}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied unless $self->check_short_rights('add');

    $self->app->validator->check(
        data  => \%opts,
        app   => $self,
        throw => TRUE,
        $self->get_template(),
    );

    $self->partner_db_table()->add(
        {
            create_date => curdate(oformat => 'db_time'),
            hash_transform(\%opts, [qw(domain_id user_id)]),
        }
    );
}

sub query_filter {
    my ($self, $filter) = @_;

    my $filter_acc = $self->partner_db->filter();

    if ($self->app->get_option('cur_user', {})->{'is_tutby_aggregator'}) {
        $filter_acc->or($self->limit_filter_tutby($self->partner_db->filter()));
    } elsif ($self->check_rights('is_assistant')) {
        $filter_acc->or($self->limit_filter_assistant($self->partner_db->filter()));
    } else {
        $filter_acc->or($self->limit_filter_by_user($self->partner_db->filter()));
    }
    return $filter->and($filter_acc);
}

TRUE;
