package Application::Model::Product::InternalAN::InternalContextOnSite::Campaign;

use qbit;

use base qw(
  Application::Model::Page::Site::Internal
  RestApi::MultistateModel
  );

consume qw(
  Application::Model::Role::Has::Context
  Application::Model::Role::Has::Tier
  Application::Model::Role::Page::Has::Tags
  Application::Model::Role::Page::Has::MobileAppMode
  Application::Model::Role::Page::Has::MetricaCounters
  Application::Model::Role::Page::Has::MoneyMap::Web
  Application::Model::Role::Page::Has::BlocksLimit
  );

use PiConstants qw(
  $CONTEXT_TARGET_TYPE
  $TECHNICAL_RTB_BLOCK_ID
  $DSP_DIRECT_ID
  $INTERNAL_PAGE_OWNER_LOGIN
  $SYSTEM_CRON_USER_ID
  );

use Exception::Denied;
use Exception::Validation::BadArguments;

sub accessor             {'internal_context_on_site_campaign'}
sub db_table_name        {'internal_context_on_site_campaign'}
sub get_opts_schema_name {'internal_context_on_site_campaign_opts'}
sub get_product_name     {gettext('internal_context_on_site_campaign')}

sub get_page_id_field_name {'id'}

sub get_block_model_names {
    return [map {"internal_context_on_site_$_"} qw(direct rtb stripe content natural)];
}

sub db_banner_lang {$_[0]->partner_db->internal_context_on_site_campaign_banner_lang()}

sub block_seq_db_table {$_[0]->partner_db->internal_context_on_site_block_seq}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        internal_context_on_site_stripe  => 'Application::Model::Product::InternalAN::InternalContextOnSite::Stripe',
        internal_context_on_site_direct  => 'Application::Model::Product::InternalAN::InternalContextOnSite::Direct',
        internal_context_on_site_rtb     => 'Application::Model::Product::InternalAN::InternalContextOnSite::RTB',
        internal_context_on_site_content => 'Application::Model::Product::InternalAN::InternalContextOnSite::Content',
        internal_context_on_site_natural => 'Application::Model::Product::InternalAN::InternalContextOnSite::Natural',

        internal_context_on_site_ya_categories =>
          'Application::Model::Product::InternalAN::InternalContextOnSite::Campaign::YaCategories',
        internal_context_on_site_mirrors =>
          'Application::Model::Product::InternalAN::InternalContextOnSite::Campaign::Mirrors',
    };
}

sub accessor_ya_categories {'internal_context_on_site_ya_categories'}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $fields = $self->SUPER::get_available_fields($obj);

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights($fields,
        {$accessor . '_view_field__%s' => [qw(ya_categories fast_context only_picture performance_tgo_disable)],});

    return $fields;
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my %res = (%{$self->SUPER::collect_editable_fields($data)}, %{$self->_get_common_add_edit_fields()});

    $res{$_} = TRUE foreach (
        qw(
        caption
        family_filter
        behavioral_targeting
        view_images
        reload_timeout
        excluded_domains
        excluded_phones
        mirrors
        page_lang
        )
    );

    foreach (
        qw(
        block_title
        fast_context
        only_picture
        banner_lang
        comment
        )
      )
    {
        $res{$_} = TRUE
          if $self->check_short_rights('edit_field__' . $_);
    }

    if ($data->{'is_read_only'}) {
        foreach (
            qw(
            block_title
            fast_context
            ),
          )
        {
            delete($res{$_});
        }
    }

    if ($self->check_multistate_flag($data->{'multistate'}, 'protected')
        && !$self->check_rights('edit_protected_pages'))
    {
        foreach (qw(mirrors), keys(%{$self->get_need_update_in_bk_fields()})) {
            delete($res{$_});
        }
    }

    return \%res;
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        res_fields   => $self->SUPER::_get_common_add_edit_fields(),
        right_fields => {edit => ['performance_tgo_disable']}
    );

    return $fields;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my $fields = $self->SUPER::get_need_update_in_bk_fields();

    $fields->{$_} = TRUE foreach (
        qw(
        behavioral_targeting
        fast_context
        only_picture
        view_images
        performance_tgo_disable
        )
    );

    return $fields;
}

sub get_bk_data {
    my ($self, $page) = @_;

    my %data = $self->SUPER::get_bk_data($page);

    #PI-10512 set redir-to-page-handler
    my %custom_opts = (
        enable  => [qw(adblock redir-to-page-handler)],
        disable => [],
    );
    if ($page->{'performance_tgo_disable'}) {
        push @{$custom_opts{'enable'}}, 'performance-tgo-disable';
    } else {
        push @{$custom_opts{'disable'}}, "performance-tgo-disable";
    }

    %data = (
        %data,
        target_type => $CONTEXT_TARGET_TYPE,
        options     => {
            dontshowbehavior => $page->{'behavioral_targeting'} ? 0 : 1,
            dontshowsex      => $page->{'family_filter'},
            ReloadTimeout    => $page->{'reload_timeout'},
            BlockTitle       => $page->{'block_title'},
        },
        page_options => $self->page_options->get_options($page->{'page_id'}, %custom_opts)
    );

    return %data;
}

sub api_available_actions {
    return qw(
      del_mirror
      delete
      edit
      reset_read_only
      restore
      set_read_only
      start
      start_testing
      stop
      );
}

sub api_can_edit {TRUE}

sub api_can_add {TRUE}

sub get_default_settings {
    return {
        fast_context    => 1,
        only_picture    => 0,
        mobile_app_mode => 0,
    };
}

sub get_next_id {
    my ($self, $id) = @_;

    $id++ if $id == $TECHNICAL_RTB_BLOCK_ID;

    return $id;
}

sub import_from_bk {
    my ($self, $id) = @_;

    throw Exception::Denied unless $self->check_rights($self->get_rights_by_actions('add'));

    throw Exception::Validation::BadArguments gettext('Must specify Page ID') unless defined $id;

    $id = trim($id);
    throw Exception::Validation::BadArguments gettext('Incorrect Page ID: %s', $id) unless $id =~ /^[0-9]+\z/;

    my $opts = $self->api_yt_bk->get_settings_for_page($id);

    throw Exception::Validation::BadArguments gettext('No such id: %s', $id) unless %$opts;

    throw Exception::Validation::BadArguments gettext('Its not a context campaign')
      if !defined($opts->{'TargetType'}) || $opts->{'TargetType'} != $CONTEXT_TARGET_TYPE;

    my $domain = $self->get_and_check_domain($opts->{'Name'} // '')
      // throw Exception::Validation::BadArguments gettext('Invalid domain: %s', $opts->{'Name'} // 'undef');

    my $status2state = {
        1    => 'working',
        0    => 'testing',
        '-1' => 'deleted',
    };
    throw Exception::Validation::BadArguments gettext('Invalid state: %s', $opts->{'State'} // 'undef')
      if !defined($opts->{'State'}) || !exists($status2state->{$opts->{'State'}});

    my $state = $status2state->{$opts->{'State'}};

    my $busy_domain = $self->internal_site->get_all(filter => {domain => $domain});

    my $site_id;
    if (@$busy_domain) {
        $site_id = $busy_domain->[0]{'id'};
    } else {
        $site_id = $self->internal_site->add($domain);
    }

    try {
        $self->partner_db->transaction(
            sub {
                $self->partner_db->internal_context_on_site_campaign->add(
                    {
                        id                   => $id,
                        site_id              => $site_id,
                        create_date          => curdate(oformat => 'db_time'),
                        creator_id           => $self->get_option('cur_user', {})->{'id'} // $SYSTEM_CRON_USER_ID,
                        family_filter        => $opts->{'OptionsFamily'} ? 1 : 0,
                        behavioral_targeting => $opts->{'OptionsNobehav'} ? 0 : 1,
                        view_images          => $opts->{'OptionsDirectpic'} ? 1 : 0,
                        fast_context         => $opts->{'OptionsNocontext'} ? 0 : 1,
                        only_picture         => $opts->{'OptionsOnlyPicture'} ? 1 : 0,
                        mobile_app_mode => $opts->{'OptionsOnlyMobileApp'} ? 2
                        : $opts->{'OptionsNoMobileApp'} ? 1
                        : 0,
                        opts => '{}',
                    }
                );

                my $mirrors =
                  [map {{campaign_id => $id, domain => $_}} @{array_uniq([split(/,/, $opts->{'DomainList'} // '')])}];
                $self->partner_db->internal_context_on_site_mirrors->add_multi($mirrors);

                $self->excluded_domains->replace($id, array_uniq([split(/,/, $opts->{'DomainFilter'} // '')]), 'add');
            }
        );
    }
    catch {
        throw Exception::Validation::BadArguments gettext('Already imported');
    };

    my $tmp_rights = $self->app->add_tmp_rights(
        'do_internal_context_on_site_set_read_only',
        'do_internal_context_on_site_set_protected',
        "do_internal_context_on_site_bk_import_$state"
    );

    $self->do_action($id, "set_read_only") if $opts->{'ReadOnly'};
    $self->do_action($id, "set_protected") if $opts->{'OptionsProtected'};
    $self->do_action($id, "bk_import_$state");

    return $id;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    my %page_options = (Enable => [], Disable => []);
    $opts->{'performance_tgo_disable'} //= 0;
    if ($opts->{'performance_tgo_disable'}) {
        push @{$page_options{'Enable'}}, 'performance-tgo-disable';
    } else {
        push @{$page_options{'Disable'}}, 'performance-tgo-disable';
    }
    $self->hook_stash->set('page_options', \%page_options);

    $self->SUPER::hook_set_initialize_settings($opts);
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    if ($self->hook_stash->mode('add')) {
        $opts->{allowed_turbo}           //= 1;
        $opts->{performance_tgo_disable} //= 0;
    }
    $self->SUPER::hook_fields_processing_before_validation($opts);
}

TRUE;
