package Application::Model::Product::InternalAN::InternalContextOnSite::Direct;

use qbit;

use base qw(
  Application::Model::Block::Internal::Site
  Application::Model::Block::MIXIN::Direct
  Application::Model::ValidatableMixin
  );

consume qw(
  Application::Model::Role::Has::CustomBkData
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::Block::DesignSettings
  Application::Model::Role::Block::Has::BlocksLimit
  );

use PiVariables qw(IGNORE_RTB_ON_DIRECT_SWITHER);
use PiConstants qw($DIRECT_PLACE_ID);

use Application::Model::Product::InternalAN::InternalContextOnSite::BlockTypes::Direct qw(
  direct_block_types
  @FIELDS
  @COLOR_FIELDS
  direct_appearance_fields
  %DIRECT_BLOCKS
  );

use Exception::Denied;
use Exception::Validation::BadArguments;

sub accessor      {'internal_context_on_site_direct'}
sub db_table_name {'internal_context_on_site_direct'}

sub get_campaign_model_name {
    return 'internal_context_on_site_campaign';
}

sub get_product_name {gettext('internal_context_on_site_direct')}

sub get_page_id_field_name {'campaign_id'}

sub public_id_prefix {'D-I-'}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            name        => 'internal_context_on_site_direct',
            description => d_gettext('Right to manage Direct blocks on internal sites'),
            rights      => {
                internal_context_on_site_direct_view_all =>
                  d_gettext('Right to view all Direct blocks on internal sites'),
                internal_context_on_site_direct_view_action_log =>
                  d_gettext('Right to view Direct blocks action logs on internal sites'),
                internal_context_on_site_direct_view_zero_block =>
                  d_gettext('Right to view zero internal context Direct blocks'),
                internal_context_on_site_direct_view =>
                  d_gettext('Right to view Direct blocks on internal sites in menu'),
                internal_context_on_site_direct_view_field__comment =>
                  d_gettext('Right to view field comment of direct blocks on internal context campaign'),
                internal_context_on_site_direct_edit => d_gettext('Right to edit Direct block to internal site'),
                internal_context_on_site_direct_edit_all =>
                  d_gettext('Right to edit all Direct block to internal site'),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        %{$self->get_direct_model_fields()},
        alternative_code     => {db => TRUE, label      => d_gettext('Alternative code')},
        images_first         => {db => TRUE, need_check => {type => 'boolean'}},
        url_background_color => {db => TRUE, type       => 'string', need_check => {type => 'color'}, api => 1},
        adaptive_width =>
          {db => TRUE, label => d_gettext('Adaptive block. width'), need_check => {type => 'adaptive_width'}},
        adaptive_height => {
            db         => TRUE,
            type       => 'string',
            label      => d_gettext('Height'),
            need_check => {type => 'adaptive_height'}
        },
        block_caption => {
            depends_on => ['type', 'limit'],
            get        => sub {
                throw gettext('Block with type "%s" not found', $_[1]->{'type'})
                  unless $_[0]->{'BLOCK_TYPES_CAPTION'}{$_[1]->{'type'}};

                my $limit =
                  exists($DIRECT_BLOCKS{$_[1]->{'type'}}->{'banners'})
                  ? $DIRECT_BLOCKS{$_[1]->{'type'}}->{'banners'}
                  : $_[1]->{'limit'};

                $_[0]->{'BLOCK_TYPES_CAPTION'}{$_[1]->{'type'}}($limit);
              }
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [
            {name => 'managers.manager.login', label => gettext('Manager\'s login')},
            {name => 'id',                     label => gettext('Block ID')},
            {name => 'campaign_id',            label => gettext('Page ID')},
        ],
        [
            {name => 'caption',             label => gettext('Block\'s caption')},
            {name => 'campaign.all_domain', label => gettext('Domain and mirror')},
            {name => 'multistate',          label => gettext('Status')},
        ],
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_pgettext('Block multistate', 'New'),
        multistates => [
            [deleted           => d_pgettext('Block multistate', 'Archived')],
            [working           => d_pgettext('Block multistate', 'Working')],
            [check_statistics  => d_pgettext('Block multistate', 'Check statistics')],
            [deleted_with_page => d_pgettext('Block multistate', 'Archived with page'), private => TRUE],
            [need_update       => d_pgettext('Block multistate', 'Need update')],
            [updating          => d_pgettext('Block multistate', 'Updating')],
        ],
        actions => {
            edit                   => d_pgettext('Block action', 'Edit'),
            delete                 => d_pgettext('Block action', 'Archive'),
            delete_with_page       => d_pgettext('Block action', 'Archive with page'),
            restore                => d_pgettext('Block action', 'Restore'),
            restore_with_page      => d_pgettext('Block action', 'Restore with page'),
            start                  => d_pgettext('Block action', 'Start'),
            stop                   => d_pgettext('Block action', 'Stop'),
            set_check_statistics   => d_pgettext('Block action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Block action', 'Reset "check_statistics"'),
        },
        right_name_prefix => $self->accessor . '_',
        right_actions     => {
            add => d_pgettext('Block action', 'Add'),
            set_need_update =>
              {label => d_pgettext('Block action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Block action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Block action', 'Stop update'),  dont_write_to_action_log => TRUE},
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action    => 'delete_with_page',
                from      => 'not deleted',
                set_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page',
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action    => 'start',
                from      => 'not working and not deleted',
                set_flags => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
            },
            {
                action    => 'set_check_statistics',
                from      => 'not deleted',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('block_caption')) {
        $fields->{'BLOCK_TYPES_CAPTION'} =
          {map {$_->{'id'} => $_->{'constuctor_caption'}} @{$self->get_types_direct_blocks}};
    }
}

sub get_available_fields_depends {
    [qw(multistate)];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor = $self->accessor();
    $self->app->delete_field_by_rights(\%fields, {$accessor . '_view_field__%s' => 'comment',});

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    $fields->{'campaign_id'} = TRUE;
    $fields->{'comment'} = TRUE if $self->check_short_rights('view_field__comment');

    return $fields;
}

sub get_actions_depends {
    [qw(id campaign_id page multistate is_my is_protected)];
}

sub get_editable_fields_depends {
    [qw(page_id campaign_id id multistate is_my is_protected)];
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my %res = %{$self->_get_common_add_edit_fields()};

    $res{'type'} = $self->get_types_direct_blocks() if $res{'type'};
    foreach my $block (@{$res{'type'}}) {
        delete($block->{'constuctor_caption'});
        $block->{'caption'} = $block->{'caption'}->();
    }

    $res{'alternative_code'} = TRUE;
    $res{'comment'}          = TRUE
      if ($self->check_short_rights('view_field__comment'));

    return \%res;
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        no_right_fields => ['caption', $self->direct_block_fields(), $self->direct_appearance_fields(),],);

    if ($fields->{'type'}) {
        my $blocks = $self->get_types_direct_blocks();
        foreach my $b (@$blocks) {
            delete($b->{'constuctor_caption'});
            $b->{'caption'} = $b->{'caption'}->();
        }
        $fields->{'type'} = $blocks;
    }

    return $fields;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $self->partner_db->transaction(
        sub {
            $opts->{id} = $self->page->get_next_block_id($opts->{$self->get_page_id_field_name()});
        }
    );
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    my $object = $self->hook_stash->get('id');

    $self->update_in_bk($object) if $self->need_update_in_bk([keys(%$opts)]);
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_processing_before_validation($opts);
    $self->set_block_default_values($opts);
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    my $block_id = $self->hook_stash->get('id');

    $self->maybe_do_action($block_id, 'start');

    $opts->{'page_id'} //= delete($opts->{$self->get_page_id_field_name()});
    $self->update_in_bk($block_id) if $self->need_update_in_bk([keys(%$opts)]);
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_campaign($filter);
    $filter = $self->limit_filter_view_zero_block($filter);
    $filter = $self->limit_filter_active($filter);

    return $filter;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = map {$_ => TRUE} $self->direct_block_fields, $self->direct_appearance_fields;

    return \%fields_need_update;
}

sub boolean_fields {
    return {map {$_ => TRUE} qw(border_radius no_sitelinks favicon links_underline images_first)};
}

sub check_type_direct_block {
    my ($self, $type) = @_;

    return $DIRECT_BLOCKS{$type};
}

sub direct_block_fields {
    return qw(
      limit type adaptive_width adaptive_height
      );
}

sub direct_blocks {
    return \%DIRECT_BLOCKS;
}

sub get_bk_block_data {
    my ($self, $block) = @_;

    return {};
}

sub get_bk_block_data_key {'direct_blocks'}

sub get_bk_common_data {
    return (places => {$DIRECT_PLACE_ID => {}});
}

sub get_cnt {
    my ($self, $arr_campaign_id) = @_;

    my $data = $self->partner_db->query->select(
        table  => $self->partner_db_table(),
        fields => {
            campaign_id => '',
            cnt         => {count => ['id']}
        },
        filter => [
            AND => [
                [campaign_id => 'IN'     => \$arr_campaign_id],
                [id          => '<>'     => \0],
                [multistate  => 'NOT IN' => \$self->get_multistates_by_filter('deleted')],
                IGNORE_RTB_ON_DIRECT_SWITHER()
                ? ()
                : [active => '=' => \1]
            ]
        ],
    )->group_by('campaign_id')->get_all();

    return {map {$_->{'campaign_id'} => $_->{'cnt'}} @$data};
}

sub get_types_direct_blocks {
    my ($self) = @_;

    return $self->direct_block_types();
}

sub design_field {'Design'}

TRUE;
