package Application::Model::Product::InternalAN::InternalSearchOnSite::Campaign;

use qbit;

use base qw(
  Application::Model::Page::Site::Internal
  RestApi::MultistateModel
  );

use Exception::Denied;
use Exception::Validation::BadArguments;

use PiConstants qw(
  $BIG_SEARCH_TARGET_TYPE
  $SEARCH_TARGET_TYPE
  $INTERNAL_PAGE_OWNER_LOGIN
  $SYSTEM_CRON_USER_ID
  );

consume qw(
  Application::Model::Role::Has::Search
  Application::Model::Role::Has::Tier
  Application::Model::Role::Page::Has::MobileAppMode
  Application::Model::Role::Page::Has::MetricaCounters
  Application::Model::Role::Page::Has::MoneyMap::Web
  Application::Model::Role::Page::Has::BlocksLimit
  );

sub accessor             {'internal_search_on_site_campaign'}
sub db_table_name        {'internal_search_on_site_campaign'}
sub get_opts_schema_name {'internal_search_on_site_campaign_opts'}
sub get_product_name     {gettext('internal_search_on_site_campaign')}

sub get_page_id_field_name {'id'}

sub get_block_model_names {
    return [map {"internal_search_on_site_$_"} qw(premium direct)];
}

sub db_banner_lang {$_[0]->partner_db->internal_search_on_site_campaign_banner_lang()}

sub block_seq_db_table {$_[0]->partner_db->internal_search_on_site_block_seq}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        internal_search_on_site_direct  => 'Application::Model::Product::InternalAN::InternalSearchOnSite::Direct',
        internal_search_on_site_premium => 'Application::Model::Product::InternalAN::InternalSearchOnSite::Premium',

        internal_search_on_site_mirrors =>
          'Application::Model::Product::InternalAN::InternalSearchOnSite::Campaign::Mirrors',
    };
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my $res = {%{$self->SUPER::collect_editable_fields($data)}, %{$self->_get_common_add_edit_fields()}};

    $res->{$_} = TRUE foreach (
        qw(
        caption
        family_filter
        reload_timeout
        excluded_domains
        excluded_phones
        mirrors
        page_lang
        )
    );

    foreach (
        qw(
        banner_lang
        )
      )
    {
        $res->{$_} = TRUE
          if $self->check_short_rights('edit_field__' . $_);
    }

    if ($data->{'is_read_only'}) {
        foreach (
            qw(
            block_title
            ),
          )
        {
            delete($res->{$_});
        }
    }

    if ($self->check_multistate_flag($data->{'multistate'}, 'protected')
        && !$self->check_rights('edit_protected_pages'))
    {
        foreach (qw(mirrors), keys(%{$self->get_need_update_in_bk_fields()})) {
            delete($res->{$_});
        }
    }

    return $res;
}

sub get_bk_data {
    my ($self, $page) = @_;

    my %data = $self->SUPER::get_bk_data($page);

    # Хак для того чтобы можно было работать с площадками у которых в БК TargetType = 0
    my $target_type =
      in_array($page->{'page_id'}, [qw(114 131 136 3897 158973)]) ? $BIG_SEARCH_TARGET_TYPE : $SEARCH_TARGET_TYPE;

    %data = (
        %data,
        target_type => $target_type,
        options     => {
            dontshowsex   => $page->{'family_filter'},
            ReloadTimeout => $page->{'reload_timeout'},
            BlockTitle    => $page->{'block_title'},
        },
        pictures_enabled => 0,
        dsp_info         => [],
        page_options     => $self->page_options->get_options($page->{'id'}),
    );

    return %data;
}

sub get_default_settings {
    return {mobile_app_mode => 0,};
}

sub import_from_bk {
    my ($self, $id) = @_;

    throw Exception::Denied unless $self->check_rights($self->get_rights_by_actions('add'));

    throw Exception::Validation::BadArguments gettext('Must specify Page ID') unless defined $id;

    $id = trim($id);
    throw Exception::Validation::BadArguments gettext('Incorrect Page ID: %s', $id) unless $id =~ /^[0-9]+\z/;

    my $opts = $self->api_yt_bk->get_settings_for_page($id);

    throw Exception::Validation::BadArguments gettext('No such id: %s', $id) unless %$opts;

    throw Exception::Validation::BadArguments gettext('Its not a search campaign')
      if !defined($opts->{'TargetType'}) || $opts->{'TargetType'} != $SEARCH_TARGET_TYPE;

    my $domain = $self->get_and_check_domain($opts->{'Name'} // '')
      // throw Exception::Validation::BadArguments gettext('Invalid domain: %s', $opts->{'Name'} // 'undef');

    my $status2state = {
        1    => 'working',
        0    => 'testing',
        '-1' => 'deleted',
    };
    throw Exception::Validation::BadArguments gettext('Invalid state: %s', $opts->{'State'} // 'undef')
      if !defined($opts->{'State'}) || !exists($status2state->{$opts->{'State'}});

    my $state = $status2state->{$opts->{'State'}};

    my $busy_domain = $self->internal_site->get_all(filter => {domain => $domain});

    my $site_id;
    if (@$busy_domain) {
        $site_id = $busy_domain->[0]{'id'};
    } else {
        $site_id = $self->internal_site->add($domain);
    }

    my %languages =
      map {$_->{'language'} => $_->{'id'}} @{$self->partner_db->bk_language->get_all(fields => [qw(id language)])};

    my $lang = $opts->{'Lang'} // '';

    my $lang_id = $languages{$lang};

    throw Exception::Validation::BadArguments gettext('No such language: %s', $lang) unless $lang_id;

    try {
        $self->partner_db->transaction(
            sub {
                $self->partner_db->internal_search_on_site_campaign->add(
                    {
                        id            => $id,
                        site_id       => $site_id,
                        create_date   => curdate(oformat => 'db'),
                        creator_id    => $self->get_option('cur_user', {})->{'id'} // $SYSTEM_CRON_USER_ID,
                        family_filter => $opts->{'OptionsFamily'} ? 1 : 0,
                        mobile_app_mode => $opts->{'OptionsOnlyMobileApp'} ? 2
                        : $opts->{'OptionsNoMobileApp'} ? 1
                        : 0,
                    }
                );

                my $mirrors =
                  [map {{campaign_id => $id, domain => $_}} @{array_uniq([split(/,/, $opts->{'DomainList'} // '')])}];
                $self->partner_db->internal_search_on_site_mirrors->add_multi($mirrors);

                $self->excluded_domains->replace($id, array_uniq([split(/,/, $opts->{'DomainFilter'} // '')]), 'add');

                $self->partner_db->internal_search_on_site_campaign_banner_lang->add(
                    {
                        campaign_id => $id,
                        language_id => $lang_id,
                    }
                );
            }
        );
    }
    catch {
        throw Exception::Validation::BadArguments gettext('Already imported');
    };

    my $tmp_rights = $self->app->add_tmp_rights(
        'do_internal_search_on_site_set_read_only',
        'do_internal_search_on_site_set_protected',
        "do_internal_search_on_site_bk_import_$state"
    );

    $self->do_action($id, "set_read_only") if $opts->{'ReadOnly'};
    $self->do_action($id, "set_protected") if $opts->{'OptionsProtected'};

    $self->do_action($id, "bk_import_$state");

    return $id;
}

sub api_available_actions {
    return qw(
      del_mirror
      delete
      edit
      reset_read_only
      restore
      set_read_only
      start
      start_testing
      stop
      );
}

sub api_can_add {TRUE}

sub api_can_edit {TRUE}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    my %page_options = (Enable => [], Disable => []);
    $self->hook_stash->set('page_options', \%page_options);

    $self->SUPER::hook_set_initialize_settings($opts);
}

TRUE;
