package Application::Model::Product::InternalAN::InternalSearchOnSite::Premium;

use qbit;
use PiConstants qw($DIRECT_PLACE_ID);

use base qw(
  Application::Model::Block::Internal::Site
  Application::Model::Block::MIXIN::Direct
  Application::Model::ValidatableMixin
  );

consume qw(
  Application::Model::Role::Has::Block::DesignSettings
  Application::Model::Role::Block::Has::BlocksLimit
  );

use Application::Model::Product::InternalAN::InternalSearchOnSite::BlockTypes::Premium qw(
  direct_block_types
  @FIELDS
  @COLOR_FIELDS
  direct_appearance_fields
  %DIRECT_BLOCKS
  );

consume qw(
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Actions
  );

use Exception::Denied;
use Exception::Validation::BadArguments;

sub accessor      {'internal_search_on_site_premium'}
sub db_table_name {'internal_search_on_site_premium'}

sub get_campaign_model_name {
    return 'internal_search_on_site_campaign';
}

sub get_product_name {gettext('internal_search_on_site_premium')}

sub get_page_id_field_name {'campaign_id'}

sub public_id_prefix {'P-I-'}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            name        => 'internal_search_on_site_premium',
            description => d_gettext('Right to manage Premium blocks on internal search sites'),
            rights      => {
                internal_search_on_site_premium_view_all =>
                  d_gettext('Right to view all Premium blocks on internal search sites'),
                internal_search_on_site_premium_view_action_log =>
                  d_gettext('Right to view Premium blocks action logs on internal search sites'),
                internal_search_on_site_premium_view_zero_block =>
                  d_gettext('Right to view zero internal search Premium blocks'),
                internal_search_on_site_premium_view =>
                  d_gettext('Right to view Premium blocks on internal search sites in menu'),
                internal_search_on_site_premium_view_field__comment =>
                  d_gettext('Right to view field comment of premium blocks on internal search campaigns'),
                internal_search_on_site_premium_edit =>
                  d_gettext('Right to edit Premium block to internal search site'),
                internal_search_on_site_premium_edit_all =>
                  d_gettext('Right to edit all Premium block to internal search site'),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        %{$self->get_direct_model_fields()},
        adaptive_height => {
            db         => TRUE,
            type       => 'string',
            label      => d_gettext('Height'),
            need_check => {type => 'adaptive_height'}
        },
        block_caption => {
            depends_on => ['type', 'limit'],
            get        => sub {
                throw gettext('Block with type "%s" not found', $_[1]->{'type'})
                  unless $_[0]->{'BLOCK_TYPES_CAPTION'}{$_[1]->{'type'}};

                $_[0]->{'BLOCK_TYPES_CAPTION'}{$_[1]->{'type'}}($_[1]->{'limit'});
              }
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [
            {name => 'managers.manager.login', label => gettext('Manager\'s login')},
            {name => 'id',                     label => gettext('Block ID')},
            {name => 'campaign_id',            label => gettext('Page ID')},
        ],
        [
            {name => 'caption',             label => gettext('Block\'s caption')},
            {name => 'campaign.all_domain', label => gettext('Domain and mirror')},
            {name => 'multistate',          label => gettext('Status')},
        ],
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_pgettext('Block multistate', 'New'),
        multistates => [
            [deleted           => d_pgettext('Block multistate', 'Archived')],
            [working           => d_pgettext('Block multistate', 'Working')],
            [check_statistics  => d_pgettext('Block multistate', 'Check statistics')],
            [deleted_with_page => d_pgettext('Block multistate', 'Archived with page'), private => TRUE],
            [need_update       => d_pgettext('Block multistate', 'Need update')],
            [updating          => d_pgettext('Block multistate', 'Updating')],
        ],
        actions => {
            edit                   => d_pgettext('Block action', 'Edit'),
            delete                 => d_pgettext('Block action', 'Archive'),
            delete_with_page       => d_pgettext('Block action', 'Archive with page'),
            restore                => d_pgettext('Block action', 'Restore'),
            restore_with_page      => d_pgettext('Block action', 'Restore with page'),
            start                  => d_pgettext('Block action', 'Start'),
            stop                   => d_pgettext('Block action', 'Stop'),
            set_check_statistics   => d_pgettext('Block action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Block action', 'Reset "check_statistics"'),
        },
        right_name_prefix => $self->accessor . '_',
        right_actions     => {
            add => d_pgettext('Block action', 'Add'),
            set_need_update =>
              {label => d_pgettext('Block action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Block action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Block action', 'Stop update'),  dont_write_to_action_log => TRUE},
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action    => 'delete_with_page',
                from      => 'not deleted',
                set_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page',
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action    => 'start',
                from      => 'not working and not deleted',
                set_flags => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
            },
            {
                action    => 'set_check_statistics',
                from      => 'not deleted',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('block_caption')) {
        $fields->{'BLOCK_TYPES_CAPTION'} =
          {map {$_->{'id'} => $_->{'constuctor_caption'}} @{$self->direct_block_types()}};
    }
}

sub get_actions_depends {
    [qw(id campaign_id page multistate is_my is_protected)];
}

sub get_available_fields_depends {
    [qw(multistate)];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor      = $self->accessor();
    my $page_accessor = $self->get_campaign_model_name();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s' => 'comment',

            $page_accessor . '_view_field__managers' => 'manager_ids',
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my %fields = map {$_ => TRUE} qw(campaign_id caption), $self->direct_block_fields(),
      $self->direct_appearance_fields();

    return \%fields;
}

sub get_editable_fields_depends {
    [qw(page_id campaign_id id multistate is_my is_protected)];
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    return {} unless $self->check_action($_[1], 'edit');

    my %res =
      map {$_ => TRUE} (qw(caption type limit), direct_appearance_fields());

    $res{'type'} = $self->get_types_premium_blocks() if $res{'type'};

    $res{'comment'} = TRUE
      if $self->check_short_rights('view_field__comment');

    return \%res;
}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied unless $self->check_rights($self->get_rights_by_actions('add'));

    $self->_trim_opts(\%opts);

    throw Exception::Validation::BadArguments gettext('Invalid type block')
      unless $self->check_type_premium_block($opts{'type'});

    my $allowed_fields = $self->get_add_fields();

    $self->check_bad_fields_in_add(\%opts, $allowed_fields);

    $self->set_block_default_values(\%opts);

    $self->app->validator->check(
        data  => \%opts,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => $allowed_fields),
    );

    my $block_id;
    $self->partner_db->transaction(
        sub {
            my $id = $self->internal_search_on_site_campaign->get_next_block_id($opts{'campaign_id'});

            $block_id = $self->partner_db_table()->add(
                {
                    id => $id,
                    map {$_ => $opts{$_}} qw(campaign_id caption), $self->direct_block_fields(),
                    $self->direct_appearance_fields
                }
            );

            $self->do_action($block_id, 'add', %opts);
            $self->maybe_do_action($block_id, 'start');

            $self->update_in_bk($block_id);
        }
    );

    return $block_id;
}

sub on_action_edit {
    my ($self, $block, %data) = @_;

    $self->_trim_opts(\%data);

    $self->check_bad_fields($block, \%data);

    my $model_fields = $self->get_model_fields();

    my @need_check_fields = grep {$model_fields->{$_}->{'need_check'}} keys(%$model_fields);

    my $tmp_all_rights = $self->app->add_all_tmp_rights();

    my $old_block_settings = $self->get($block, fields => \@need_check_fields);

    undef($tmp_all_rights);

    my $new_block_settings = $self->set_block_default_values(\%data, $old_block_settings);

    $self->app->validator->check(
        data  => $new_block_settings,
        app   => $self,
        throw => TRUE,
        $self->get_template(),
    );

    if (%data) {
        $self->partner_db_table()->edit($block, \%data);
        $self->update_in_bk($block) if $self->need_update_in_bk([keys(%data)]);
    }

    return TRUE;
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_campaign($filter);
    $filter = $self->limit_filter_view_zero_block($filter);

    return $filter;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = map {$_ => TRUE} direct_block_fields(), direct_appearance_fields();

    return \%fields_need_update;
}

sub boolean_fields {
    return {map {$_ => TRUE} qw(border_radius no_sitelinks favicon links_underline)};
}

sub check_type_premium_block {
    my ($self, $type) = @_;

    return $DIRECT_BLOCKS{$type};
}

sub direct_block_fields {
    return qw(type limit);
}

sub direct_blocks {
    return \%DIRECT_BLOCKS;
}

sub get_bk_block_data {
    my ($self, $block) = @_;

    return {};
}

sub get_bk_block_data_key {'direct_blocks'}

sub get_bk_common_data {
    return (places => {436 => {}, $DIRECT_PLACE_ID => {}});
}

sub get_types_direct_blocks {&get_types_premium_blocks}

sub get_types_premium_blocks {
    my ($self) = @_;

    my $block_types = $self->direct_block_types();

    foreach (@$block_types) {
        delete($_->{'constuctor_caption'});
        $_->{'caption'} = $_->{'caption'}->();
    }

    return $block_types;
}

sub _trim_opts {
    my ($self, $opts) = @_;

    foreach my $field_name (qw(caption)) {
        $opts->{$field_name} =~ s/^\s+|\s+$//g if $opts->{$field_name};
    }
}

sub design_field {'Design'}

TRUE;
