package Application::Model::Product::InternalAN::MobileApp;

use qbit;

use base qw(
  Application::Model::Page::Mobile
  Application::Model::Page::MIXIN::Internal
  RestApi::MultistateModel
  );

use PiConstants qw(
  $ADINSIDE_CLIENT_ID
  $ADINSIDE_USER_ID
  $APP_TYPES
  $CONTEXT_TARGET_TYPE
  $INTERNAL_PAGE_OWNER_LOGIN
  @BK_LANGUAGES_BY_DEFAULT
  );

consume qw(
  Application::Model::Role::Page::Has::MoneyMap::SDK
  Application::Model::Role::Page::Has::BlocksLimit
  Application::Model::Role::Page::Has::IgnoreDomainCheck
  Application::Model::Role::Page::Has::AppMarketFields
  Application::Model::Role::Page::Has::PICategories
  );

use Exception::DB::DuplicateEntry;

use Utils::PhoneBK;
use Utils::DomainBK;

use Application::Model::Product::InternalAN::BlockTypes::MobileApp qw(
  @BLOCK_TYPES
  %BLOCK_TYPES
  );

use Exception::BK::Protected;
use Exception::Denied;
use Exception::Validation::BadArguments;

sub accessor      {'internal_mobile_app'}
sub db_table_name {'internal_mobile_app'}

sub get_product_name {gettext('internal_mobile_app')}

sub get_page_id_field_name {'id'}

sub get_block_model_names {
    return [map {"internal_mobile_app_$_"} qw(rtb)];
}
sub block_seq_db_table {$_[0]->partner_db->internal_mobile_app_block_seq}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {internal_mobile_app_rtb => 'Application::Model::Product::InternalAN::MobileApp::RTB',};
}

sub is_mobile_mediation_from_generated_column {TRUE}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        %{$rights->[0]{'rights'}},
        %{$self->get_internal_structure_rights_to_register()},

        map {$self->get_description_right($_)} qw(
          view_field__bk_languages
          edit_field__bk_languages

          view_field__block_title
          edit_field__block_title

          view_field__comment
          view_field__only_picture
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        %{$self->get_internal_structure_model_fields()},

        multistate_name => {
            depends_on => ['multistate'],
            label      => d_gettext('Multistate name'),
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
        },
        store_id => {
            api        => TRUE,
            default    => TRUE,
            db         => TRUE,
            label      => d_gettext('Store ID'),
            type       => 'string',
            need_check => {
                check => sub {
                    my ($qv, $store_id) = @_;
                    throw Exception::Validator::Fields gettext('Invalid Bundle ID "%s"', $store_id)
                      unless $qv->app->check_mobile_app_bundle_id($store_id, $qv->data->{'type'});
                },
            },
        },
        domain => {db => TRUE, db_expr => 'store_id', label => d_gettext('Domain'), type => 'string'},
        type   => {
            default     => TRUE,
            db          => TRUE,
            label       => d_gettext('Type ID'),
            type        => 'string',
            need_check  => {in => [sort keys %{$APP_TYPES}],},
            adjust_type => 'str',
        },
        type_name => {
            depends_on => ['type'],
            label      => d_gettext('Type'),
            get        => sub {
                $_[0]->model->get_app_type_caption($_[1]->{'type'});
            },
            type => 'string',
            api  => 1
        },
        store => {
            depends_on => ['type'],
            label      => d_gettext('Store'),
            get        => sub {
                $_[0]->model->get_app_store($_[1]->{'type'});
            },
            type => 'string',
        },
    };
}

sub get_single_states {
    return (
        {
            id    => 'archived',
            label => d_gettext('single_state archived'),
            flags => ['deleted'],
        },
        {
            id    => 'active',
            label => d_gettext('single_state active'),
            flags => [qw'working testing'],
        },
        {
            id    => 'stopped',
            label => d_gettext('single_state stopped'),
            flags => ['stopped'],
        },
        {
            id    => 'testing',
            label => d_gettext('single_state testing'),
            flags => [],
        },
    );
}

sub init {
    my ($self) = @_;
    $self->SUPER::init();
    $self->_init_single_state_multistates();
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $filter = $self->SUPER::get_structure_model_filter();

    $filter->{'fields'} = {
        %{$filter->{'fields'}},
        %{$self->get_internal_structure_model_filter()},

        store_id => {type => 'text', label => d_gettext('Domain')},
        type     => {
            type   => 'dictionary',
            label  => d_gettext('Type'),
            values => [map {+{hash_transform($_, ['id'], {caption => 'label'})}} get_app_types()]
        },
        page_id => {
            type      => 'number',
            label     => d_gettext('Page ID'),
            db_filter => sub {
                return ['id' => $_[1]->[1] => \$_[1]->[2]];
            },
        },
        apple_store_id => {
            type       => 'json',
            value_type => 'text',
            label      => d_gettext('Apple Store ID'),
        },
        domain_text => {type => 'alias', path => [qw(store_id)]},
    };

    return $filter;
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [
            {name => 'managers.manager.login', label => gettext('Manager\'s login')},
            {name => 'page_id',                label => gettext('Page ID')},
            {name => 'store_id',               label => gettext('Store ID')},
        ],
        [
            {name => 'caption',    label => gettext('Caption')},
            {name => 'type',       label => gettext('Type')},
            {name => 'multistate', label => gettext('Status')},
        ]
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => 'New',
        multistates => [
            [working => d_pgettext('Application status', 'Working')],
            [
                testing    => d_pgettext('Application status', 'Testing'),
                private    => TRUE,
                deprecated => TRUE
            ],
            [stopped          => d_pgettext('Application status', 'Stopped')],
            [deleted          => d_pgettext('Application status', 'Archived')],
            [check_statistics => d_pgettext('Application status', 'Check statistics')],
            [protected        => d_pgettext('Application status', 'Protected')],
            [need_update      => d_pgettext('Application status', 'Need update')],
            [updating         => d_pgettext('Application status', 'Updating')],
        ],
        actions => {
            start                  => d_pgettext('Application action', 'Start'),
            stop                   => d_pgettext('Application action', 'Stop'),
            edit                   => d_pgettext('Application action', 'Edit'),
            set_check_statistics   => d_pgettext('Application action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Application action', 'Reset "check_statistics"'),
            start_block            => d_pgettext('Application action', 'Start block on campaign'),
            restore_block          => d_pgettext('Application action', 'Restore block on campaign'),
            can_update_in_bk       => d_pgettext('Application action', 'Can update in bk'),
        },
        right_actions => {
            add             => d_pgettext('Application action', 'Add'),
            delete          => d_pgettext('Application action', 'Archive'),
            restore         => d_pgettext('Application action', 'Restore'),
            set_protected   => d_pgettext('Application action', 'Set protected'),
            reset_protected => d_pgettext('Application action', 'Reset protected'),
            set_need_update =>
              {label => d_pgettext('Application action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update =>
              {label => d_pgettext('Application action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update => {label => d_pgettext('Application action', 'Stop update'), dont_write_to_action_log => TRUE},
        },
        right_group       => [internal_mobile_app => d_gettext('Right to manage internal mobile applications')],
        right_name_prefix => accessor()
          . '_',    # TODO: после рефакторинга тут должен быть $self->
        multistate_actions => [
            {
                action    => 'add',
                from      => '__EMPTY__',
                set_flags => ['stopped'],
            },
            {
                action      => 'start',
                from        => '(stopped) and not (deleted or protected)',
                set_flags   => ['working'],
                reset_flags => ['stopped']
            },
            {
                action      => 'stop',
                from        => 'working and not (stopped or protected)',
                set_flags   => ['stopped'],
                reset_flags => ['working'],
            },
            {
                action    => 'delete',
                from      => 'not (working or deleted)',
                set_flags => ['deleted']
            },
            {
                action      => 'restore',
                from        => 'deleted and not protected',
                reset_flags => ['deleted', 'stopped'],
                set_flags   => ['working'],
            },
            {
                action => 'edit',
                from   => 'not deleted'
            },
            {
                action    => 'set_check_statistics',
                from      => '__EMPTY__',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action => 'start_block',
                from   => 'working and not protected',
            },
            {
                action => 'restore_block',
                from   => 'not deleted',
            },
            {
                action => 'can_update_in_bk',
                from   => 'not (deleted or protected)',
            },
            {
                action    => 'set_protected',
                from      => 'not protected',
                set_flags => ['protected'],
            },
            {
                action      => 'reset_protected',
                from        => 'protected',
                reset_flags => ['protected'],
            },
            {
                action    => 'set_need_update',
                from      => 'not (deleted)',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update or updating',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ]
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    my $ids = array_uniq(map {$_->{'page_id'} // ()} @$result);

    $self->SUPER::pre_process_fields($fields, $result, ids => $ids, page_ids => $ids);
    Application::Model::Page::MIXIN::Internal::pre_process_fields(
        $self, $fields, $result,
        ids      => $ids,
        page_ids => $ids
    );

    if ($fields->need('bk_languages') || $fields->need('bk_languages_name')) {
        my %bk_languages =
          map {$_->{'id'} => $_->{'language'}} @{$self->bk_language->get_all(fields => [qw(id language)])};

        my @bk_languages_by_application_id = @{
            $self->partner_db->internal_mobile_app_bk_language->get_all(
                fields => [qw(application_id language_id)],
                filter => {application_id => $ids},
            )
          };

        $fields->{'bk_languages'}{$_->{'application_id'}}{$_->{'language_id'}} = $bk_languages{$_->{'language_id'}}
          foreach @bk_languages_by_application_id;

        %bk_languages = reverse(%bk_languages);

        foreach my $application_id (@$ids) {
            $fields->{'bk_languages'}{$application_id} ||=
              {map {$bk_languages{$_} => $_} grep {$bk_languages{$_}} @BK_LANGUAGES_BY_DEFAULT};
        }
    }
}

sub get_actions_depends {
    [qw(id multistate is_my)];
}

sub get_editable_fields_depends {
    [qw(id multistate is_my)];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->SUPER::get_available_fields($obj);
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor
              . '_view_field__%s' =>
              [qw( block_title fast_context only_picture view_images comment false_click performance_tgo_disable )],

            $accessor . '_view_field__bk_languages' => [qw( bk_langauges bk_languages_name )],
            $accessor . '_view_field__managers'     => [qw( managers manager_ids )],
        }
    );

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->SUPER::get_add_fields();

    $fields->{$_} = TRUE foreach (qw(type store_id store_url));

    foreach (qw(block_title)) {
        $fields->{$_} = TRUE if $self->check_short_rights("edit_field__$_");
    }

    return $fields;
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my %res = %{$self->SUPER::collect_editable_fields($data)};

    map {$res{$_} = 1} qw(
      caption
      family_filter
      behavioral_targeting
      excluded_domains
      excluded_phones
      );

    foreach (
        qw(
        block_title
        fast_context
        only_picture
        view_images
        comment
        bk_languages
        false_click
        performance_tgo_disable
        )
      )
    {
        $res{$_} = TRUE
          if $self->check_short_rights('edit_field__' . $_);
    }

    if ($self->check_multistate_flag($data->{'multistate'}, 'protected')
        && !$self->check_rights('edit_protected_pages'))
    {
        foreach (keys(%{$self->get_need_update_in_bk_fields()})) {
            delete($res{$_});
        }
    }

    return \%res;
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_managers_and_robot_assistant($filter);

    return $filter;
}

sub get_bk_data {
    my ($self, $page) = @_;

    my %data = $self->SUPER::get_bk_data($page);

    my @domains = check_domains_for_absorption($page->{'excluded_domains'});

    my %hash_excluded_domains = map {$_ => TRUE} @domains, bs_format_phone($page->{'excluded_phones'});

    %data = (%data, excluded_domains => [sort keys(%hash_excluded_domains)],);

    unless ($page->{store_id}) {
        # flag for BK/SDK to not show real ads and instead show mocks
        $data{MockMode} = 1;
        # https://st.yandex-team.ru/BSSERVER-19902#61a9c6883e3d400053772b83
        $data{domain} = sprintf('com.test.%d.%s', $page->{page_id}, lc($page->{type_name} // 'undef'));
    }

    return %data;
}

# API
sub api_can_add {TRUE}

sub api_available_actions {qw()}

sub api_can_edit {TRUE}

sub check_action {&Application::Model::Page::MIXIN::Internal::check_action}

sub do_action {
    my ($self, $object, $action, %opts) = @_;

    my $result;
    my $is_protected;
    try {
        $result = $self->SUPER::do_action($object, $action, %opts);
    }
    catch Exception::BK::Protected with {
        $is_protected = TRUE;
    };

    if ($is_protected) {
        my $tmp_rights =
          $self->app->add_tmp_rights('do_internal_mobile_app_set_protected', 'do_internal_mobile_app_reset_protected');

        $self->change_status_if_need($object, 'protected', $is_protected);

        throw Exception::Validation::BadArguments gettext(
            'The application is closed for editing in BK. Ask your manager.');
    }

    return $result;
}

sub get_app_store {
    my ($self, $id) = @_;

    return undef unless exists($APP_TYPES->{$id});

    return $APP_TYPES->{$id}{'store'};
}

sub get_app_type_caption {
    my ($self, $id) = @_;

    return undef unless exists($APP_TYPES->{$id});

    return $APP_TYPES->{$id}{'caption'}->();
}

sub get_app_types {
    return map {{id => $_->{'id'}, store => $_->{'store'}, caption => $_->{'caption'}->()}}
      sort {$a->{'id'} <=> $b->{'id'}} values(%$APP_TYPES);
}

sub get_block_types {
    my ($self, %opts) = @_;

    return [map {{id => $_, label => $BLOCK_TYPES{$_}{'label'}()}} sort keys %BLOCK_TYPES];
}

sub get_fields_to_trim {qw(store_id store_url)}

sub related_models {
    my ($self) = @_;

    return {%{$self->SUPER::related_models()}, %{$self->internal_related_models()}};
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_processing_before_validation($opts);

    if ($self->app->user_features->page_owner_has_feature_internal_simple_inapp()) {
        unless (exists $opts->{store_id} && $opts->{store_id}) {
            delete $opts->{store_id};
        }
    } else {
        # todo пока не перешли на internal_simple_inapp это поле не используется
        # в базе сейчас лежит пустая строка а поле optional и не проходит валидация
        unless (exists $opts->{store_url} && $opts->{store_url}) {
            $opts->{store_url} = undef;
        }
    }
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    my $id = $self->hook_stash->get('id');

    $self->api_balance->create_or_update_place(
        operator_uid  => $self->get_option(cur_user => {})->{'id'},
        client_id     => $ADINSIDE_CLIENT_ID,
        page_id       => $opts->{'id'},
        domain        => $opts->{'store_id'},
        campaign_type => 3,                         # Контекстная
        viptype => 2,    # Внутренние площадки
    );

    $self->do_action($id, 'start')
      if $self->check_action($id, 'start');
}

sub hook_owner_processing { }

sub get_opts_schema_name {'internal_mobile_app_opts'}

sub fix_template {
    my ($self, $qv) = @_;

    my $fields = $qv->template->{'fields'};
    # todo пока не перешли на internal_simple_inapp это поле не используется
    $fields->{store_url}{optional} = TRUE;
    if ($self->app->user_features->page_owner_has_feature_internal_simple_inapp()) {
        $fields->{store_id}{optional} = TRUE;
    }
}

sub check_simple_inapp_feature {
    my ($self) = @_;
    return $self->app->user_features->page_owner_has_feature_internal_simple_inapp();
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = ();

    if ($need_fields->{bk_languages}) {
        $result{'bk_languages'} = $self->bk_language->get_all(fields => [qw(id language display_name)]);
    }

    return \%result;
}

sub has_feature_simple_inapp {
    my ($self) = @_;

    return ($self->app->user_features->has_feature_internal_simple_inapp());
}

sub page_owner_has_feature_simple_inapp {
    my ($self) = @_;

    return ($self->app->user_features->page_owner_has_feature_internal_simple_inapp());
}

TRUE;
