package Application::Model::Product::InternalAN::MobileApp::RTB;

use qbit;

use base qw(
  Application::Model::Block::Internal
  RestApi::MultistateModel
  Application::Model::ValidatableMixin
  );

use QBit::Validator;

use Partner::Filters qw(get_filters_not_adaptive_banner_mobile_blocks);

consume qw(
  Application::Model::Role::Block::Has::CloseButtonDelay
  Application::Model::Role::Block::Has::MediaBlock
  Application::Model::Role::Block::Has::MobileAppMode
  Application::Model::Role::Block::Has::ShowSlider
  Application::Model::Role::Block::Has::Tags
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::Categories
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::CreateDate
  Application::Model::Role::Has::CustomBKOptions
  Application::Model::Role::Has::Block::DesignSettings
  Application::Model::Role::Has::DSPS::MobileNativeRule
  Application::Model::Role::Has::DSPS::MobileBannerRule
  Application::Model::Role::Has::DSPS::MobileAdaptiveBannerRule
  Application::Model::Role::Has::DSPS::MobileRewardedRule
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::RTB
  Application::Model::Role::Has::RTBDesignFields
  Application::Model::Role::Has::ShowVideo
  Application::Model::Role::Has::Strategies::ShiftingFields
  Application::Model::Role::Block::Has::UnmoderatedDSPs
  Application::Model::Role::Block::Has::BlocksLimit
  Application::Model::Role::Block::Has::InitUniqueId
  Application::Model::Role::JavaJsonApiProxy
  );

use PiConstants qw(
  $DSP_DIRECT_ID
  $DSP_MOBILE_TYPE_ID
  $FONT_SIZES
  @INTERNAL_MOBILE_DEFAULT_DSPS
  @VIDEO_BANNER_DSPS
  @VIDEO_INTERSTITIAL_DSPS
  $SEPARATE_CPM_STRATEGY_ID
  );

use Application::Model::Product::BlockTypes::MobileAppRTB qw(
  direct_block_types
  @FIELDS
  @COLOR_FIELDS
  @BOOLEAN_FIELDS
  direct_appearance_fields
  %DIRECT_BLOCKS
  @MEDIA_BLOCKS
  %MEDIA_BLOCKS
  @DSP_BLOCKS
  %DSP_BLOCKS
  );

use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Errors;
use Exception::Validator::Fields;

sub accessor      {'internal_mobile_app_rtb'}
sub db_table_name {'internal_mobile_app_rtb'}

sub get_campaign_model_name {'internal_mobile_app'}

sub get_product_name {gettext('internal_mobile_app_rtb')}

sub get_page_id_field_name {'campaign_id'}

sub public_id_prefix {'R-IM-'}

sub design_fields_from_opts {TRUE}

sub get_dsp_type {[$DSP_MOBILE_TYPE_ID]}

sub is_block_table_with_multiple_models {TRUE}

sub is_show_video_applicable {
    my ($self, $block) = @_;

    return unless defined($block->{block_type});

    my %show_video_by_block_type = (
        native          => {applicable => 1, default_value => 0},
        interstitial    => {applicable => 1},
        banner          => {applicable => 1},
        adaptive_banner => {applicable => 0},
    );

    return $show_video_by_block_type{$block->{block_type}};
}

# TODO: Block::Mobile ???
sub get_video_dsp_list {
    my ($self, $block) = @_;

    return () unless defined($block->{block_type});

    my %video_dsp_list_by_block_type = (
        native       => [],
        interstitial => \@VIDEO_INTERSTITIAL_DSPS,
        banner       => \@VIDEO_BANNER_DSPS,
    );

    return @{$video_dsp_list_by_block_type{$block->{block_type}}};
}

sub get_video_dsp_list_depends {qw(block_type)}

sub get_video_dsp_turn_on_block_filter {
    my ($self, $dsp) = @_;

    my %block_type_by_video_dsp;
    push @{$block_type_by_video_dsp{$_}}, 'interstitial' for @VIDEO_INTERSTITIAL_DSPS;
    push @{$block_type_by_video_dsp{$_}}, 'banner'       for @VIDEO_BANNER_DSPS;

    if (my $block_types = $block_type_by_video_dsp{$dsp->{id}}) {
        return [block_type => IN => $block_types];
    } else {
        return;
    }
}

sub role_show_video_get_fields_depends {[qw(show_video block_type)]}

sub get_opts_schema_name {'internal_mobile_app_rtb_opts'}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        %{$class->SUPER::get_structure_model_accessors()},
        internal_mobile_app => 'Application::Model::Product::InternalAN::MobileApp',
        tns_dict_brand      => 'Application::Model::TNSDict::Brand',
        dsp                 => 'Application::Model::DSP',
        kv_store            => 'QBit::Application::Model::KvStore',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            name        => 'internal_mobile_app',
            description => d_gettext('Right to manage RTB blocks on internal mobile applications'),
            rights      => {
                map {$self->get_description_right($_)}
                  qw(
                  edit_all
                  edit_field__geo
                  edit_field__is_custom_format_direct
                  view_field__comment
                  ),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},
        domain => {
            depends_on => ['store_id'],
            get        => sub {$_[1]->{store_id}},
            type       => 'string',
            api        => 1,
        },
        campaign_id => {
            default    => TRUE,
            db         => TRUE,
            pk         => TRUE,
            need_check => {type => 'int_un',},
            type       => 'string',
        },
        application_id => {
            default => TRUE,
            db      => TRUE,
            db_expr => 'campaign_id',
            label   => d_gettext('Application ID'),
            hint    => d_gettext('The unique application identifier (the same in BK)'),
            api     => 1,
            type    => 'number',
        },
        block_type => {
            db         => TRUE,
            label      => d_gettext('Mobile block type'),
            api        => 1,
            type       => 'string',
            need_check => {
                len_max => 18,     # as in database
                check   => sub {
                    my ($qv, $block_type, $template) = @_;

                    my $page_id = $qv->data->{'page_id'};
                    my $apps    = $qv->app->internal_mobile_app->get_all(
                        fields => ['block_types'],
                        filter => {id => $page_id}
                    );

                    $qv->_add_error($template, gettext('Mobile app with page id %s not found', $page_id), [])
                      unless $apps && @$apps;

                    my $block_types = $apps->[0]{'block_types'};

                    throw Exception::Validator::Fields gettext('Invalid block_type value')
                      unless grep {$_->{'id'} eq $block_type} @{$block_types};
                  }
            },
        },
        block_type_label => {
            depends_on => [qw(block_type)],
            get        => sub {
                return $Application::Model::Product::InternalAN::MobileApp::BLOCK_TYPES{$_[1]->{'block_type'}}{'label'}(
                );
            },
            api  => 1,
            type => 'string',
        },
        is_custom_format_direct => {
            from_opts => 'from_hash',
            get       => sub {
                $_[1]->{'is_custom_format_direct'} // 0;
            },
            label      => d_gettext('Custom Direct format enabled'),
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
        },
        border_type => {
            from_opts  => 'from_hash',
            type       => 'string',
            need_check => {in => [qw(none block ad collapse)],},
            api        => 1,
        },
        title_font_size => {from_opts => 'from_hash', type => 'string', need_check => {in => [qw(1 2 3)],}, api => 1,},
        font_family     => {
            from_opts  => 'from_hash',
            type       => 'string',
            need_check => {in => ['', 'arial', 'courier new', 'tahoma', 'times new roman', 'verdana',],},
            api        => 1,
        },
        font_size     => {from_opts => 'from_hash', type => 'string', need_check => {in   => $FONT_SIZES,}, api => 1,},
        site_bg_color => {from_opts => 'from_hash', type => 'string', need_check => {type => 'color'},      api => 1,},
        bg_color =>
          {from_opts => 'from_hash', type => 'string', need_check => {type => 'color', optional => TRUE,}, api => 1,},
        border_color => {from_opts => 'from_hash', type => 'string', need_check => {type => 'color',}, api => 1,},
        header_bg_color =>
          {from_opts => 'from_hash', type => 'string', need_check => {type => 'color', optional => TRUE,}, api => 1,},
        title_color => {from_opts => 'from_hash', type => 'string', need_check => {type => 'color'}, api => 1,},
        text_color  => {from_opts => 'from_hash', type => 'string', need_check => {type => 'color'}, api => 1,},
        url_background_color =>
          {from_opts => 'from_hash', type => 'string', need_check => {type => 'color'}, api => 1,},
        url_color       => {from_opts => 'from_hash', type => 'string',  need_check => {type => 'color'},    api => 1,},
        hover_color     => {from_opts => 'from_hash', type => 'string',  need_check => {type => 'color'},    api => 1,},
        sitelinks_color => {from_opts => 'from_hash', type => 'string',  need_check => {type => 'color'},    api => 1,},
        border_radius   => {from_opts => 'from_hash', type => 'boolean', need_check => {type => 'boolean',}, api => 1,},
        no_sitelinks    => {from_opts => 'from_hash', type => 'boolean', need_check => {type => 'boolean'},  api => 1,},
        favicon         => {from_opts => 'from_hash', type => 'boolean', need_check => {type => 'boolean'},  api => 1,},
        links_underline => {from_opts => 'from_hash', type => 'boolean', need_check => {type => 'boolean'},  api => 1,},
        store_id        => {
            depends_on => ['page_id', 'pages.store_id'],
            label      => d_gettext('store_id'),
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'store_id'};
            },
            api  => 1,
            type => 'string',
        },
        application => {
            depends_on => ['page_id', 'pages.id', 'pages.multistate'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}};
            },
            api  => 1,
            type => 'internal_mobile_app',
        },
        application_caption => {
            depends_on => ['page_id', 'pages.caption'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'caption'};
            },
            api  => 1,
            type => 'string',
        },
        application_platform => {
            depends_on => ['page_id', 'pages.type_name'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'type_name'};
            },
            api  => 1,
            type => 'string',
        },
        is_application_deleted => {
            depends_on => ['is_page_deleted'],
            get        => sub {
                $_[1]->{'is_page_deleted'};
            },
            api  => 1,
            type => 'boolean',
        },
        bk_state_name => {
            depends_on => ['page_id', 'pages.bk_state_name'],
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'bk_state_name'};
            },
        },
        #redefine
        blind => {
            db      => TRUE,
            db_expr => \0,
        },
        rich_media => {
            from_opts  => 'from_hash',
            need_check => {
                optional => TRUE,
                type     => 'boolean',
            },
            # нет флага api, в отличие от внешних RTB
            type => 'boolean',
        },
    };
}

sub is_page_reachable {
    my ($self, $page_id_value) = @_;
    return 0 < @{
        $self->get_all_campaigns_for_adding(
            filter => [id => '=' => $page_id_value],
            fields => [qw(id store_id type_name caption)]
        )
      };
}

sub get_all_campaigns_for_adding {
    my ($self, %opts) = @_;
    return $self->SUPER::get_all_campaigns_for_adding(%opts,
        (fields => [qw(id store_id type_name caption), @{$opts{'fields'} // []}]));
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $parent_filter = $self->SUPER::get_structure_model_filter();
    return {
        db_accessor => $parent_filter->{'db_accessor'},
        fields      => {
            %{$parent_filter->{'fields'}},
            campaign_id    => {type => 'number', label => d_gettext('Application ID')},
            application_id => {type => 'alias',  path  => [qw(campaign_id)], label => d_gettext('Application ID')},
            page_id => {
                type      => 'number',
                label     => d_gettext('Page ID'),
                db_filter => sub {
                    return ['campaign_id', $_[1]->[1], \$_[1]->[2]];
                  }
            },
            application => {
                type           => 'subfilter',
                model_accessor => 'internal_mobile_app',
                field          => 'campaign_id',
                label          => d_gettext('application'),
            },
            block_type => {
                type   => 'dictionary',
                label  => d_gettext('Block type'),
                values => sub {
                    [
                        map {
                            +{
                                id    => $_,
                                label => $Application::Model::Product::InternalAN::MobileApp::BLOCK_TYPES{$_}{'label'}()
                             }
                          }
                          sort keys(%Application::Model::Product::InternalAN::MobileApp::BLOCK_TYPES)
                    ];
                  }
            },
            page_caption => {type => 'alias', path => [qw(page caption)]},
        }
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [
            {name => 'managers.manager.login', label => gettext('Manager\'s login')},
            {name => 'id',                     label => gettext('Block ID')},
            {name => 'caption',                label => gettext('Block caption')},
            {name => 'multistate',             label => gettext('Status')},
        ],
        [
            {name => 'page_id',              label => gettext('Page ID')},
            {name => 'application.store_id', label => gettext('Store ID')},
            {name => 'application.type',     label => gettext('Application type')},
            {name => 'block_type',           label => gettext('Block type')},
        ],
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_pgettext('Block multistate', 'New'),
        multistates => [
            [deleted           => d_pgettext('Block multistate', 'Archived')],
            [working           => d_pgettext('Block multistate', 'Working')],
            [check_statistics  => d_pgettext('Block multistate', 'Check statistics')],
            [deleted_with_page => d_pgettext('Block multistate', 'Archived with page'), private => TRUE],
            [need_update       => d_pgettext('Block multistate', 'Need update')],
            [updating          => d_pgettext('Block multistate', 'Updating')],
        ],
        actions => {
            edit                   => d_pgettext('Block action', 'Edit'),
            delete                 => d_pgettext('Block action', 'Archive'),
            delete_with_page       => d_pgettext('Block action', 'Archive with page'),
            restore_with_page      => d_pgettext('Block action', 'Restore with page'),
            start                  => d_pgettext('Block action', 'Start'),
            stop                   => d_pgettext('Block action', 'Stop'),
            set_check_statistics   => d_pgettext('Block action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Block action', 'Reset "check_statistics"'),
            duplicate              => d_pgettext('Block action', 'Duplicate'),
        },
        right_name_prefix => $self->accessor . '_',
        right_actions     => {
            add     => d_pgettext('Block action', 'Add'),
            restore => d_pgettext('Block action', 'Restore'),
            set_need_update =>
              {label => d_pgettext('Block action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Block action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Block action', 'Stop update'),  dont_write_to_action_log => TRUE},
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action    => 'delete_with_page',
                from      => 'not deleted',
                set_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page',
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action    => 'start',
                from      => 'not working and not deleted',
                set_flags => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
            },
            {
                action    => 'set_check_statistics',
                from      => 'not deleted',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action => 'duplicate',
                from   => 'not (deleted or deleted_with_page)',
            },
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('dsp_blocks')) {
        $fields->{'dsp_blocks'} = {};

        my $page_ids = $fields->stash->{'page_ids'} //= array_uniq(map {$_->{'page_id'}} @$result);

        foreach (
            @{
                $self->partner_db->media_sizes->get_all(
                    fields => [qw(page_id block_id type)],
                    filter => {page_id => $page_ids},
                )
            }
          )
        {
            push(@{$fields->{'dsp_blocks'}{$_->{'page_id'}, $_->{'block_id'}}}, $_->{'type'});
        }
    }
}

sub get_available_fields_depends {
    [qw(multistate block_type)];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);
    delete($fields{'campaign_id'});

    my $accessor      = $self->accessor();
    my $page_accessor = $self->get_campaign_model_name();

    $self->app->delete_field_by_rights(\%fields, {$page_accessor . '_view_field__managers' => 'manager_ids',});

    my $block_type = $obj->{'block_type'};

    if ($block_type eq 'native') {
        delete(@fields{qw(media_block direct_block)});
    } else {

        delete(@fields{qw(media_block direct_block)})
          if $block_type eq 'interstitial';
    }

    return \%fields;
}

sub get_add_fields {
    my ($self, $block_type) = @_;

    $block_type = $block_type->{block_type} if ref $block_type;

    my $fields = $self->_get_common_add_edit_fields($block_type);

    $fields->{'application_id'} = TRUE;
    $fields->{'block_type'}     = TRUE;
    $fields->{'page_id'}        = TRUE;

    if (defined($block_type) && $block_type eq 'native') {
        delete($fields->{'direct_block'});
    }

    if (defined($block_type) && $block_type eq 'interstitial') {
        delete($fields->{'dsp_blocks'});
    }

    # Frontend should start sending block_type to get_add_fields
    if (defined($block_type)) {
        $self->set_show_video_add_fields({block_type => $block_type}, $fields);
    } else {
        $fields->{show_video} = 1;
    }

    return $fields;
}

sub get_actions_depends {
    [qw(id campaign_id page_id page multistate is_my is_protected)];
}

sub get_editable_fields_depends {
    [qw(id campaign_id page_id multistate is_my is_protected block_type)];
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my %res = %{$self->_get_common_add_edit_fields($data->{'block_type'})};

    if ($data->{'is_custom_format_direct'}
        && !$self->check_short_rights('edit_field__is_custom_format_direct'))
    {
        delete($res{'direct_block'});
        delete($res{'limit'});
    }

    my $block_type = $data->{'block_type'};

    if (defined($block_type) && $block_type eq 'native') {
        delete(@res{('direct_block'), @{$DIRECT_BLOCKS{$block_type}->{'skip_fields'}}});
    }

    return \%res;
}

sub _get_common_add_edit_fields {
    my ($self, $block_type) = @_;

    my $fields = $self->get_fields_by_right(
        no_right_fields => [
            qw(
              adaptive_height
              adaptive_width
              caption
              comment
              dsp_blocks
              ),
            direct_block_fields(),
            direct_appearance_fields()
        ],
        right_fields => {edit => ['is_custom_format_direct']}
    );

    if (defined($block_type) && $block_type eq 'interstitial') {
        delete(
            @$fields{
                qw(
                  adaptive_width
                  adaptive_height
                  favicon
                  no_sitelinks
                  direct_block
                  media_block
                  limit
                  )
              }
        );
    }

    if (defined($block_type) && $block_type eq 'native') {
        delete(@$fields{qw(media_block adaptive_height adaptive_width favicon no_sitelinks callouts)});
    }

    return $fields;
}

sub can_action_duplicate {
    my ($self, $obj, %opts) = @_;

    return FALSE unless $self->check_rights($self->get_rights_by_actions('add'));

    my $object = $self->_get_object_fields($obj, [qw(is_protected)]);

    return !$object->{'is_protected'} || $self->check_rights('edit_protected_pages') ? TRUE : FALSE;
}

sub on_action_restore {
    my ($self, $block, %opts) = @_;

    $self->maybe_do_action($block, 'start');

    $self->update_in_bk($block) unless $opts{'do_not_update_in_bk'};

    return TRUE;
}

sub fix_template {
    my ($self, $qv) = @_;

    $self->SUPER::fix_template($qv);

    my $template   = $qv->template;
    my $block_type = $qv->data->{'block_type'};

    if ($qv->data->{'is_custom_format_direct'}) {
        $template->{'fields'}{'callouts'} = {eq => undef};
    } else {
        my $direct_block = (defined($block_type) && $block_type eq 'native') ? 'native' : $qv->data->{'direct_block'};

        if (   defined($direct_block)
            && $DIRECT_BLOCKS{$direct_block}
            && $DIRECT_BLOCKS{$direct_block}{'skip_fields'})
        {
            foreach (@{$DIRECT_BLOCKS{$direct_block}{'skip_fields'}}) {
                $template->{'fields'}{$_} = {eq => undef};
            }
        }
    }

    if ($block_type eq 'native') {
        $template->{'fields'}{'limit'} = {eq => undef, optional => TRUE}
          unless $qv->data->{'show_slider'};

        foreach (
            qw(
            bg_color        border_color         border_radius   border_type     callouts
            direct_block    favicon         font_family     font_size
            header_bg_color horizontal_align     hover_color     links_underline media_block
            no_sitelinks    site_bg_color        sitelinks_color text_color      title_color
            title_font_size url_background_color url_color
            )
          )
        {
            $template->{'fields'}{$_}{'optional'} = TRUE;
        }
    }

    if ($block_type eq 'interstitial') {
        $template->{'fields'}{'font_size'}            = {eq   => undef,     optional => TRUE};
        $template->{'fields'}{'title_font_size'}      = {eq   => undef,     optional => TRUE};
        $template->{'fields'}{'font_family'}          = {eq   => undef,     optional => TRUE};
        $template->{'fields'}{'url_background_color'} = {eq   => undef,     optional => TRUE};
        $template->{'fields'}{'border_radius'}        = {type => 'boolean', optional => TRUE};
        $template->{'fields'}{'border_type'} = {in => [qw(none block ad collapse)], optional => TRUE};
        $template->{'fields'}{'sitelinks_color'} = {type => 'color', optional => TRUE};

        foreach (qw(limit media_block direct_block no_sitelinks favicon callouts)) {
            $template->{'fields'}{$_}{'optional'} = TRUE;
        }
    }

    if ($qv->data->{'border_type'} && $qv->data->{'border_type'} eq 'none') {
        foreach (
            qw(
            border_color
            border_radius
            )
          )
        {
            $template->{'fields'}{$_} = {eq => undef};
        }
    }

    if ($qv->data->{'no_sitelinks'}) {
        $template->{'fields'}{'sitelinks_color'} = {eq => undef};
    }

    $qv->template($template);
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_campaign($filter);

    return $filter;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = map {$_ => TRUE} qw(
      adaptive_height
      adaptive_width
      block_type
      caption
      dsp_blocks
      ), direct_block_fields(), direct_appearance_fields();

    return \%fields_need_update;
}

sub boolean_fields {
    return {map {$_ => TRUE} @BOOLEAN_FIELDS};
}

sub direct_block_fields {
    return qw(
      direct_block media_block limit
      );
}

sub direct_blocks {
    return \%DIRECT_BLOCKS;
}

sub media_blocks {
    return \%MEDIA_BLOCKS;
}

sub dsp_blocks {
    return \%DSP_BLOCKS;
}

sub get_bk_block_data {
    my ($self, $block) = @_;

    return $self->get_bk_mobile_rtb_settings($block);
}

sub get_bk_block_data_key {'rtb_blocks'}

sub get_bk_common_data {
    return ();
}

sub get_dsp_rules {
    my ($self) = @_;

    return [
        @{$self->SUPER::get_dsp_rules()},
        $self->get_show_video_dsp_rule(),
        $self->get_mobile_native_dsp_rule(),
        $self->get_mobile_banner_dsp_rule(),
        $self->get_mobile_adaptive_banner_dsp_rule(),
        $self->get_mobile_rewarded_dsp_rule(),
        $self->get_special_show_video_dsp_rule(),
        # TODO: almost the same in external
        Partner::DSP::Rule->new(
            default_dsps => Partner::DSP::Rule::Part->new(
                sub => sub {
                    extend => {id => \@INTERNAL_MOBILE_DEFAULT_DSPS},;
                },
            ),
        ),
    ];
}

sub get_filter_applicable_to_by_strategy {
    my ($self, $strategy_id) = @_;

    if ($SEPARATE_CPM_STRATEGY_ID eq $strategy_id) {
        return (applicable_to => get_filters_not_adaptive_banner_mobile_blocks());
    } else {
        return ();
    }
}

sub get_ad_type_list {
    return [qw(media text)];
}

sub get_types_direct_blocks {
    my ($self, $opts) = @_;

    my $block_type = $opts->{block_type};

    my $direct_blocks = $self->direct_block_types();
    my @result        = ();
    my $block_types;

    foreach (@$direct_blocks) {
        $_->{'short_caption'} = $_->{'short_caption'}();
        $_->{'caption'}       = $_->{'caption'}();
        $block_types          = $_->{'block_types'};

        $_ = {hash_transform($_, [qw(id short_caption caption min_banners max_banners skip_fields banners)])};

        push(@result, $_) if in_array($block_type, $block_types);
    }

    return \@result;
}

sub get_types_dsp_blocks {
    my ($self, $opts) = @_;

    my $block_type = $opts->{block_type};

    my $dsp_blocks = clone(\@DSP_BLOCKS);

    my @result = ();

    foreach (@$dsp_blocks) {
        $_->{'short_caption'} = $_->{'short_caption'}();

        push(@result, $_) if !$block_type || in_array($block_type, $_->{'block_types'});
    }

    return \@result;
}

sub get_types_media_blocks {
    my ($self, $opts) = @_;

    my $block_type = $opts->{block_type};

    my $media_blocks = clone(\@MEDIA_BLOCKS);

    my @result = ();

    foreach (@$media_blocks) {
        $_->{'short_caption'} = $_->{'short_caption'}();

        push(@result, $_) if in_array($block_type, $_->{'block_types'});
    }

    return \@result;
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my $result = $self->_rtb_make_fields_defaults($opts, $need_fields);

    if ($need_fields->{'direct_block'}) {
        $result->{'direct_block'} = $self->get_types_direct_blocks($opts->{attributes});
    }

    return $result;
}

sub get_fields_depends {
    return {
        #если поменялось поле из ключа, то
        #нужно перезапросить поля из значения
        depends => {
            page_id    => [qw(dsps block_type)],
            show_video => [qw(dsps)],
            block_type => [qw(dsps direct_block media_block show_video)],
            id         => [qw(strategy)],
        },
        #для поля из ключа обязятельно нужно передать поля из значения
        required => {
            dsps         => [qw(page_id block_type show_video)],
            block_type   => [qw(page_id)],
            direct_block => [qw(block_type)],
            strategy     => [qw(id block_type)],
            media_block  => [qw(block_type)],
            show_video   => [qw(block_type)],
        },
    };
}

sub _rtb_direct_block_has_limit {
    my ($self, $direct_block) = @_;

    return in_array($direct_block, [qw(adaptive)]);
}

sub api_available_actions {
    return qw(duplicate edit delete restore);
}

sub api_can_edit {TRUE}

sub api_can_add {TRUE}

sub hook_owner_processing { }

sub get_fields_to_trim {qw(caption)}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $opts->{'campaign_id'} //= $opts->{'page_id'};

    $self->SUPER::hook_set_initialize_settings($opts);

    $self->partner_db->transaction(
        sub {
            $opts->{id} = $self->internal_mobile_app->get_next_block_id($opts->{'page_id'});
        }
    );
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    my $settings = $self->hook_stash->get('settings');
    $opts->{'page_id'} //= $settings->{'campaign_id'} // $settings->{'application_id'};

    $self->SUPER::hook_fields_processing_before_validation($opts);

    if ($self->hook_stash->mode('edit')) {

        my $current = $self->hook_stash->get('current');

        my $rules = $self->get_dsp_rule_set();
        $rules->edit_dsp_list($current, $opts);
    }

    if ($self->hook_stash->mode('add')) {
        my $block_type = $opts->{'block_type'};

        $opts->{'show_video'} //= 0;

        if ($block_type eq 'native') {
            $opts->{'direct_block'} //= 'native';
            $opts->{'media_block'}  //= 'native';
            $opts->{'dsp_blocks'}   //= [qw(native)];
            $opts->{'favicon'}      = 0;
            $opts->{'no_sitelinks'} = 0;
            foreach (
                qw(bg_color     border_color     border_radius   border_type          font_family   font_size
                header_bg_color horizontal_align hover_color     links_underline      site_bg_color sitelinks_color
                text_color      title_color      title_font_size url_background_color url_color)
              )
            {
                $opts->{$_} = undef;
            }
        }

        if ($block_type eq 'interstitial') {
            $opts->{'direct_block'} //= '320x480';
            $opts->{'media_block'}  //= '320x480';
            $opts->{'dsp_blocks'}   //= [qw(320x480 480x320 240x400 400x240 300x250 300x300)];
            $opts->{'favicon'}      = 0;
            $opts->{'no_sitelinks'} = 0;
        }
    }
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    my $id      = $self->hook_stash->get('id');
    my $related = $self->hook_stash->get('fields_from_related_models');
    my $mode    = $self->hook_stash->mode('add') ? 'add' : 'edit';

    if (my $dsp_blocks = $related->{'dsp_blocks'}) {
        $self->partner_db->media_sizes->delete(
            $self->partner_db->filter(
                {
                    page_id  => $id->{'campaign_id'},
                    block_id => $id->{'id'},
                }
            )
        ) if $mode eq 'edit';

        $self->partner_db->media_sizes->add_multi(
            [
                map +{
                    page_id  => $id->{'campaign_id'},
                    block_id => $id->{id},
                    type     => $_,
                },
                @$dsp_blocks
            ]
        );
    }
    if (my $brands = $related->{'brands'}) {
        $self->brands->replace(
            {
                page_id  => $id->{campaign_id},
                block_id => $id->{id},
            },
            $brands, $mode
        );
    }
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    my $id = $self->hook_stash->get('id');

    $self->maybe_do_action($id, 'start');

    $self->update_in_bk($id);
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    $self->update_in_bk($self->hook_stash->get('id')) if $self->need_update_in_bk($self->hook_stash->get('fields'));
}

sub get_dsp_working_on_all_pages_multistate_name {
    'working_on_yandex_services'
}

TRUE;
