package Application::Model::Product::SSP::Application;

use qbit;

use base qw(Application::Model::Product   Application::Model::ValidatableMixin  );

use PiConstants qw/$OS_ID_ANDROID $OS_ID_IOS/;

use Utils::Logger qw/INFOF/;

use Exception::MAAS::NotFound;

sub accessor      {'ssp_application'}
sub db_table_name {'ssp_application'}

__PACKAGE__->model_accessors(
    partner_db    => 'Application::Model::PartnerDB',
    api_http_maas => 'QBit::Application::Model::API::Yandex::HTTPMAAS',
    ssp_imps      => 'Application::Model::Product::SSP::ImpressionLog',
);

__PACKAGE__->model_fields(
    id => {
        default => TRUE,
        pk      => TRUE,
        label   => d_gettext('ID'),
        db      => TRUE
    },
    bundle_id => {
        label => d_gettext('Applicaiont ID'),
        db    => TRUE,
        # NOTE! если не боишься разгребать десяток тыщ конфликтов - раскоментируй
        #      (см https://st.yandex-team.ru/PI-10870#1511455363000 )
        #need_check => {
        #    type => 'bundle_id'
        #}
    },
    apple_id => {
        label => d_gettext('Application apple ID'),
        db    => TRUE
    },
    store_id => {
        label      => d_gettext('Store ID'),
        db         => TRUE,
        need_check => {
            type => 'int_un',
            in   => [1, 2]
        }
    },
    store_app_url => {label => d_gettext('App url'), db => TRUE},
    domain        => {
        db      => TRUE,
        db_expr => 'bundle_id',
    }
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id => {
            type  => 'number',
            label => d_gettext('ID'),
        },
        bundle_id => {
            type  => 'text',
            label => d_gettext('Bundle ID'),
        },
    }
);

sub _get_ssp_application_id($$) {
    my ($self, $app) = @_;

    return $self->partner_db_table()->add($app, ignore => TRUE)
      || $self->partner_db_table()->get_all(
        fields => [qw/id/],
        filter => [AND => [['bundle_id' => '=' => \$app->{bundle_id}], ['apple_id' => '=' => \$app->{apple_id}],]]
      )->[0]->{id};
}

sub create_app_list {
    my ($self, $ssp_app_list) = @_;

    my $LINK_TMPL_ANDROID = 'https://play.google.com/store/apps/details?id=%s&hl=ru&gl=ru';
    my $LINK_TMPL_IOS     = 'https://itunes.apple.com/ru/app/id%s';

    my @app_list = map {
        # to handle \d+ and id\d+ as apple_id
        (my $apple_id_num_only = $_->{app_id}) =~ s/\D//g;
        {
            bundle_id => $_->{bundle} || $_->{app_id},
            apple_id => $OS_ID_IOS == $_->{app_os} ? $apple_id_num_only : '',
            store_id => $_->{app_os},
            store_app_url => $OS_ID_IOS == $_->{app_os}
            ? sprintf($LINK_TMPL_IOS,     $apple_id_num_only)
            : sprintf($LINK_TMPL_ANDROID, $_->{app_id}),
            name => $_->{app_name},
        };
    } @$ssp_app_list;

    my @added_app_list;
    foreach my $app (@app_list) {
        push @added_app_list, {%$app, (application_id => $self->_get_ssp_application_id($app))};
    }
    return \@added_app_list;
}

sub find_by_imp_list {
    my ($self, $ssp_log) = @_;

    my ($itunes_app_id_in, $itunes_bundle_id_in, $gplay_app_id_in) = ([], [], []);
    foreach my $imp (@$ssp_log) {
        my ($bundle_id, $apple_id) = $self->ssp_imps->extract_app_ids($imp);
        my $os_id = $self->ssp_imps->extract_app_os_id($imp);

        next unless ($bundle_id || $apple_id) && $os_id && ($OS_ID_IOS == $os_id || $OS_ID_ANDROID == $os_id);

        if ($apple_id) {
            # $apple_id can be numeric id OR string bundle
            if ($apple_id =~ /^\d+$/) {
                push @$itunes_app_id_in, $apple_id;
            } else {
                push @$itunes_bundle_id_in, $apple_id;
            }
        } elsif ($bundle_id) {
            push @$gplay_app_id_in, $bundle_id;
        } else {
            next;
        }

    }

    return [] unless (@$itunes_app_id_in + @$gplay_app_id_in + @$itunes_bundle_id_in);

    my $ssp_app_list = $self->partner_db_table()->get_all(
        fields => [
            qw(
              id
              bundle_id
              apple_id
              store_id
              )
        ],
        filter => [
            OR => [
                [
                    AND => [
                        ['store_id'  => '='  => \$OS_ID_ANDROID],
                        ['apple_id'  => '='  => \''],
                        ['bundle_id' => 'IN' => \$gplay_app_id_in],
                    ]
                ],
                [
                    AND => [
                        ['store_id' => '=' => \$OS_ID_IOS],
                        [
                            OR => [
                                ['apple_id'  => 'IN' => \$itunes_app_id_in],
                                ['bundle_id' => 'IN' => \$itunes_bundle_id_in]
                            ]
                        ]
                    ]
                ]
            ]
        ],
    );

    return $ssp_app_list;
}

TRUE;
