package Application::Model::Product::SSP::Link::MobileApp;

use qbit;

use base qw(Application::Model::Product);

use Utils::MonitoringUtils qw/send_to_graphite/;

sub accessor              {'ssp_link_mobile_app'}
sub db_table_name         {'ssp_link_mobile_app'}
sub get_campaign_id_field {'mobile_app_id'}

__PACKAGE__->model_accessors(
    partner_db              => 'Application::Model::PartnerDB',
    ssp_imps                => 'Application::Model::Product::SSP::ImpressionLog',
    ssp_seller              => 'Application::Model::Product::SSP::Seller',
    ssp_application         => 'Application::Model::Product::SSP::Application',
    ssp_mobile_app_settings => 'Application::Model::Product::SSP::MobileApp::Settings',
    users                   => 'Application::Model::Users',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'SSP Moderation',
            description => d_gettext('Right to moderate SSP links'),
            rights      => {ssp_moderation_view_all => d_gettext('Right to view all ssp moderations'),},
        }
    ]
);

__PACKAGE__->model_fields(
    id => {default => TRUE, db => TRUE, pk => TRUE, type => 'number', label => d_gettext('ID')},
    multistate     => {db      => TRUE, label => d_gettext('Status')},
    application_id => {default => TRUE, db    => TRUE, pk => FALSE, label => d_gettext('SSP app ID')},
    mobile_app_id  => {default => TRUE, db    => TRUE, pk => FALSE, label => d_gettext('Mobile app ID')},
    tokens    => {db => TRUE, label => d_gettext('Source Tokens'), t => 'json'},
    weight    => {db => TRUE, label => d_gettext('Order')},
    seller_id => {db => TRUE, label => d_gettext('SSP ID')},
    actions   => {
        label      => d_gettext('Actions'),
        depends_on => [qw(id multistate mobile_app_id)],
        get        => sub {
            $_[0]->model->get_actions($_[1]);
        },
    },
    seller => {
        label    => d_gettext('SSP'),
        submodel => {
            seller_id => sub {
                shift->model->partner_db->ssp_seller;
            },
            fields => [qw(id name client_id)],
        },
    },
    comment => {
        db    => TRUE,
        label => d_gettext('Comment'),
    },
    moderation_reason_id => {
        db    => TRUE,
        label => d_gettext('Moderation'),
    },
    application => {
        label    => d_gettext('SSP Application'),
        submodel => {
            application_id => sub {
                shift->model->partner_db->ssp_application;
            },
            fields => [qw(id bundle_id name store_id store_app_url)],
        },
    },
    mobile_app => {
        label    => d_gettext('Mobile application'),
        submodel => {
            mobile_app_id => sub {
                shift->model->partner_db->ssp_mobile_app_settings;
            },
            fields => [qw(id page_id)],
        },
    },
    state_name => {
        label      => d_gettext('State'),
        depends_on => [qw(multistate)],
        get        => sub {
            my ($self, $obj) = ($_[0]->model, $_[1]);
            if ($self->check_multistate_flag($obj->{'multistate'}, 'approved')) {
                return pgettext('SSP moderation state name', 'Approved');
            } elsif ($self->check_multistate_flag($obj->{'multistate'}, 'rejected')) {
                return pgettext('SSP moderation state name', 'Rejected');
            } else {
                return pgettext('SSP moderation state name', 'New');
            }
        },
    },
    editable_fields => {
        label => d_gettext('Editable fields'),
        get   => sub {
            $_[0]->model->get_editable_fields();
          }
    },
    available_fields => {
        label => d_gettext('Available fields'),
        get   => sub {
            return $_[0]->model->get_available_fields();
          }
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        # без этого не работает do_action (Exception::Validation::BadArguments: Неизвестное поле "id")
        id => {
            type  => 'number',
            label => d_gettext('ID'),
        },
        application_id => {
            type  => 'number',
            label => d_gettext('Application ID'),
        },
        mobile_app_id => {
            type  => 'number',
            label => d_gettext('Mobile App ID'),
        },
        mobile_app => {
            type           => 'subfilter',
            model_accessor => 'ssp_mobile_app_settings',
            field          => 'mobile_app_id',
            fk_field       => 'id',
            label          => d_gettext('Page ID')
        },
        seller_id => {
            type   => 'dictionary',
            label  => d_gettext('SSP'),
            values => sub {
                return [map {{id => $_->{'id'}, label => $_->{'name'}}}
                      @{$_[0]->ssp_seller->get_all(fields => [qw(id name)], order_by => ['name', 1])}];
            },
        },
        moderation_reason_id => {
            type   => 'dictionary',
            label  => d_gettext('Reasons'),
            values => sub {
                return [map {{id => $_->{'id'}, label => $_->{'title'}}} @{$_[0]->dict_reasons()}];
            },
        },
        application => {
            type           => 'subfilter',
            model_accessor => 'ssp_application',
            field          => 'application_id',
            fk_field       => 'id',
            label          => d_gettext('Domain, Bundle ID')
        },
        multistate => {type => 'multistate', label => d_gettext('Status')},
    }
);

__PACKAGE__->multistates_graph(
    empty_name  => 'New',
    multistates => [
        [new      => d_pgettext('SSP moderation state', 'New')],
        [approved => d_pgettext('SSP moderation state', 'Approved')],
        [rejected => d_pgettext('SSP moderation state', 'Rejected')],
    ],
    actions => {
        'new'           => d_pgettext('SSP moderation action', 'new'),
        'approve'       => d_pgettext('SSP moderation action', 'approve'),
        'approve_first' => d_pgettext('SSP moderation action', 'approve'),
        'reject'        => d_pgettext('SSP moderation action', 'reject'),
        'edit'          => d_pgettext('SSP moderation action', 'edit'),
    },
    multistate_actions => [
        {
            action      => 'new',
            from        => '__EMPTY__',
            set_flags   => [qw(new)],
            reset_flags => [qw()],
        },
        {
            action      => 'approve',
            from        => '__EMPTY__ or new or rejected',
            set_flags   => [qw(approved)],
            reset_flags => [qw(rejected new)],
        },
        {
            action      => 'approve_first',
            from        => '__EMPTY__ or new or rejected',
            set_flags   => [qw(approved)],
            reset_flags => [qw(rejected new)],
        },
        {
            action      => 'reject',
            from        => '__EMPTY__ or new or approved or rejected',
            set_flags   => [qw(rejected)],
            reset_flags => [qw(approved new)],
        },
        {
            action => 'edit',
            from   => 'new or approved or rejected',
        },
    ],
);

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [
            {name => 'application.bundle_id', label => gettext('Bundle ID')},
            {name => 'seller_id',             label => gettext('SSP')},
            {name => 'mobile_app.page_id',    label => gettext('Page ID')},
        ],
        [
            {name => 'multistate',           label => gettext('State')},
            {name => 'moderation_reason_id', label => gettext('Reason')},
        ],
    ];
}

sub get_available_fields {
    my ($self) = @_;

    my $model_fields = $self->get_model_fields;

    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub get_editable_fields {{moderation_reason_id => 1, comment => 1}}

sub on_action_edit {
    my ($self, $obj, %data) = @_;

    my $fields_hs = $self->get_editable_fields();
    %data = map(exists($data{$_}) ? ($_ => $data{$_}) : (), keys(%$fields_hs));

    if (defined($data{moderation_reason_id})) {
        my $tmp_rights = $self->app->add_all_tmp_rights();

        my $reasons_by_id = $self->dict_reasons(by_id => 1);
        if ($data{moderation_reason_id} eq '0') {
            if ($self->check_action($obj, 'approve_first')) {
                $self->do_action($obj, 'approve_first');
            } else {
                $self->do_action($obj, 'approve');
            }
        } elsif ($reasons_by_id->{$data{moderation_reason_id}}) {
            $self->do_action($obj, 'reject', %data);
        } else {
            throw Exception::Validation::BadArguments gettext("Wrong reason");
        }
    }

    if (keys(%data) > 0) {
        $self->partner_db->ssp_link_mobile_app->edit($obj, \%data);
    }

    return TRUE;
}

sub can_action_approve {
    my ($self, $obj, %opts) = @_;

    my $link = $self->_get_object_fields($obj, [qw(mobile_app_id)]);

    return $link->{'mobile_app_id'};
}

sub on_action_approve {
    my ($self, $obj, %opts) = @_;

    my $link = $self->_get_object_fields($obj, [qw(mobile_app_id)]);

    $self->ssp_mobile_app_settings->do_action($link->{'mobile_app_id'}, 'start');
}

sub can_action_approve_first {
    my ($self, $obj, %opts) = @_;

    my $link = $self->_get_object_fields($obj, [qw(mobile_app_id)]);

    return $link->{'mobile_app_id'} ? FALSE : TRUE;
}

sub on_action_approve_first {
    my ($self, $obj, %opts) = @_;

    $self->_app_create($obj);
}

sub on_action_reject {
    my ($self, $obj, %opts) = @_;

    my $link = $self->_get_object_fields($obj, [qw(mobile_app_id)]);

    $self->ssp_mobile_app_settings->do_action($link->{'mobile_app_id'}, 'stop') if $link->{'mobile_app_id'};
}

=head2 dict_reasons

Front method

Params:

 * (opt) by_id => TRUE

Returns: [{reason}, {reason}]

=cut

sub dict_reasons {
    my ($self, %opts) = @_;

    my $dict_reasons = [
        {id => '1', title => gettext('app. bad bundle ID')},
        {id => '2', title => gettext('partner. direct')},
        {id => '3', title => gettext('app. law')},
        {id => '4', title => gettext('app. child')},
        {id => '5', title => gettext('app. not interesting')},
        {id => '6', title => gettext('ssp. bad traffic')},
        {id => '7', title => gettext('ssp. minor traffic')},
    ];

    if ($opts{by_id}) {
        my $dict_reasons_by_id //= {map(+($_->{id} => $_), @$dict_reasons)};
        return $dict_reasons_by_id;
    } else {
        return $dict_reasons;
    }
}

sub get_all {
    my ($self, %opts) = @_;
    my $data = $self->SUPER::get_all(%opts);

    foreach my $row (@$data) {
        $self->fields_from_db($row);
    }

    return $data;
}

=head2 get_hist_all

Front method

Params:

 * <link_object>
 * (opt) limit => 100

Returns: [{hits}, {hits}]

=cut

sub get_hits_all {
    my ($self, $obj, %opts) = @_;

    $obj = $self->_get_object_fields($obj, [qw(id application_id seller_id)]);

    my $hits = $self->partner_db->ssp_impression_log->get_all(
        fields => [qw(date hits seller_id)],
        filter => [AND => [[application_id => '=' => \$obj->{application_id}]]],
        limit => $opts{limit} // 5,
        order_by => [[$opts{order_by} || 'date', 1]],
    );

    return $hits;
}

=head2 get_moderation_history

Front method

Params:

 * <link_object>

Returns:

 * [{action_log},{action_log}]

=cut

sub get_moderation_history {
    my ($self, $obj) = @_;

    my $logs =
      $self->get_action_log_entries($obj, explain_actions => TRUE, explain_multistates => TRUE, expand_opts => TRUE);

    my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_all));
    rows_expand($logs, 'user_id' => $self->users => 'user');

    my @out;
    foreach (@$logs) {
        next if $_->{'action'} eq 'edit';
        my $data = $_->{'opts'};
        if (ref($data) eq 'HASH' && $data->{'moderation_reason_id'}) {
            my $reason = $self->dict_reasons('by_id' => 1)->{$data->{'moderation_reason_id'}};
            $_->{'comment'} = $reason ? $reason->{'title'} : gettext('Unknown reason');
        } else {
            $_->{'comment'} = '';
        }

        push @out, $_;
    }

    return \@out;
}

=head2 get_other_links

Front method

Params:

 * <link_object>

Returns: [{link}, {link}]

=cut

sub get_other_links {
    my ($self, $obj, %opts) = @_;

    $obj = $self->_get_object_fields($obj, [qw(id application_id)]);

    my $links = $self->get_all(
        filter => [AND => [[application_id => '=' => $obj->{application_id}], [id => '<>' => $obj->{id}]]],
        fields   => [qw(id mobile_app seller state_name application_id weight)],
        order_by => [[$opts{order_by} || 'weight', 1]],
    );

    return $links;
}

sub update_data {
    my ($self, $date) = @_;

    my $send_value = dates_delta_days($date, curdate(oformat => 'db'), iformat => 'db');
    send_to_graphite(
        value   => $send_value,
        path    => 'SSP.Link.MobileApp.link_process_age',
        solomon => {
            sensor => 'SSP.Link.MobileApp.link',
            metric => 'process_age',
        }
    );
    #die "here";

    my $imps = $self->ssp_imps->get_sources($date);

    send_to_graphite(
        value   => scalar(@$imps),
        path    => 'SSP.Link.MobileApp.link_count_sources',
        solomon => {
            sensor => 'SSP.Link.MobileApp.link',
            metric => 'count_sources',
        }
    );

    my ($cnt_link_updated, $cnt_link_created) = (0, 0);

    foreach my $imp (@$imps) {
        next unless $imp->{application_id};
        next unless $imp->{source_type} eq 'app-media';
        if ($self->_update_from_impression($imp)) {
            $cnt_link_updated += 1;
        } elsif ($self->_create_from_impression($imp)) {
            $cnt_link_created += 1;
        }
    }

    send_to_graphite(
        value   => $cnt_link_updated,
        path    => 'SSP.Link.MobileApp.link_count_updated',
        solomon => {
            sensor => 'SSP.Link.MobileApp.link',
            metric => 'count_updated',
        }
    );
    send_to_graphite(
        value   => $cnt_link_created,
        path    => 'SSP.Link.MobileApp.link_count_created',
        solomon => {
            sensor => 'SSP.Link.MobileApp.link',
            metric => 'count_created',
        }
    );
}

sub _app_create {
    my ($self, $obj) = @_;

    my $link = $self->get($obj, fields => [qw(id seller_id application)]);

    $link->{'source_id'} = $link->{'application'}{'id'};

    my $mobile_app_id = $self->ssp_mobile_app_settings->create_from_impression($link);

    # save
    $self->partner_db->ssp_link_mobile_app->edit($link, {mobile_app_id => $mobile_app_id});

    return $mobile_app_id;
}

# create moderation from data from BK
sub _create_from_impression {
    my ($self, $imp) = @_;

    my $link = $self->partner_db->ssp_link_mobile_app->add(
        {
            application_id => $imp->{application_id},
            tokens         => to_json($imp->{source_token}),
            weight         => $imp->{hits},
            seller_id      => $imp->{seller_id},
        }
    );

    $self->do_action($link, 'new');

    return $link;
}

sub _update_from_impression {
    my ($self, $imp) = @_;

    # get old moderation
    my $link = $self->partner_db->ssp_link_mobile_app->get_all(
        fields => [
            qw(
              id
              application_id
              mobile_app_id
              tokens
              weight
              seller_id
              )
        ],
        filter => [
            AND =>
              [['application_id' => '=' => \[$imp->{'application_id'}]], ['seller_id' => '=' => \$imp->{'seller_id'}]],
        ],
    );

    # no link
    return 0 unless $link && @$link;

    # update record

    $link = $link->[0];

    # new token
    my $tokens = from_json($link->{'tokens'} || '[]');
    my $new_tokens = arrays_difference($imp->{'source_token'}, $tokens);

    # save data in db
    $self->partner_db->ssp_link_mobile_app->edit(
        $link,
        {
            tokens => to_json([@$new_tokens, @$tokens]),
            weight => $imp->{'hits'},
        }
    );

    # save data in bk (mobile app)
    if ($link->{'mobile_app_id'} && @$new_tokens) {
        $self->ssp_mobile_app_settings->do_action($link->{'mobile_app_id'}, 'set_need_update');
    }

    return 1;
}

TRUE;
