package Application::Model::Product::SSP::RTB;

use qbit;

use base qw(Application::Model::Block::External);

use Application::Model::Product::SSP::BlockTypes::RTB qw(
  direct_block_types
  direct_appearance_fields
  @BOOLEAN_FIELDS
  %DIRECT_BLOCKS
  %MEDIA_BLOCKS
  %DSP_BLOCKS
  );

use Exception::Denied;

__PACKAGE__->abstract_methods(qw(get_dsp_type));

consume qw(
  Application::Model::Role::Block::Has::MediaBlock
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::Block::DesignSettings
  Application::Model::Role::Has::Categories
  Application::Model::Role::Has::CreateDate
  Application::Model::Role::Has::DSPS
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::RTB
  Application::Model::Role::Has::RTBDesignFields
  Application::Model::Role::Has::ShowVideo
  Application::Model::Role::Has::Strategies
  );

sub is_show_video_applicable {+{applicable => 0, default_value => 0}}
sub get_video_dsp_list {()}
sub get_video_dsp_list_depends         {()}
sub get_video_dsp_turn_on_block_filter { }
sub role_show_video_get_fields_depends {[qw(show_video)]}

sub get_page_id_field_name {'campaign_id'}

sub design_fields_from_opts {FALSE}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        %{$class->SUPER::get_structure_model_accessors()},
        tns_dict_brand => 'Application::Model::TNSDict::Brand',
        dsp            => 'Application::Model::DSP',
    };
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},

        campaign_id => {
            default => TRUE,
            db      => TRUE,
            pk      => TRUE,
            label   => d_gettext('Campaign ID'),
            type    => 'number',
        },
        block_type                 => {db => TRUE},
        adfox_block                => {db => TRUE,},
        cur_user_is_read_assistant => {
            depends_on => ['page_id', 'pages.cur_user_is_read_assistant'],
            label      => d_gettext('Current user is read assistant'),
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'cur_user_is_read_assistant'};
            },
            type => 'boolean',
        },
        seller_id => {
            depends_on => [qw(campaign_id pages.seller_id)],
            get        => sub {
                return $_[0]->{'pages'}{$_[1]->{'campaign_id'}}{'seller_id'} // '';
            },
        },
        seller_name => {
            depends_on => [qw(campaign_id pages.seller_name)],
            get        => sub {
                return $_[0]->{'pages'}{$_[1]->{'campaign_id'}}{'seller_name'} // '';
            },
        },
        is_custom_format_direct => {
            db    => TRUE,
            label => d_gettext('Custom Direct format enabled')
        },
        (
            map {
                $_ => {db => TRUE}
              } direct_appearance_fields()
        ),
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $parent_filter = $self->SUPER::get_structure_model_filter();
    return {
        db_accessor => $parent_filter->{'db_accessor'},
        fields      => {
            %{$parent_filter->{'fields'}},
            campaign_id => {type => 'number', label => d_gettext('Page ID')},
            page        => {
                type           => 'subfilter',
                model_accessor => 'page',
                fk_field       => 'page_id',
                field          => 'campaign_id',
                label          => d_gettext('Page'),
            }
        }
    };
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_pgettext('Block multistate', 'New'),
        multistates => [
            [deleted           => d_pgettext('Block multistate', 'Archived')],
            [working           => d_pgettext('Block multistate', 'Working')],
            [check_statistics  => d_pgettext('Block multistate', 'Check statistics')],
            [deleted_with_page => d_pgettext('Block multistate', 'Archived with page'), private => TRUE],
            [need_update       => d_pgettext('Block multistate', 'Need update')],
            [updating          => d_pgettext('Block multistate', 'Updating')],
        ],
        actions => {
            edit                   => d_pgettext('Block action', 'Edit'),
            delete                 => d_pgettext('Block action', 'Archive'),
            delete_with_page       => d_pgettext('Block action', 'Archive with page'),
            restore_with_page      => d_pgettext('Block action', 'Restore with page'),
            start                  => d_pgettext('Block action', 'Start'),
            stop                   => d_pgettext('Block action', 'Stop'),
            set_check_statistics   => d_pgettext('Block action', 'Set "check_statistics"'),
            reset_check_statistics => d_pgettext('Block action', 'Reset "check_statistics"'),
        },
        right_name_prefix => $self->accessor . '_',
        right_actions     => {
            add             => d_pgettext('Block action', 'Add'),
            restore         => d_pgettext('Block action', 'Restore'),
            set_need_update => d_pgettext('Block action', 'Set "need_update"'),
            start_update    => d_pgettext('Block action', 'Start update'),
            stop_update     => d_pgettext('Block action', 'Stop update'),
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action    => 'delete_with_page',
                from      => 'not deleted',
                set_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page',
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action    => 'start',
                from      => 'not working and not deleted',
                set_flags => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
            },
            {
                action    => 'set_check_statistics',
                from      => 'not deleted',
                set_flags => ['check_statistics'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('dsp_blocks')) {
        $fields->{'dsp_blocks'} = {};

        my $page_ids = $fields->stash->{'page_ids'} //= array_uniq(map {$_->{'page_id'}} @$result);

        foreach (
            @{
                $self->app->partner_db->media_sizes->get_all(
                    fields => [qw(page_id block_id type)],
                    filter => {page_id => $page_ids},
                )
            }
          )
        {
            push(@{$fields->{'dsp_blocks'}{$_->{'page_id'}, $_->{'block_id'}}}, $_->{'type'});
        }
    }
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor      = $self->accessor();
    my $page_accessor = $self->get_campaign_model_name();

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    $fields->{$_} = TRUE foreach (qw(campaign_id create_date show_video));

    $fields->{'comment'} = TRUE if $self->check_short_rights('view_field__comment');

    return $fields;
}

sub get_actions_depends {
    [qw(id campaign_id page multistate is_custom_format_direct is_protected cur_user_is_read_assistant)];
}

sub get_available_fields_depends {
    [qw(multistate id)];
}

sub get_editable_fields_depends {
    [qw(page_id page campaign_id id multistate is_custom_format_direct is_protected cur_user_is_read_assistant)];
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    my %res = %{$self->_get_common_add_edit_fields()};

    if ($data->{'is_custom_format_direct'}
        && !$self->check_short_rights('edit_field__is_custom_format_direct'))
    {
        delete($res{'direct_block'});
        delete($res{'limit'});
    }

    return \%res;
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        no_right_fields => [
            qw(
              brands
              caption
              dsp_blocks
              is_custom_format_direct
              ),
            direct_block_fields(),
            direct_appearance_fields()
        ],
    );

    return $fields;
}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied unless $self->check_rights($self->get_rights_by_actions('add'));

    $self->_trim_opts(\%opts);

    $self->set_default_settings(\%opts);

    my $block_id = $self->_add(%opts);

    return $block_id;
}

sub set_default_settings {
    my ($self, $opts) = @_;

    $opts->{'block_type'}       //= 'banner';
    $opts->{'adfox_block'}      //= 0;
    $opts->{'adaptive_width'}   //= '';
    $opts->{'adaptive_height'}  //= '';
    $opts->{'blind'}            //= 0;
    $opts->{'show_video'}       //= 0;
    $opts->{'callouts'}         //= 0;
    $opts->{'horizontal_align'} //= 1;
}

sub on_action_edit {
    my ($self, $block, %data) = @_;

    $self->_trim_opts(\%data);

    my @fields = keys(%data);

    if (exists($data{'dsp_blocks'})) {
        my $dsp_blocks_table = $self->app->partner_db->media_sizes;

        $dsp_blocks_table->delete(
            $self->partner_db->filter(
                {
                    page_id  => $block->{'campaign_id'},
                    block_id => $block->{'id'},
                }
            )
        );
        $dsp_blocks_table->add_multi(
            [
                map {page_id => $block->{'campaign_id'}, block_id => $block->{'id'}, type => $_,},
                @{$data{'dsp_blocks'}},
            ]
        );

        delete($data{'dsp_blocks'});
    }

    if (exists($data{'dsps'})) {
        $self->block_dsps->replace({page_id => $block->{'campaign_id'}, block_id => $block->{'id'}},
            $data{'dsps'}, 'edit');

        delete($data{'dsps'});
    }

    if (exists($data{'brands'})) {
        $self->brands->replace({page_id => $block->{'campaign_id'}, block_id => $block->{'id'}},
            $data{'brands'}, 'edit');

        delete($data{'brands'});
    }

    $self->partner_db_table()->edit($block, \%data) if %data;

    $self->update_in_bk($block) if $self->need_update_in_bk(\@fields);

    return TRUE;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my %fields_need_update = map {$_ => TRUE} qw(
      brands
      dsp_blocks
      caption
      ), direct_block_fields(), direct_appearance_fields();

    return \%fields_need_update;
}

sub boolean_fields {
    return {map {$_ => TRUE} @BOOLEAN_FIELDS};
}

sub direct_block_fields {
    return qw(
      direct_block media_block limit
      );
}

sub direct_blocks {
    return \%DIRECT_BLOCKS;
}

sub media_blocks {
    return \%MEDIA_BLOCKS;
}

sub dsp_blocks {
    return \%DSP_BLOCKS;
}

sub fix_template { }

sub get_bk_block_data {
    my ($self, $block) = @_;

    my $bk_data = $self->get_bk_mobile_rtb_settings($block);

    $bk_data->{'AdFoxBlock'} = $block->{'adfox_block'};

    return $bk_data;
}

sub get_bk_block_data_key {'rtb_blocks'}

sub get_bk_common_data {
    return ();
}

sub get_cnt {
    my ($self, $arr_campaign_id) = @_;

    my $data = $self->partner_db->query->select(
        table  => $self->partner_db_table(),
        fields => {
            campaign_id => '',
            cnt         => {count => ['id']}
        },
        filter => [
            AND => [
                [campaign_id => 'IN'     => \$arr_campaign_id],
                [id          => '<>'     => \0],
                [multistate  => 'NOT IN' => \$self->get_multistates_by_filter('deleted')],
            ]
        ],
    )->group_by('campaign_id')->get_all();

    return {map {$_->{'campaign_id'} => $_->{'cnt'}} @$data};
}

sub _add {
    my ($self, %opts) = @_;

    my $block_id;
    $self->partner_db->transaction(
        sub {
            my $dsp_blocks = delete($opts{'dsp_blocks'});
            my $dsps       = delete($opts{'dsps'});
            my $brands     = delete($opts{'brands'});
            my $articles   = delete($opts{'articles'});

            $block_id = $self->partner_db_table()->add(
                {
                    id          => 1,
                    create_date => curdate(oformat => 'db_time'),
                    %opts
                }
            );

            if (defined($dsp_blocks)) {
                $self->app->partner_db->media_sizes->add_multi(
                    [map {page_id => $block_id->[0], block_id => $block_id->[1], type => $_,}, @$dsp_blocks,]);
            }

            $self->brands->replace({page_id => $block_id->[0], block_id => $block_id->[1]}, $brands, 'add');

            $self->articles->replace({page_id => $block_id->[0], block_id => $block_id->[1]}, $articles, 'add')
              if defined $articles;

            if (defined($dsps)) {
                $self->block_dsps->replace({page_id => $block_id->[0], block_id => $block_id->[1]}, $dsps, 'add');
            }

            $self->do_action($block_id, 'add', %opts);

            $self->do_action($block_id, 'start');

            $self->update_in_bk($block_id);
        }
    );

    return $block_id;
}

sub _trim_opts {
    my ($self, $opts) = @_;

    foreach my $field_name (qw(caption)) {
        $opts->{$field_name} =~ trim($opts->{$field_name}) if $opts->{$field_name};
    }
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_campaign($filter);

    return $filter;
}

sub get_filter_applicable_to_by_strategy {
    my ($self, $strategy_id) = @_;

    return ();
}

sub get_ad_type_list {
    return [qw(media text video)];
}

TRUE;
