package Application::Model::Product::Stripe;

use qbit;

use Exception::Validation::BadArguments;

use constant TURKEY_PLACE_ID => 956;

our $TYPE_SCROLL_OFF = 1;
our $TYPE_SCROLL_ON  = 4;

our $ANIMATION_OFF = 0;
our $ANIMATION_ON  = 1;

our $DISPOSITION_TOP    = 0;
our $DISPOSITION_BOTTOM = 1;

our $INTERACTION_COVERS_CONTENT = 0;
our $INTERACTION_PUSH_CONTENT   = 1;

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        caption => {
            db      => TRUE,
            label   => d_gettext('Block caption'),
            db_expr => 'title',
            api     => 1,
            type    => 'string',
        },
        title => {
            db         => TRUE,
            label      => d_gettext('Block title'),
            need_check => {
                optional => TRUE,
                len_max  => 255,
            },
            need_trim => TRUE,
            api       => 1,
            type      => 'string',
        },
        disposition_id => {
            db         => TRUE,
            label      => d_gettext('Disposition ID'),
            need_check => {
                in  => [sort keys %{$self->get_disposition_ids()}],
                msg => d_gettext('Invalid disposition'),
            },
            api         => 1,
            type        => 'number',
            adjust_type => 'str',
        },
        interaction_id => {
            db         => TRUE,
            label      => d_gettext('Interaction ID'),
            need_check => {
                in  => [sort keys %{$self->get_interaction_with_content_ids()}],
                msg => d_gettext('Invalid interaction'),
            },
            api         => 1,
            type        => 'number',
            adjust_type => 'str',
        },
        type_id => {
            db         => TRUE,
            label      => d_gettext('Stripe type ID'),
            need_check => {
                in  => [sort keys %{$self->get_type_ids()}],
                msg => d_gettext('Invalid type'),
            },
            api         => 1,
            type        => 'number',
            adjust_type => 'str',
        },
        animation => {
            db         => TRUE,
            label      => d_gettext('Animation ID'),
            need_check => {
                in  => [sort keys %{$self->get_animation_ids()}],
                msg => d_gettext('Invalid animation'),
            },
            api         => 1,
            type        => 'number',
            adjust_type => 'str',
        },
        disposition_name => {
            depends_on => [qw(disposition_id)],
            label      => d_gettext('Disposition'),
            get        => sub {
                $_[0]->{'disposition_ids'}->{$_[1]->{'disposition_id'}};
            },
            api  => 1,
            type => 'string',
        },
        interaction_name => {
            depends_on => [qw(interaction_id)],
            label      => d_gettext('Interaction'),
            get        => sub {
                $_[0]->{'interaction_ids'}->{$_[1]->{'interaction_id'}};
            },
            api  => 1,
            type => 'string',
        },
        type_name => {
            depends_on => [qw(type_id)],
            label      => d_gettext('Stripe type'),
            get        => sub {
                $_[0]->{'type_ids'}->{$_[1]->{'type_id'}};
            },
            api  => 1,
            type => 'string',
        },
        animation_name => {
            depends_on => [qw(animation)],
            label      => d_gettext('Animation'),
            get        => sub {
                $_[0]->{'animation_ids'}->{$_[1]->{'animation'}};
            },
            api  => 1,
            type => 'string',
        },
    };
}

sub get_animation_ids {
    return {$ANIMATION_OFF => gettext('No animation'), $ANIMATION_ON => gettext('With animation')};
}

sub get_disposition_ids {
    return {$DISPOSITION_TOP => gettext('Top'), $DISPOSITION_BOTTOM => gettext('Bottom')};
}

sub get_interaction_with_content_ids {
    return {
        $INTERACTION_PUSH_CONTENT   => gettext('Pushes content down'),
        $INTERACTION_COVERS_CONTENT => gettext('Covers content')
    };
}

sub get_type_ids {
    return {
        $TYPE_SCROLL_OFF => gettext('Scroll off'),
        $TYPE_SCROLL_ON  => gettext('Scroll on'),
    };
}

sub normalize_appearance {
    my ($self, $opts) = @_;

    if ($opts->{'disposition_id'} == $DISPOSITION_BOTTOM) {
        $opts->{'animation'} = $ANIMATION_OFF;
        $opts->{'type_id'}   = $TYPE_SCROLL_OFF;
    } elsif ($opts->{'interaction_id'} == $INTERACTION_PUSH_CONTENT) {
        $opts->{'animation'} = $ANIMATION_OFF;
        $opts->{'type_id'}   = $TYPE_SCROLL_OFF;
    }

    return FALSE;
}

sub fix_template {
    my ($self, $qv) = @_;

    my $template = $qv->template;

    push @{$template->{'all_set'}}, [qw(disposition_id interaction_id type_id animation)];

    $qv->template($template);
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    return $fields;
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        no_right_fields => [
            qw(
              animation
              campaign_id
              disposition_id
              interaction_id
              title
              type_id
              ),
        ],
        right_fields => {view => ['comment']}
    );

    return $fields;
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    $self->Application::Model::Common::hook_fields_processing_before_validation($opts);

    $opts->{'disposition_id'} //= 0;
    $opts->{'interaction_id'} //= 0;
    $opts->{'type_id'}        //= 1;
    $opts->{'animation'}      //= 0;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->Application::Model::Common::hook_set_initialize_settings($opts);

    $self->partner_db->transaction(
        sub {
            $opts->{id} = $self->page->get_next_block_id($opts->{$self->get_page_id_field_name()});
        }
    );

    $opts->{'title'} = gettext(
        'Stripe at the site %s',
        $self->page->get_all(
            fields => ['domain'],
            filter => {page_id => $opts->{$self->get_page_id_field_name()}},
          )->[0]{'domain'}
    );
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    my $id = $self->hook_stash->get('id');

    $self->maybe_do_action($id, 'start');

    $self->update_in_bk($id);
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    my $id = $self->hook_stash->get('id');

    $self->update_in_bk($id) if $self->need_update_in_bk($self->hook_stash->get('fields'));
}

sub can_action_duplicate {
    return FALSE;
}
#API

sub api_available_actions {
    return qw (edit delete restore duplicate);
}

sub api_can_edit {TRUE}

sub api_can_add {TRUE}

sub get_fields_depends {
    return {
        #если поменялось поле из ключа, то
        #нужно перезапросить поля из значения
        depends => {},
        #для поля из ключа обязятельно нужно передать поля из значения
        required => {},
    };
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = ();

    if ($need_fields->{'disposition_id'}) {
        $result{'disposition_id'} = $self->get_disposition_ids();
    }

    if ($need_fields->{'interaction_id'}) {
        $result{'interaction_id'} = $self->get_interaction_with_content_ids();
    }

    if ($need_fields->{'animation'}) {
        $result{'animation'} = $self->get_animation_ids();
    }

    if ($need_fields->{'type_id'}) {
        $result{'type_id'} = $self->get_type_ids();
    }

    return \%result;
}

sub get_available_fields_depends {
    [
        qw(
          id
          multistate
          disposition_id
          interaction_id
          )
    ];
}

TRUE;
