package Application::Model::Product::VideoAN::Site::Categories;

use qbit;

use Application::Model::DAC;
use base qw(
  Application::Model::DBManager::Base
  Application::Model::DAC
  );

use Exception::Validation::BadArguments;

sub accessor      {'video_an_site_categories'}
sub db_table_name {'video_an_site_categories'}

__PACKAGE__->model_accessors(
    partner_db    => 'Application::Model::PartnerDB',
    video_an_site => 'Application::Model::Page::Video',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'video_an_site_categories',
            description => d_gettext('Rights to use video an site categories'),
            rights      => {video_an_site_categories_view_all => d_gettext('Right to view all categories'),}
        }
    ]
);

__PACKAGE__->model_fields(
    id      => {default => TRUE, db => TRUE, pk => TRUE, type => 'number', api => 1, adjust_type => 'str',},
    page_id => {default => TRUE, db => TRUE, pk => TRUE, type => 'number', api => 1, adjust_type => 'str',},
    parent_id => {default => TRUE, db   => TRUE,     type => 'number', api  => 1,        adjust_type => 'str',},
    name      => {default => TRUE, db   => TRUE,     i18n => TRUE,     type => 'string', api         => 1},
    comment   => {db      => TRUE, i18n => TRUE,     type => 'string', api  => 1},
    link      => {db      => TRUE, type => 'string', api  => 1},
    archived => {db => TRUE, type => 'boolean', api => 1, adjust_type => 'str',},
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id        => {type => 'number',  label => d_gettext('ID')},
        page_id   => {type => 'number',  label => d_gettext('Page ID')},
        parent_id => {type => 'number',  label => d_gettext('Parent ID')},
        name      => {type => 'text',    label => d_gettext('Name')},
        comment   => {type => 'text',    label => d_gettext('Comment')},
        link      => {type => 'text',    label => d_gettext('Link')},
        archived  => {type => 'boolean', label => d_gettext('Archived')}
    },
);

sub replace {
    my ($self, $page_id, $data, $mode) = @_;

    my $locales = $self->get_option('locales', {});

    my %video_categories = map {$_->{'id'} => $_} @{$self->get_default_video_categories()};

    my %uniq_ids = ();
    foreach my $row (@$data) {
        $uniq_ids{$row->{'id'}}++;

        if (exists($video_categories{$row->{'id'}})) {
            foreach (qw(name comment)) {
                $row->{$_} = $video_categories{$row->{'id'}}->{$_};
            }
        } else {
            foreach my $field (qw(name comment)) {
                $row->{$field} = {map {$_ => $row->{$field}} keys(%$locales)};
            }
        }
    }

    $self->app->validator->check(
        data     => $data,
        template => {
            type => 'array',
            all  => {
                type   => 'hash',
                fields => {
                    id => {
                        type  => 'int_un',
                        check => sub {
                            return $uniq_ids{$_[1]} > 1 ? gettext('Duplicated id %s', $_[1]) : '';
                          }
                    },
                    parent_id => {
                        type     => 'int_un',
                        optional => TRUE,
                        check    => sub {
                            return $uniq_ids{$_[1]} ? '' : gettext('Unknown id %s', $_[1]);
                          }
                    },
                    link => {},
                    name => {
                        type   => 'hash',
                        fields => {
                            map {
                                $_ => {}
                              } keys(%$locales)
                        }
                    },
                    comment => {
                        type   => 'hash',
                        fields => {
                            map {
                                $_ => {optional => TRUE}
                              } keys(%$locales)
                        },
                    },
                    archived => {type => 'boolean'},
                },
            },
            check => sub {
                return $self->video_an_site->get($page_id, fields => [qw(id)])
                  ? ''
                  : gettext('Page "%s" not found', $page_id);
            },
        }
    );

    $self->partner_db->transaction(
        sub {
            $self->partner_db_table()
              ->delete($self->partner_db->filter({page_id => $page_id, id => [sort keys(%uniq_ids)]}))
              if $mode eq 'edit';

            $self->partner_db_table()->add_multi(
                [
                    map {
                        {%$_, page_id => $page_id}
                      } @$data
                ]
            );
        }
    );

    return TRUE;
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_special($filter, 'page_id', 'video_an_site', 'id');

    return $filter;
}

sub get_category_ids_for_yandex_video_search {qw(1000 1001)}

sub get_category_ids_for_yahosting {(1010 .. 1026)}

sub get_default_video_categories {
    my ($self, $domain) = @_;

    return [
        {
            id        => 0,
            parent_id => 0,
            name      => {
                ru => 'Основной видеоресурс',
                en => 'Domain of video resourse',
            },
            link     => $domain,
            comment  => {ru => '', en => ''},
            archived => 0,
        },
        {
            id        => 1000,
            parent_id => 0,
            name      => {
                ru => 'Я.Видео',
                en => 'Yandex.Video'
            },
            link    => 'https://www.yandex.ru/video/search',
            comment => {
                ru => 'Органическая выдача',
                en => 'Video search results'
            },
            archived => 0,
        },
        {
            id        => 1001,
            parent_id => 0,
            name      => {
                ru => 'Я.Видео Спецразмещение',
                en => 'Yandex.Video commercial'
            },
            link    => 'https://www.yandex.ru/video/search',
            comment => {
                ru => 'Спецразмещение',
                en => 'Video search commercial'
            },
            archived => 0,
        },
        {
            id        => 1010,
            parent_id => 0,
            name      => {
                ru => 'VH: Телепрограмма',
                en => 'VH: Yandex.TV',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1011,
            parent_id => 0,
            name      => {
                ru => 'VH: Главная',
                en => 'VH: Yandex.Main',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1012,
            parent_id => 0,
            name      => {
                ru => 'VH: Поиск',
                en => 'VH: Yandex.SERP',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1013,
            parent_id => 0,
            name      => {
                ru => 'VH: Я.Видео',
                en => 'VH: Yandex.Video',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1014,
            parent_id => 0,
            name      => {
                ru => 'VH: Кинопоиск',
                en => 'VH: KinoPoisk',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1015,
            parent_id => 0,
            name      => {
                ru => 'VH: Сайт Правообладателя',
                en => 'VH: Owners site',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1016,
            parent_id => 0,
            name      => {
                ru => 'VH: Zen',
                en => 'VH: Zen',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1017,
            parent_id => 0,
            name      => {
                ru => 'VH: Другое',
                en => 'VH: Other',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1018,
            parent_id => 0,
            name      => {
                ru => 'VH: Главная SmartTV',
                en => 'VH: Main SmartTV',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1019,
            parent_id => 0,
            name      => {
                ru => 'VH: Я.Видео SmartTV',
                en => 'VH: Yandex.Video SmartTV',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1020,
            parent_id => 0,
            name      => {
                ru => 'VH: Я.Музыка',
                en => 'VH: Yandex.Music',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1021,
            parent_id => 0,
            name      => {
                ru => 'VH: Я.Новости-Трагичное',
                en => 'VH: Yandex.News-tragic',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1022,
            parent_id => 0,
            name      => {
                ru => 'VH: Я.Новости',
                en => 'VH: Yandex.News',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1023,
            parent_id => 0,
            name      => {
                ru => 'VH: Турбо',
                en => 'VH: Turbo',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1024,
            parent_id => 0,
            name      => {
                ru => 'VH: Я.Спорт',
                en => 'VH: Yandex.Sport',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1025,
            parent_id => 0,
            name      => {
                ru => 'VH: Я.Погода',
                en => 'VH: Yandex.Weather',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
        {
            id        => 1026,
            parent_id => 0,
            name      => {
                ru => 'VH: Я.Зеркало',
                en => 'VH: Yandex.Mirror',
            },
            link    => $domain,
            comment => {
                ru => '',
                en => '',
            },
            archived => 0,
        },
    ];
}

sub get_position_type_by_category_id {
    my ($self, $category_id) = @_;

    throw Exception::Validation::BadArguments gettext('Expected category_id') unless defined($category_id);

    my %position_types = (1000 => 'organic', 1001 => 'commercial');

    throw gettext('Unknown category_id "%s"', $category_id) unless $position_types{$category_id};

    return $position_types{$category_id};
}

sub get_system_category_ids {
    map {$_->{id}} @{$_[0]->get_default_video_categories()};
}

sub is_yandex_video_search {
    my ($self, $category_id) = @_;

    throw Exception::Validation::BadArguments gettext('Expected category_id') unless defined($category_id);

    return !!grep {$_ == $category_id} $self->get_category_ids_for_yandex_video_search();
}

sub is_yahosting {
    my ($self, $category_id) = @_;

    throw Exception::Validation::BadArguments gettext('Expected category_id') unless defined($category_id);

    return !!grep {$_ == $category_id} $self->get_category_ids_for_yahosting();
}

TRUE;
