package Application::Model::Product::VideoAN::Site::InStream;

use qbit;

use base qw(Application::Model::Block::Video);

use PiConstants qw(
  $PREROLL
  $POSTROLL
  $MIDROLL
  $MIDROLL_MIN_START_TIME
  $PAUSE
  $INROLL
  $S2SROLL
  $OVERLAY
  $INROLL_OVERLAY
  $OVERLAY_TYPES
  );

consume qw(
  Application::Model::Role::Block::Has::AdfoxBlock
  Application::Model::Role::Block::Has::MoneyMap
  Application::Model::Role::Has::CustomBKOptions
  Application::Model::Role::Has::MidRoll
  Application::Model::Role::Has::Jingles
  );

use Exception::Validator::Fields;
use Exception::Validation::BadArguments;

my $MAX_COUNT_POSITIONS = 30;

sub accessor      {'video_an_site_instream'}
sub db_table_name {'video_an_site_instream'}

sub get_opts_schema_name {'video_an_site_instream_opts'}

sub get_product_name {gettext('video_an_site_instream')}

sub public_id_prefix {'R-V-'}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {video_scenaries => 'Application::Model::VideoScenaries',};
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{rights} = {
        %{$rights->[0]{rights}},
        map {$self->get_description_right($_)}
          qw(
          edit_field__allow_multiple_dsp_ads
          view_field__allow_multiple_dsp_ads
          )
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},

        allow_multiple_dsp_ads => {
            db         => TRUE,
            label      => d_gettext('Allow Multiple DSP ads'),
            type       => 'boolean',
            api        => 1,
            need_check => {
                type     => 'boolean',
                optional => TRUE,
                check    => sub {
                    my ($qv, $allow_multiple_dsp_ads) = @_;

                    throw Exception::Validator::Fields gettext("Can't create adfox block with allow_multiple_dsp_ads")
                      if $qv->{'data'}{'adfox_block'} && $allow_multiple_dsp_ads;
                  }
            },
            adjust_type => 'str',
        },
        vmap_id => {
            db         => TRUE,
            label      => d_gettext('Scenario ID'),
            type       => 'number',
            api        => 1,
            need_check => {
                type  => 'int_un',
                check => sub {
                    my ($qv, $vmap_id) = @_;

                    if ($qv->app->check_feature_on_page($qv->{data}{page_id}, 'instream_vmap')) {
                        unless ($vmap_id) {
                            throw Exception::Validator::Fields gettext('Key "%s" required', 'vmap_id');
                        }
                    }
                  }
            },
            adjust_type => 'str',
        },
        stopped_in_vmap => {
            from_opts  => 'from_hash',
            label      => d_gettext('Stopped in vmap'),
            type       => 'boolean',
            need_check => {
                optional => TRUE,
                check    => sub {
                    my ($qv, $stopped_in_vmap) = @_;
                    throw Exception::Validator::Fields gettext('Key "%s" is not valid', 'stopped_in_vmap')
                      if $stopped_in_vmap
                          && $qv->app->check_multistate_flag($qv->data->{'multistate'}, 'working');
                  }
            },
            api         => 1,
            adjust_type => 'str',
        },
        # redefine
        count_positions => {
            db         => TRUE,
            label      => d_gettext('Count positions'),
            type       => 'number',
            need_check => {
                optional => TRUE,
                min      => 1,
                max      => $MAX_COUNT_POSITIONS,
                msg      => gettext('Field "count_positions" must be in range [%s,%s]', 1, $MAX_COUNT_POSITIONS)
            },
            api         => 1,
            adjust_type => 'str',
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $parent_filter = $self->SUPER::get_structure_model_filter();
    return {
        db_accessor => $parent_filter->{'db_accessor'},
        fields      => {
            %{$parent_filter->{'fields'}},
            vmap_id => {
                type  => 'number',
                label => d_gettext('Scenario ID'),
            }
        }
    };
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    my $base_graph = $self->SUPER::get_structure_multistates_graph();

    $base_graph->{actions} = {
        %{$base_graph->{actions}},
        delete_with_vmap  => d_pgettext('Block action', 'Archive with vmap'),
        restore_with_vmap => d_pgettext('Block action', 'Restore with vmap'),
        start_in_vmap     => d_pgettext('Block action', 'Start in vmap'),
        stop_in_vmap      => d_pgettext('Block action', 'Stop in vmap'),
    };
    push @{$base_graph->{multistates}},
      [
        deleted_with_vmap => d_pgettext('Block multistate', 'Archived with vmap'),
        private           => TRUE
      ];
    push @{$base_graph->{multistate_actions}},
      {
        action      => 'stop_in_vmap',
        from        => 'not deleted and working',
        reset_flags => ['working'],
      },
      {
        action    => 'start_in_vmap',
        from      => 'not deleted and not working',
        set_flags => ['working'],
      },
      {
        action    => 'delete_with_vmap',
        from      => 'not deleted',
        set_flags => ['deleted', 'deleted_with_vmap'],
      },
      {
        action      => 'restore_with_vmap',
        from        => 'deleted and deleted_with_vmap',
        reset_flags => ['deleted', 'deleted_with_vmap'],
      };

    return $base_graph;
}

sub get_actions_depends {
    my ($self) = @_;
    return [@{$self->SUPER::get_actions_depends()}, 'stopped_in_vmap'];
}

sub get_available_fields_depends {
    my ($self, $obj) = @_;

    return [@{$self->SUPER::get_available_fields_depends($obj) // []}, ('page')];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $fields = $self->SUPER::get_available_fields($obj);

    delete($fields->{'opts'});

    $self->app->delete_field_by_rights($fields,
        {$self->get_right('view_field__allow_multiple_dsp_ads') => 'allow_multiple_dsp_ads',});

    $fields->{vmap_id} = TRUE if grep {'instream_vmap' eq $_} @{$obj->{page}{owner}{features}};

    return $fields;
}

sub get_add_fields {
    my ($self, $obj) = @_;

    my $fields = $self->SUPER::get_add_fields();

    $fields->{count_positions} = TRUE;

    $fields->{'allow_multiple_dsp_ads'} = TRUE if $self->check_short_rights("edit_field__allow_multiple_dsp_ads");
    $fields->{vmap_id} = TRUE
      if ($self->check_feature_on_page($obj->{page_id}, 'instream_vmap') && !$obj->{adfox_block});

    return $fields;
}

sub collect_editable_fields {
    my ($self, $data) = @_;

    if ($data->{type} && $OVERLAY_TYPES->{$data->{type}}) {
        return {} unless $self->check_rights('video_an_site_instream_can_add_overlay');
    }

    my $res = $self->SUPER::collect_editable_fields($data);

    $res->{count_positions} = TRUE;

    unless ($data->{'adfox_block'}) {
        $res->{'allow_multiple_dsp_ads'} = TRUE if $self->check_short_rights("edit_field__allow_multiple_dsp_ads");
    }

    $res->{'stopped_in_vmap'} = TRUE if $self->check_short_rights("edit_field__stopped_in_vmap");
    return $res;
}

sub on_action_delete_with_vmap {
    my ($self, $block) = @_;

    $self->maybe_do_action($block, 'stop');
    $self->update_in_bk($block);
}

sub on_action_restore_with_vmap {
    my ($self, $block) = @_;

    $self->maybe_do_action($block, 'start');
    $self->update_in_bk($block);
}

sub can_action_start {
    my ($self, $block) = @_;
    my $obj = $self->_get_object_fields($block, [qw(stopped_in_vmap)]);
    return $self->SUPER::can_action_start($block) && !$obj->{stopped_in_vmap};
}

sub can_action_start_in_vmap {
    my ($self, $block) = @_;
    my $obj = $self->_get_object_fields($block, [qw(stopped_in_vmap)]);
    return $self->SUPER::can_action_start($block) && $obj->{stopped_in_vmap};
}

sub on_action_start_in_vmap {
    my ($self, $block) = @_;

    my $tmp_rights = $self->app->add_tmp_rights('video_an_site_instream_edit_field__stopped_in_vmap');
    $self->partner_db_table()->edit($block, {opts => {json_set => ['opts', \'$.stopped_in_vmap', \0,]}});
    $self->update_in_bk($block);
}

sub can_action_stop_in_vmap {
    my ($self, $block) = @_;
    return !$block->{stopped_in_vmap};
}

sub on_action_stop_in_vmap {
    my ($self, $block) = @_;

    my $tmp_rights = $self->app->add_tmp_rights('video_an_site_instream_edit_field__stopped_in_vmap');
    $self->partner_db_table()->edit($block, {opts => {json_set => ['opts', \'$.stopped_in_vmap', \1,]}});
    $self->update_in_bk($block);
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my $result = $self->SUPER::get_need_update_in_bk_fields();
    $result->{$_} = TRUE foreach (qw(allow_multiple_dsp_ads count_positions));

    return $result;
}

sub get_types_ids {
    [$PREROLL, $POSTROLL, $MIDROLL, $PAUSE, $INROLL, $S2SROLL, $OVERLAY, $INROLL_OVERLAY];
}

sub get_multistate_filter_for_update_in_bk {'working and not deleted'}

sub get_bk_block_data {
    my ($self, $block) = @_;

    my $bk_data = $self->SUPER::get_bk_block_data($block);

    if (exists($block->{'allow_multiple_dsp_ads'})) {
        my $enable_disable = $block->{'allow_multiple_dsp_ads'} ? 'Enable' : 'Disable';

        $bk_data->{'PageImpOptions'}{$_} //= [] foreach (qw(Enable Disable));

        push @{$bk_data->{'PageImpOptions'}{$enable_disable}}, 'allow-multiple-dsp-ads';
    }

    if ($self->check_feature_on_page($block->{page_id}, 'instream_vmap')) {
        $bk_data->{'VmapID'} = $block->{'vmap_id'};
    }

    return $bk_data;
}

sub hook_fields_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_validation($opts);

    if ($opts->{vmap_id}) {
        my $vmap = $self->video_scenaries->get_all(
            fields => ['for_adfox'],
            filter => {
                page_id    => $opts->{page_id},
                id         => $opts->{vmap_id},
                multistate => 'not deleted',
            }
        )->[0];

        throw Exception::Validation::BadArguments gettext('Scenary not found') unless $vmap;

        throw Exception::Validation::BadArguments gettext('Cannot adfox block for not adfox scenary')
          if $opts->{adfox_block} && !$vmap->{for_adfox};

        throw Exception::Validation::BadArguments gettext('Cannot block for adfox scenary')
          if !$opts->{adfox_block} && $vmap->{for_adfox};
    }
}

sub hook_set_initialize_settings {
    my ($self, $obj, %opts) = @_;

    $self->SUPER::hook_set_initialize_settings($obj, %opts);

    # если адфокс то создавать специальный сценарий
    my $is_adfox = $obj->{'adfox_block'} ? 1 : 0;
    unless ($self->check_feature_on_page($obj->{page_id}, 'instream_vmap') && !$is_adfox) {
        $obj->{'vmap_id'} //= $self->video_scenaries->get_or_create_default_scenario($obj->{'page_id'}, $is_adfox);
    }

    $obj->{'allow_multiple_dsp_ads'} //= $is_adfox ? 0 : 1;
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my $result = $self->SUPER::make_fields_defaults($opts, $need_fields);

    if ($need_fields->{count_positions}) {
        $result->{count_positions} = [1 .. $MAX_COUNT_POSITIONS];
    }

    return $result;
}

sub get_dsp_rules {
    my ($self) = @_;

    return [
        @{$self->SUPER::get_dsp_rules()},
        # overlay dsps validation
        get_midroll_dsp_rule(),
    ];
}

sub api_available_actions {
    my ($self) = @_;
    return ($self->SUPER::api_available_actions(), 'start_in_vmap', 'stop_in_vmap');
}
TRUE;
