package Application::Model::Product::VideoAN::Site::Video::Blockings;

use qbit;

use base qw(Application::Model::DBManager::Base);

use Exception::Validation::BadArguments::InvalidBlocking;
use Exception::DB::DuplicateEntry;

my %TYPES = (all_content => d_gettext('All content'),);

sub accessor      {'video_an_site_blockings'}
sub db_table_name {'video_an_site_blockings'}

__PACKAGE__->model_accessors(partner_db => 'Application::Model::PartnerDB::VideoAN',);

__PACKAGE__->model_fields(
    site_id    => {default => TRUE, db => TRUE, pk => TRUE},
    content_id => {default => TRUE, db => TRUE, pk => TRUE, type => 'number', api => 1},
    type       => {default => TRUE, db => TRUE, pk => TRUE, type => 'string', api => 1},
    name => {default => TRUE, db => TRUE, type => 'string', api => 1},
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        site_id    => {type => 'number', label => d_gettext('Site ID')},
        content_id => {type => 'number', label => d_gettext('Content ID')},
        type       => {
            type   => 'dictionary',
            label  => gettext('Types'),
            values => sub {
                [map {{id => $_, label => $TYPES{$_}->()}} keys(%TYPES)];
              }
        },
        name => {type => 'text', label => d_gettext('Name')},
    }
);

sub add {
    my ($self, %opts) = @_;

    my @caller = caller(1);

    throw gettext('Cannot call method "add" from "%s". Use Video site action "add_blocking".', $caller[3])
      unless $caller[3] eq 'Application::Model::Page::Video::on_action_add_blocking';

    my @bad_fields =
      grep {!defined($opts{$_}) || $opts{$_} =~ /^\s*$/} qw(site_id content_id type name);

    throw Exception::Validation::BadArguments::InvalidBlocking gettext('The following fields: %s are required',
        join(', ', @bad_fields))
      if @bad_fields;

    $opts{'name'} =~ /^\s+|\s+$/g;

    throw Exception::Validation::BadArguments::InvalidBlocking gettext('content_id must be whole number')
      unless $opts{'content_id'} =~ /^[0-9]+$/;

    throw Exception::Validation::BadArguments::InvalidBlocking gettext('Unknown type: %s', $opts{'type'})
      unless exists($TYPES{$opts{'type'}});

    try {
        $self->partner_db_table()->add({hash_transform(\%opts, [qw(site_id content_id type name)])});
    }
    catch Exception::DB::DuplicateEntry with {
        throw Exception::Validation::BadArguments::InvalidBlocking gettext('This blocking already exists');
    };
}

sub delete {
    my ($self, %opts) = @_;

    my @caller = caller(1);

    throw gettext('Cannot call method "delete" from "%s". Use Video site action "del_blocking".', $caller[3])
      unless $caller[3] eq 'Application::Model::Page::Video::on_action_del_blocking';

    throw Exception::Validation::BadArguments::InvalidBlocking gettext('The following fields: %s are required',
        join(', ', grep {!exists($opts{$_})} qw(site_id content_id type)))
      if grep {!exists($opts{$_})} qw(site_id content_id type);

    $self->partner_db_table()->delete({hash_transform(\%opts, [qw(site_id content_id type)])});
}

TRUE;
