package Application::Model::Product::VideoAN::Site::Video::InternalContent;

use qbit;

use base qw(Application::Model::Common);

use Exception::Denied;
use Exception::Validation::BadArguments;

sub accessor      {'internal_content'}
sub db_table_name {'internal_content'}

__PACKAGE__->model_accessors(
    partner_db       => 'Application::Model::PartnerDB::VideoAN',
    partner_content  => 'Application::Model::Product::VideoAN::Site::Video::PartnerContent',
    internal_genre   => 'Application::Model::Product::VideoAN::Site::Video::InternalGenre',
    dictionary_words => 'Application::Model::Product::VideoAN::Site::Video::InternalContent::DictionaryWords',
    content_type     => 'Application::Model::Product::VideoAN::Site::Video::InternalContent::ContentType',
    content_brand    => 'Application::Model::Product::VideoAN::Site::Video::InternalContent::ContentBrand',
    internal_content_genres =>
      'Application::Model::Product::VideoAN::Site::Video::InternalContent::InternalContentGenres',
    video_an_site => 'Application::Model::Page::Video',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'internal_content',
            description => d_gettext('Right to manage internal content'),
            rights      => {
                internal_content_add                  => d_gettext('Right to add internal content'),
                internal_content_view_all             => d_gettext('Right to view all internal contents'),
                internal_content_edit                 => d_gettext('Right to edit internal content'),
                internal_content_view_action_log      => d_gettext('Right to view action logs'),
                internal_content_edit_field__type_id  => d_gettext('Right to edit field "type_id"'),
                internal_content_edit_field__brand_id => d_gettext('Right to edit field "brand_id"'),
            }
        }
    ]
);

__PACKAGE__->model_fields(
    id => {
        default => TRUE,
        db      => TRUE,
        pk      => TRUE,
        type    => 'number',
        label   => d_gettext('Internal content ID'),
        hint    => d_gettext('The unique content identifier')
    },
    create_date => {db => TRUE, label   => d_gettext('Create date')},
    name        => {db => TRUE, default => TRUE, label => d_gettext('Name')},
    type_id     => {db => TRUE, default => TRUE, label => d_gettext('Type id')},
    brand_id    => {db => TRUE, default => TRUE, label => d_gettext('Brand id')},
    publisher   => {db => TRUE, default => TRUE, label => d_gettext('Publisher')},
    multistate  => {db => TRUE, label   => d_gettext('Status (raw)')},
    multistate_name => {
        label      => d_gettext('Status'),
        depends_on => ['multistate'],
        get        => sub {
            $_[0]->model->get_multistate_name($_[1]->{'multistate'});
        },
    },
    actions => {
        label      => d_gettext('Actions'),
        depends_on => [qw(id multistate)],
        get        => sub {
            $_[0]->model->get_actions($_[1]);
          }
    },
    editable_fields => {
        depends_on => [qw(id multistate)],
        get        => sub {
            return {}
              unless $_[0]->model->check_action($_[1], 'edit');

            my %res = map {$_ => 1} qw(
              name
              );

            foreach (qw(type_id brand_id)) {
                $res{$_} = TRUE
                  if $_[0]->model->check_rights('internal_content_edit_field__' . $_);
            }

            return \%res;
          }
    },
    available_fields => {
        depends_on => [qw(multistate)],
        label      => d_gettext('Available fields'),
        get        => sub {
            return $_[0]->model->get_available_fields($_[1]);
          }
    },
    partner_contents => {
        depends_on => [qw(id)],
        get        => sub {
            return $_[0]->{'__PARTNER_CONTENTS__'}{$_[1]->{'id'}} || [];
          }
    },
    type => {
        depends_on => [qw(type_id)],
        get        => sub {
            return $_[0]->{'__TYPES__'}{$_[1]->{'type_id'}} || '';
          }
    },
    brand => {
        depends_on => [qw(brand_id)],
        get        => sub {
            return $_[1]->{'brand_id'} ? $_[0]->{'__BRANDS__'}{$_[1]->{'brand_id'}} || '' : '';
          }
    },
    genres => {
        depends_on => [qw(id)],
        get        => sub {
            return $_[0]->{'__GENRES__'}{$_[1]->{'id'}} || [];
          }
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id               => {type => 'number',     label => d_gettext('Internal content ID'), pk => TRUE},
        name             => {type => 'text',       label => d_gettext('Name')},
        type_id          => {type => 'number',     label => d_gettext('Type ID'),},
        brand_id         => {type => 'number',     label => d_gettext('Brand ID'),},
        multistate       => {type => 'multistate', label => d_gettext('Status')},
        publisher        => {type => 'text',       label => d_gettext('Publisher')},
        dictionary_words => {
            type           => 'subfilter',
            model_accessor => 'dictionary_words',
            field          => 'id',
            fk_field       => 'content_id',
            label          => d_gettext('Dictionary words'),
        },
        type => {
            type      => 'dictionary',
            db_filter => sub {
                $_[0]->{'__DB_FILTER__'}{$_[1]->[0]}->as_filter(['type_id' => $_[1]->[1] => $_[1]->[2]], $_[2]);
            },
            label  => gettext('Type'),
            values => sub {
                return [map {{id => $_->{'id'}, label => $_->{'caption'}}} @{$_[0]->content_type->get_all()}];
              }
        },
        brand => {
            type           => 'subfilter',
            model_accessor => 'content_brand',
            field          => 'brand_id',
            fk_field       => 'id',
            label          => d_gettext('Brand'),
        },
        partner_content => {
            type           => 'subfilter',
            model_accessor => 'partner_content',
            field          => 'id',
            fk_field       => 'internal_id',
            label          => d_gettext('Partner content'),
        },
        internal_content_genres => {
            type           => 'subfilter',
            model_accessor => 'internal_content_genres',
            field          => 'id',
            fk_field       => 'content_id',
            label          => d_gettext('Internal content genres'),
        }
    }
);

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [
            {name => 'id',                                   label => gettext('Internal content ID')},
            {name => 'name',                                 label => gettext('Caption')},
            {name => 'partner_content.public_id',            label => gettext('Partner content ID')},
            {name => 'partner_content.name',                 label => gettext('Partner content caption')},
            {name => 'partner_content.site_id',              label => gettext('Page ID')},
            {name => 'partner_content.video_an_site.domain', label => gettext('Domain')},
        ],
        [
            {name => 'internal_content_genres.internal_genre.public_id', label => gettext('Genre ID')},
            {name => 'internal_content_genres.internal_genre.name',      label => gettext('Genre name')},
            {name => 'type',                                             label => gettext('Type')},
            {name => 'brand.caption',                                    label => gettext('Brand')},
            {name => 'publisher',                                        label => gettext('Publisher')},
        ],
    ];
}

__PACKAGE__->multistates_graph(
    empty_name    => 'New',
    multistates   => [[deleted => d_pgettext('Content status', 'Deleted')],],
    actions       => {},
    right_actions => {
        delete  => d_pgettext('Content action', 'Delete'),
        restore => d_pgettext('Content action', 'Restore'),
        edit    => d_pgettext('Content action', 'Edit'),
    },
    right_group        => [internal_content => d_gettext('Right to manage internal content')],
    right_name_prefix  => 'internal_content_',
    multistate_actions => [
        {
            action    => 'delete',
            from      => 'not deleted',
            set_flags => ['deleted']
        },
        {
            action      => 'restore',
            from        => 'deleted',
            reset_flags => ['deleted']
        },
        {
            action => 'edit',
            from   => 'not deleted'
        },
    ]
);

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    my $ids = array_uniq(map {$_->{'id'} // ()} @$result);

    if ($fields->need('partner_contents')) {
        $fields->{'__PARTNER_CONTENTS__'} = {};

        foreach (
            @{
                $self->partner_content->get_all(
                    fields => [qw(id public_id name internal_id domain site_id)],
                    filter => {internal_id => array_uniq(map {$_->{'id'} // ()} @$result)}
                )
            }
          )
        {
            my $internal_id = delete($_->{'internal_id'});
            $fields->{'__PARTNER_CONTENTS__'}{$internal_id} //= [];
            push(@{$fields->{'__PARTNER_CONTENTS__'}{$internal_id}}, $_);
        }
    }

    if ($fields->need('type')) {
        $fields->{'__TYPES__'} =
          {map {$_->{'id'} => $_->{'caption'}} @{$self->content_type->get_all(fields => [qw(id caption)])}};
    }

    if ($fields->need('brand')) {
        $fields->{'__BRANDS__'} =
          {map {$_->{'id'} => $_->{'caption'}} @{$self->content_brand->get_all(fields => [qw(id caption)])}};
    }

    if ($fields->need('genres')) {
        $fields->{'__GENRES__'} = {};

        foreach (
            @{
                $self->internal_genre->query(fields => $self->internal_genre->_get_fields_obj([qw(id public_id name)]))
                  ->join(
                    table   => $self->partner_db->internal_content_genres,
                    fields  => [qw(content_id)],
                    filter  => {content_id => $ids},
                    join_on => [genre_id => '=' => {id => $self->partner_db->internal_genre}],
                  )->get_all()
            }
          )
        {
            my $content_id = delete($_->{'content_id'});
            $fields->{'__GENRES__'}{$content_id} //= [];
            push(@{$fields->{'__GENRES__'}{$content_id}}, $_);
        }
    }
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied unless $self->check_short_rights('add');

    my @require_fields = qw(name type_id genres);
    throw Exception::Validation::BadArguments gettext('Expected fields: %s',
        join(', ', grep {!defined($opts{$_})} @require_fields))
      if grep {!defined($opts{$_})} @require_fields;

    $self->_trim_opts(\%opts);
    $self->_check_opts(\%opts);

    my $genres = delete($opts{'genres'});

    throw Exception::Validation::BadArguments gettext('Expected genres') unless @$genres;

    my $id;
    $self->partner_db->transaction(
        sub {
            my $brand_id;
            if ($opts{'brand'}) {
                $brand_id =
                  $self->content_brand->get_all(fields => ['id'], filter => {caption => $opts{'brand'}})->[0]{'id'};
                $brand_id = $self->content_brand->add($opts{'brand'}) unless $brand_id;
            }

            $id = $self->partner_db_table()->add(
                {
                    create_date => curdate(oformat => 'db_time'),
                    brand_id    => $brand_id,
                    hash_transform(\%opts, [qw(name type_id publisher referer)])
                }
            );

            $self->partner_db->internal_content_genres->add_multi([map {{content_id => $id, genre_id => $_}} @$genres])
              if @$genres;
        }
    );

    return $id;
}

sub on_action_edit {
    my ($self, $obj, %opts) = @_;

    my $fields = $self->get($obj->{'id'}, fields => ['editable_fields'])->{'editable_fields'};

    my @bad_fields = grep {!$fields->{$_}} keys(%opts);
    throw Exception::Validation::BadArguments gettext('You can not edit the following fields: %s',
        join(', ', @bad_fields))
      if @bad_fields;

    $self->_trim_opts(\%opts);

    $self->partner_db_table()->edit($obj->{'id'}, \%opts) if keys(%opts);
}

sub query_filter {
    my ($self, $filter) = @_;

    throw Exception::Denied unless $self->check_short_rights('view_all');

    return $filter;
}

sub check_action {
    my ($self, $object, $action) = @_;

    return FALSE
      unless $self->check_short_rights('edit');

    $self->SUPER::check_action($object, $action);
}

sub get_internal_contents_for_hint {
    my ($self, $string, %opts) = @_;

    $string =~ s/[^a-zA-Zа-яА-Я0-9 ]/ /g;

    my @words = split(/\s+/, $string);

    my $query = $self->query(fields => $self->_get_fields_obj([qw(id)]))->join(
        table  => $self->partner_db->dictionary_words,
        fields => {cnt => {count => ['content_id']},},
        filter => ['OR', [map {['word', '=', \$_]} @words]],
        join_on => ['content_id' => '=' => {id => $self->partner_db_table()}]
    )->group_by('id', 'name')->order_by(['cnt', 1]);

    $query->limit(delete($opts{'limit'})) if defined($opts{'limit'});

    my $count;
    my %sort_hash = map {$_->{'id'} => $count++} @{$query->get_all()};

    my $filter = ['id' => 'IN' => [sort keys(%sort_hash)]];
    if (exists($opts{'filter'}) && $opts{'filter'}) {
        $filter = ['AND', [$filter, $opts{'filter'}]];
    }

    my $data = $self->get_all_with_meta(%opts, filter => $filter);

    $data->{'data'} = [sort {$sort_hash{$a->{'id'}} <=> $sort_hash{$b->{'id'}}} @{$data->{'data'}}]
      if @{$data->{'data'}};

    return $data;
}

sub _check_opts {
    my ($self, $opts) = @_;

    if ($opts->{'type_id'}) {
        my $type = $self->content_type->get($opts->{'type_id'});

        throw Exception::Validation::BadArguments gettext('Unknown content type: %s', $opts->{'type_id'})
          unless $type;
    }

    if (exists($opts->{'genres'})) {
        my $internal_genres = $self->internal_genre->get_all(filter => {id => $opts->{'genres'}});

        throw Exception::Validation::BadArguments gettext('Unknown genres with id: %s',
            join(', ', @{arrays_difference($opts->{'genres'}, [map {$_->{'id'}} @$internal_genres])}))
          unless @{$opts->{'genres'}} == @$internal_genres;
    }
}

sub _trim_opts {
    my ($self, $opts) = @_;

    my @fields = keys(%{$self->get_model_fields()});

    push(@fields, qw());

    foreach (@fields) {
        $_ =~ s/^\s*|\s*$//g if exists($opts->{$_}) && !ref($opts->{$_});
    }
}

TRUE;
