package Application::Model::Product::VideoAN::Site::Video::InternalGenre;

use qbit;

use base qw(Application::Model::Common);

use Exception::DB::DuplicateEntry;
use Exception::Denied;
use Exception::Validation::BadArguments;

sub accessor      {'internal_genre'}
sub db_table_name {'internal_genre'}

sub get_product_name {gettext('internal_genre')}

__PACKAGE__->model_accessors(
    partner_db                 => 'Application::Model::PartnerDB::VideoAN',
    partner_genre              => 'Application::Model::Product::VideoAN::Site::Video::PartnerGenre',
    partner_and_internal_genre => 'Application::Model::Product::VideoAN::Site::Video::PartnerAndInternalGenre',
    video_an_site              => 'Application::Model::Page::Video',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'internal_genre',
            description => d_gettext('Right to manage internal genre'),
            rights      => {
                internal_genre_add             => d_gettext('Right to add internal genre'),
                internal_genre_view_all        => d_gettext('Right to view all internal genres'),
                internal_genre_edit            => d_gettext('Right to edit internal genre'),
                internal_genre_view_action_log => d_gettext('Right to view action logs'),
            }
        }
    ]
);

__PACKAGE__->model_fields(
    id => {
        default => TRUE,
        db      => TRUE,
        pk      => TRUE,
        type    => 'number',
        label   => d_gettext('Internal genre ID'),
        hint    => d_gettext('The unique genre identifier')
    },
    create_date => {db => TRUE, label   => d_gettext('Create date')},
    public_id   => {db => TRUE, default => TRUE, label => d_gettext('Genre ID')},
    name        => {db => TRUE, default => TRUE, label => d_gettext('Genre name')},
    multistate  => {db => TRUE, label   => d_gettext('Status (raw)')},
    multistate_name => {
        label      => d_gettext('Status'),
        depends_on => ['multistate'],
        get        => sub {
            $_[0]->model->get_multistate_name($_[1]->{'multistate'});
        },
    },
    actions => {
        label      => d_gettext('Actions'),
        depends_on => [qw(id multistate)],
        get        => sub {
            $_[0]->model->get_actions($_[1]);
          }
    },
    editable_fields => {
        depends_on => [qw(id multistate)],
        get        => sub {
            return {}
              unless $_[0]->model->check_action($_[1], 'edit');

            my %res = map {$_ => 1} qw(
              public_id
              name
              );

            foreach (qw()) {
                $res{$_} = TRUE
                  if $_[0]->model->check_rights('internal_genre_edit_field__' . $_);
            }

            return \%res;
          }
    },
    available_fields => {
        depends_on => [qw(multistate)],
        label      => d_gettext('Available fields'),
        get        => sub {
            return $_[0]->model->get_available_fields($_[1]);
          }
    },
    partner_genres => {
        depends_on => [qw(id)],
        get        => sub {
            return $_[0]->{'__PARTNER_GENRES__'}{$_[1]->{'id'}} || [];
          }
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id                         => {type => 'number',     label => d_gettext('Internal genre ID'), pk => TRUE},
        public_id                  => {type => 'text',       label => d_gettext('Genre ID')},
        name                       => {type => 'text',       label => d_gettext('Genre name')},
        multistate                 => {type => 'multistate', label => d_gettext('Status')},
        partner_and_internal_genre => {
            type           => 'subfilter',
            model_accessor => 'partner_and_internal_genre',
            field          => 'id',
            fk_field       => 'internal_id',
            label          => d_gettext('Partner and internal genre'),
        }
    }
);

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [
            {name => 'id',        label => gettext('Internal genre ID')},
            {name => 'public_id', label => gettext('Genre ID')},
            {name => 'name',      label => gettext('Genre name')}
        ],
        [
            {name => 'partner_and_internal_genre.partner_genre.name',        label => gettext('Partner genre name')},
            {name => 'partner_and_internal_genre.partner_genre.site_id',     label => gettext('Page ID')},
            {name => 'partner_and_internal_genre.partner_genre.site.domain', label => gettext('Domain')},
        ],
    ];
}

__PACKAGE__->multistates_graph(
    empty_name    => 'New',
    multistates   => [[deleted => d_pgettext('Genre status', 'Deleted')],],
    actions       => {},
    right_actions => {
        delete  => d_pgettext('Genre action', 'Delete'),
        restore => d_pgettext('Genre action', 'Restore'),
        edit    => d_pgettext('Genre action', 'Edit'),
    },
    right_group        => [internal_genre => d_gettext('Right to manage internal genre')],
    right_name_prefix  => 'internal_genre_',
    multistate_actions => [
        {
            action    => 'delete',
            from      => 'not deleted',
            set_flags => ['deleted']
        },
        {
            action      => 'restore',
            from        => 'deleted',
            reset_flags => ['deleted']
        },
        {
            action => 'edit',
            from   => 'not deleted'
        },
    ]
);

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    my $ids = array_uniq(map {$_->{'id'} // ()} @$result);

    if ($fields->need('partner_genres')) {
        $fields->{'__PARTNER_GENRES__'} = {};

        my $data = $self->partner_genre->query(fields => $self->partner_genre->_get_fields_obj([qw(id)]))->join(
            table   => $self->partner_db->partner_and_internal_genre,
            fields  => [qw(internal_id)],
            filter  => {internal_id => $ids},
            join_on => [partner_id => '=' => {id => $self->partner_db->partner_genre}],
        )->get_all();

        my %hash_data = map {$_->{'id'} => $_} @{
            $self->partner_genre->get_all(
                fields => [qw(id public_id name site_id domain)],
                filter => {id => array_uniq(map {$_->{'id'}} @$data)}
            )
          };

        foreach (@$data) {
            $fields->{'__PARTNER_GENRES__'}{$_->{'internal_id'}} //= [];
            push(@{$fields->{'__PARTNER_GENRES__'}{$_->{'internal_id'}}}, $hash_data{$_->{'id'}});
        }
    }
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied unless $self->check_short_rights('add');

    my @require_fields = qw(public_id name);
    throw Exception::Validation::BadArguments gettext('Expected fields: %s',
        join(', ', grep {!$opts{$_}} @require_fields))
      if grep {!$opts{$_}} @require_fields;

    $self->_trim_opts(\%opts);

    my $id;
    try {
        $id = $self->partner_db_table()->add(
            {
                create_date => curdate(oformat => 'db_time'),
                hash_transform(\%opts, [qw(public_id name)])
            }
        );
    }
    catch Exception::DB::DuplicateEntry with {
        throw Exception::Validation::BadArguments gettext('Genre with this ID = %s already exists', $opts{'public_id'});
    };

    return $id;
}

sub on_action_edit {
    my ($self, $obj, %opts) = @_;

    my $fields = $self->get($obj->{'id'}, fields => ['editable_fields'])->{'editable_fields'};

    my @bad_fields = grep {!$fields->{$_}} keys(%opts);
    throw Exception::Validation::BadArguments gettext('You can not edit the following fields: %s',
        join(', ', @bad_fields))
      if @bad_fields;

    $self->_trim_opts(\%opts);

    $self->partner_db_table()->edit($obj->{'id'}, \%opts) if keys(%opts);
}

sub query_filter {
    my ($self, $filter) = @_;

    throw Exception::Denied unless $self->check_short_rights('view_all');

    return $filter;
}

sub check_action {
    my ($self, $object, $action) = @_;

    return FALSE
      unless $self->check_short_rights('edit');

    $self->SUPER::check_action($object, $action);
}

sub _trim_opts {
    my ($self, $opts) = @_;

    my @fields = keys(%{$self->get_model_fields()});

    push(@fields, qw());

    foreach (@fields) {
        $_ =~ s/^\s*|\s*$//g if exists($opts->{$_}) && !ref($opts->{$_});
    }
}

TRUE;
