package Application::Model::Product::VideoAN::Site::Video::Mirrors;

use qbit;

use base qw(Application::Model::Product::Mirrors);

use Exception::Validation::BadArguments::InvalidMirror;
use Exception::DB::DuplicateEntry;
use Exception::Validation::BadArguments;

sub accessor        {'video_an_site_mirrors'}
sub db_table_name   {'video_an_site_mirrors'}
sub page_model_name {'video_an_site'}

sub get_structure_model_accessors {
    return {
        partner_db            => 'Application::Model::PartnerDB::VideoAN',
        video_an_site         => 'Application::Model::Page::Video',
        video_an_site_mirrors => 'Application::Model::Product::VideoAN::Site::Video::Mirrors',
        api_http_gozora       => 'QBit::Application::Model::API::Yandex::HTTPGoZora',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [%{$self->SUPER::get_structure_model_fields()},];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_fields()},

        site_id => {default => TRUE, db => TRUE, pk => TRUE},
        domain  => {default => TRUE, db => TRUE, pk => TRUE},
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        %{$self->SUPER::get_structure_model_filter()},

        db_accessor => 'partner_db',
        fields      => {
            site_id => {type => 'number', label => d_gettext('Site ID')},
            domain  => {type => 'text',   label => d_gettext('Mirror')},
        },
    };
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {%{$self->SUPER::get_structure_multistates_graph()},};
}

sub add {
    my ($self, $site_id, $domains) = @_;

    $domains = [map {get_domain_or_throw($_)} @$domains];

    my $main_domain = $self->video_an_site->get($site_id, fields => ['domain'])->{'domain'};
    my @current_mirrors = map {$_->{'domain'}} @{$self->get_all(fields => ['domain'], filter => {site_id => $site_id})};

    foreach (@$domains) {
        foreach my $dd ($main_domain, @current_mirrors) {
            throw Exception::Validation::BadArguments gettext(
                "You should not add a subdomain as a mirror if a higher level domain is already added")
              if $_ =~ /\.$dd$/;
        }
        try {
            $self->partner_db_table()->add({site_id => $site_id, domain => $_});
        }
        catch Exception::DB::DuplicateEntry with {
            throw Exception::Validation::BadArguments::InvalidMirror gettext('Mirror "%s" is exists', $_);
        }
    }
    return $domains;
}

sub replace {
    my ($self, $obj, $mirrors) = @_;
    my %all_mirrors =
      map {$_->{domain} => TRUE} @{$self->get_all(fields => ['domain'], filter => {site_id => $obj->{id}})};
    my %new_mirrors     = map  {$_ => TRUE} @$mirrors;
    my @will_be_added   = grep {!$all_mirrors{$_}} keys %new_mirrors;
    my @will_be_deleted = grep {!$new_mirrors{$_}} keys %all_mirrors;
    $self->delete($obj->{'id'}, \@will_be_deleted);
    $self->add($obj->{'id'}, \@will_be_added);
}

sub delete {
    my ($self, $site_id, $domains) = @_;

    $_ = get_domain($_) // throw Exception::Validation::BadArguments::InvalidMirror gettext('Invalid mirror: "%s"', $_)
      foreach @$domains;

    $self->partner_db_table()->delete({site_id => $site_id, domain => $domains});

    return $domains;
}

TRUE;
