package Application::Model::Product::VideoAN::Site::Video::PartnerContent;

use qbit;

use base qw(Application::Model::Common);

use Exception::DB::DuplicateEntry;
use Exception::Denied;
use Exception::Validation::BadArguments;

my %SOURCE = (
    0 => d_gettext('Partner'),
    1 => d_gettext('Statistics'),
);

use constant NO_CATEGORY => 1;

sub accessor      {'partner_content'}
sub db_table_name {'partner_content'}

__PACKAGE__->model_accessors(
    partner_db             => 'Application::Model::PartnerDB::VideoAN',
    internal_content       => 'Application::Model::Product::VideoAN::Site::Video::InternalContent',
    video_an_site          => 'Application::Model::Page::Video',
    partner_genre          => 'Application::Model::Product::VideoAN::Site::Video::PartnerGenre',
    partner_content_genres => 'Application::Model::Product::VideoAN::Site::Video::PartnerContent::PartnerContentGenres',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'partner_content',
            description => d_gettext('Right to manage partner content'),
            rights      => {
                partner_content_add                     => d_gettext('Right to add partner content'),
                partner_content_view                    => d_gettext('Right to view partner contents'),
                partner_content_view_all                => d_gettext('Right to view all partner contents'),
                partner_content_edit                    => d_gettext('Right to edit partner content'),
                partner_content_view_action_log         => d_gettext('Right to view action logs'),
                partner_content_edit_field__internal_id => d_gettext('Right to edit field "internal_id"'),
            }
        }
    ]
);

__PACKAGE__->model_fields(
    id => {
        default => TRUE,
        db      => TRUE,
        pk      => TRUE,
        type    => 'number',
        label   => d_gettext('Partner content ID'),
        hint    => d_gettext('The unique content identifier')
    },
    site_id           => {db => TRUE, label   => d_gettext('Page ID')},
    modification_date => {db => TRUE, label   => d_gettext('Modification date')},
    date              => {db => TRUE, label   => d_gettext('Date')},
    public_id         => {db => TRUE, default => TRUE, label => d_gettext('Content ID')},
    name              => {db => TRUE, default => TRUE, label => d_gettext('Name')},
    internal_id       => {db => TRUE, default => TRUE, label => d_gettext('Internal content id')},
    publisher         => {db => TRUE, default => TRUE, label => d_gettext('Publisher')},
    hits              => {db => TRUE, label   => d_gettext('Hits')},
    source_id         => {db => TRUE, label   => d_gettext('Source')},
    referer           => {db => TRUE, label   => d_gettext('Referer')},
    multistate        => {db => TRUE, label   => d_gettext('Status (raw)')},
    multistate_name => {
        label      => d_gettext('Status'),
        depends_on => ['multistate'],
        get        => sub {
            $_[0]->model->get_multistate_name($_[1]->{'multistate'});
        },
    },
    actions => {
        label      => d_gettext('Actions'),
        depends_on => [qw(id multistate)],
        get        => sub {
            $_[0]->model->get_actions($_[1]);
          }
    },
    editable_fields => {
        depends_on => [qw(id multistate)],
        get        => sub {
            return {}
              unless $_[0]->model->check_action($_[1], 'edit');

            my %res = map {$_ => 1} qw(
              public_id
              name
              );

            foreach (qw(internal_id)) {
                $res{$_} = TRUE
                  if $_[0]->model->check_rights('partner_content_edit_field__' . $_);
            }

            return \%res;
          }
    },
    available_fields => {
        depends_on => [qw(multistate)],
        label      => d_gettext('Available fields'),
        get        => sub {
            return $_[0]->model->get_available_fields($_[1]);
          }
    },
    internal_content => {
        depends_on => [qw(internal_id)],
        get        => sub {
            return defined($_[1]->{'internal_id'})
              ? $_[0]->{'__INTERNAL_CONTENTS__'}{$_[1]->{'internal_id'}} // {}
              : {};
          }
    },
    video_an_site => {
        depends_on => [qw(site_id)],
        get        => sub {
            return $_[0]->{'__VIDEO_AN_SITE__'}{$_[1]->{'site_id'}} || {};
          }
    },
    domain => {
        depends_on => [qw(video_an_site)],
        get        => sub {
            return $_[1]->{'video_an_site'}{'domain'} || '';
          }
    },
    genres => {
        depends_on => [qw(id)],
        get        => sub {
            return $_[0]->{'__GENRES__'}{$_[1]->{'id'}} || [];
          }
    },
    source => {
        depends_on => [qw(source_id)],
        get        => sub {
            return $SOURCE{$_[1]->{'source_id'}}->();
        },
        label => d_gettext('Source')
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        modification_date => {type => 'number',     label => d_gettext('Modification date')},
        id                => {type => 'number',     label => d_gettext('Partner content ID'), pk => TRUE},
        site_id           => {type => 'number',     label => d_gettext('Page ID')},
        public_id         => {type => 'text',       label => d_gettext('Content ID')},
        internal_id       => {type => 'number',     label => d_gettext('Internal content id')},
        name              => {type => 'text',       label => d_gettext('Name')},
        multistate        => {type => 'multistate', label => d_gettext('Status')},
        publisher         => {type => 'text',       label => d_gettext('Publisher')},
        source_id         => {type => 'number',     label => d_gettext('Source ID')},
        source => {
            type      => 'dictionary',
            db_filter => sub {
                $_[0]->{'__DB_FILTER__'}{$_[1]->[0]}->as_filter(['source_id' => $_[1]->[1] => $_[1]->[2]], $_[2]);
            },
            label  => gettext('Source'),
            values => sub {
                [map {{id => $_, label => $SOURCE{$_}->()}} sort keys(%SOURCE)];
              }
        },
        video_an_site => {
            type           => 'subfilter',
            model_accessor => 'video_an_site',
            field          => 'site_id',
            fk_field       => 'id',
            label          => d_gettext('Video an site'),
        },
    }
);

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [{name => 'name', label => gettext('Caption')}, {name => 'site_id', label => gettext('Page ID')},],
        [{name => 'video_an_site.domain', label => gettext('Domain')}, {name => 'source', label => gettext('Source')},],
    ];
}

__PACKAGE__->multistates_graph(
    empty_name    => 'New',
    multistates   => [[deleted => d_pgettext('Content status', 'Deleted')],],
    actions       => {},
    right_actions => {
        delete  => d_pgettext('Content action', 'Delete'),
        restore => d_pgettext('Content action', 'Restore'),
        edit    => d_pgettext('Content action', 'Edit'),
    },
    right_group        => [partner_content => d_gettext('Right to manage partner content')],
    right_name_prefix  => 'partner_content_',
    multistate_actions => [
        {
            action    => 'delete',
            from      => 'not deleted',
            set_flags => ['deleted']
        },
        {
            action      => 'restore',
            from        => 'deleted',
            reset_flags => ['deleted']
        },
        {
            action => 'edit',
            from   => 'not deleted'
        },
    ]
);

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    my $ids = array_uniq(map {$_->{'id'} // ()} @$result);

    if ($fields->need('internal_content')) {
        $fields->{'__INTERNAL_CONTENTS__'} = {
            map {$_->{'id'} => $_} @{
                $self->internal_content->get_all(
                    fields => [qw(id name)],
                    filter => {id => array_uniq(grep {$_} map {$_->{'internal_id'}} @$result)}
                )
              }
        };
    }

    if ($fields->need('video_an_site')) {
        $fields->{'__VIDEO_AN_SITE__'} = {
            map {$_->{'id'} => $_} @{
                $self->video_an_site->get_all(
                    fields => [qw(id domain)],
                    filter => {id => array_uniq(map {$_->{'site_id'} // ()} @$result)}
                )
              }
        };
    }

    if ($fields->need('genres')) {
        $fields->{'__GENRES__'} = {};

        foreach (
            @{
                $self->partner_genre->query(fields => $self->partner_genre->_get_fields_obj([qw(id public_id name)]))
                  ->join(
                    table   => $self->partner_db->partner_content_genres,
                    fields  => [qw(content_id)],
                    filter  => {content_id => $ids},
                    join_on => [genre_id => '=' => {id => $self->partner_db->partner_genre}],
                  )->get_all()
            }
          )
        {
            my $content_id = delete($_->{'content_id'});
            $fields->{'__GENRES__'}{$content_id} //= [];
            push(@{$fields->{'__GENRES__'}{$content_id}}, $_);
        }
    }
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied unless $self->check_short_rights('add');

    my @require_fields = qw(site_id public_id name genres source_id);
    throw Exception::Validation::BadArguments gettext('Expected fields: %s',
        join(', ', grep {!defined($opts{$_})} @require_fields))
      if grep {!defined($opts{$_})} @require_fields;

    $opts{'date'} //= curdate(oformat => 'db');

    $self->_trim_opts(\%opts);
    $self->_check_opts(\%opts);

    my $genres = delete($opts{'genres'});

    my $id;
    try {
        $self->partner_db->transaction(
            sub {
                $id = $self->partner_db_table()->add(
                    {
                        modification_date => curdate(oformat => 'db_time'),
                        hash_transform(\%opts, [qw(site_id public_id name date publisher referer hits source_id)])
                    }
                );

                $self->partner_db->partner_content_genres->add_multi(
                    [map {{content_id => $id, genre_id => $_}} @$genres])
                  if @$genres;
            }
        );
    }
    catch Exception::DB::DuplicateEntry with {
        throw Exception::Validation::BadArguments gettext('The content already exists');
    };

    return $id;
}

sub on_action_edit {
    my ($self, $obj, %opts) = @_;

    my $fields = $self->get($obj->{'id'}, fields => ['editable_fields'])->{'editable_fields'};

    my @bad_fields = grep {!$fields->{$_}} sort keys(%opts);
    throw Exception::Validation::BadArguments gettext('You can not edit the following fields: %s',
        join(', ', @bad_fields))
      if @bad_fields;

    $self->_trim_opts(\%opts);
    $self->_check_opts(\%opts);

    if (keys(%opts)) {
        $opts{'modification_date'} //= curdate(oformat => 'db_time');

        $self->partner_db_table()->edit($obj->{'id'}, \%opts);
    }
}

sub query_filter {
    my ($self, $filter) = @_;

    throw Exception::Denied unless $self->check_short_rights('view_all');

    return $filter;
}

sub add_into_unallocated_contents {
    my ($self, $id) = @_;

    $self->do_action($id, 'edit', internal_id => NO_CATEGORY);
}

sub check_action {
    my ($self, $object, $action) = @_;

    return FALSE
      unless $self->check_short_rights('edit');

    $self->SUPER::check_action($object, $action);
}

sub get_free_partner_contents {
    my ($self, %opts) = @_;

    my @ids = map {$_->{'id'}} @{
        $self->query(
            fields => $self->_get_fields_obj([qw(id)]),
            filter => [{isnull => ['internal_id']} => '=' => \1],
          )->get_all()
      };

    return \@ids if $opts{'only_ids'};

    my $filter = ['id' => 'IN' => \@ids];
    if (exists($opts{'filter'}) && $opts{'filter'}) {
        $filter = ['AND', [$filter, $opts{'filter'}]];
    }

    return $self->get_all_with_meta(%opts, filter => $filter);
}

sub _check_opts {
    my ($self, $opts) = @_;

    if (exists($opts->{'internal_id'})) {
        my $internal_content = $self->internal_content->get($opts->{'internal_id'});

        throw Exception::Validation::BadArguments gettext('Unknown content with id: %s', $opts->{'internal_content'})
          unless $internal_content;
    }

    if (exists($opts->{'genre_id'})) {
        my $partner_genre = $self->partner_genre->get($opts->{'genre_id'});

        throw Exception::Validation::BadArguments gettext('Unknown partner genre with id: %s', $opts->{'genre_id'})
          unless $partner_genre;
    }
}

sub _trim_opts {
    my ($self, $opts) = @_;

    my @fields = keys(%{$self->get_model_fields()});

    push(@fields, qw());

    foreach (@fields) {
        $_ =~ s/^\s*|\s*$//g if exists($opts->{$_}) && !ref($opts->{$_});
    }
}

TRUE;
