package Application::Model::Product::VideoAN::Site::Video::PartnerGenre;

use qbit;

use base qw(Application::Model::Common);

use Exception::DB::DuplicateEntry;
use Exception::Denied;
use Exception::Validation::BadArguments;

use constant NO_CATEGORY => 'IAB24';

my %SOURCE = (
    0 => d_gettext('Partner'),
    1 => d_gettext('Statistics'),
);

sub accessor      {'partner_genre'}
sub db_table_name {'partner_genre'}

sub get_product_name {gettext('partner_genre')}

__PACKAGE__->model_accessors(
    partner_db     => 'Application::Model::PartnerDB::VideoAN',
    internal_genre => 'Application::Model::Product::VideoAN::Site::Video::InternalGenre',
    video_an_site  => 'Application::Model::Page::Video',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'partner_genre',
            description => d_gettext('Right to manage partner genre'),
            rights      => {
                partner_genre_add                => d_gettext('Right to add partner genre'),
                partner_genre_view               => d_gettext('Right to view partner genres'),
                partner_genre_view_all           => d_gettext('Right to view all partner genres'),
                partner_genre_edit               => d_gettext('Right to edit partner genre'),
                partner_genre_view_action_log    => d_gettext('Right to view action logs'),
                partner_genre_edit_field__genres => d_gettext('Right to edit field "genres"'),
            }
        }
    ]
);

__PACKAGE__->model_fields(
    id => {
        default => TRUE,
        db      => TRUE,
        pk      => TRUE,
        type    => 'number',
        label   => d_gettext('Partner genre ID'),
        hint    => d_gettext('The unique genre identifier')
    },
    site_id           => {db => TRUE, label   => d_gettext('Page ID')},
    modification_date => {db => TRUE, label   => d_gettext('Modification date')},
    date              => {db => TRUE, label   => d_gettext('Date')},
    public_id         => {db => TRUE, default => TRUE, label => d_gettext('Genre ID')},
    name              => {db => TRUE, default => TRUE, label => d_gettext('Genre name')},
    hits              => {db => TRUE, label   => d_gettext('Hits')},
    source_id         => {db => TRUE, label   => d_gettext('Source')},
    multistate        => {db => TRUE, label   => d_gettext('Status (raw)')},
    multistate_name => {
        label      => d_gettext('Status'),
        depends_on => ['multistate'],
        get        => sub {
            $_[0]->model->get_multistate_name($_[1]->{'multistate'});
        },
    },
    actions => {
        label      => d_gettext('Actions'),
        depends_on => [qw(id multistate)],
        get        => sub {
            $_[0]->model->get_actions($_[1]);
          }
    },
    editable_fields => {
        depends_on => [qw(id multistate)],
        get        => sub {
            return {}
              unless $_[0]->model->check_action($_[1], 'edit');

            my %res = map {$_ => 1} qw(
              public_id
              name
              );

            foreach (qw(genres)) {
                $res{$_} = TRUE
                  if $_[0]->model->check_rights('partner_genre_edit_field__' . $_);
            }

            return \%res;
          }
    },
    available_fields => {
        depends_on => [qw(multistate)],
        label      => d_gettext('Available fields'),
        get        => sub {
            return $_[0]->model->get_available_fields($_[1]);
          }
    },
    internal_genres => {
        depends_on => [qw(id)],
        get        => sub {
            return $_[0]->{'__INTERNAL_GENRES__'}{$_[1]->{'id'}} || [];
          }
    },
    video_an_site => {
        depends_on => [qw(site_id)],
        get        => sub {
            return $_[0]->{'__VIDEO_AN_SITE__'}{$_[1]->{'site_id'}} || {};
          }
    },
    domain => {
        depends_on => [qw(video_an_site)],
        get        => sub {
            return $_[1]->{'video_an_site'}{'domain'} || '';
          }
    },
    source => {
        depends_on => [qw(source_id)],
        get        => sub {
            return $SOURCE{$_[1]->{'source_id'}}->();
        },
        label => d_gettext('Source')
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id                => {type => 'number', label => d_gettext('Partner genre ID'), pk => TRUE},
        site_id           => {type => 'number', label => d_gettext('Page ID')},
        public_id         => {type => 'text',   label => d_gettext('Genre ID')},
        name              => {type => 'text',   label => d_gettext('Name')},
        modification_date => {type => 'number', label => d_gettext('Modification date')},
        source_id         => {type => 'number', label => d_gettext('Source ID')},
        source            => {
            type      => 'dictionary',
            db_filter => sub {
                $_[0]->{'__DB_FILTER__'}{$_[1]->[0]}->as_filter(['source_id' => $_[1]->[1] => $_[1]->[2]], $_[2]);
            },
            label  => gettext('Source'),
            values => sub {
                [map {{id => $_, label => $SOURCE{$_}->()}} sort keys(%SOURCE)];
              }
        },
        site => {
            type           => 'subfilter',
            model_accessor => 'video_an_site',
            field          => 'site_id',
            fk_field       => 'id',
            label          => d_gettext('Site'),
        },
        multistate => {type => 'multistate', label => d_gettext('Status')},
    }
);

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [{name => 'name',   label => gettext('Genre name')}, {name => 'site.domain', label => gettext('Domain')},],
        [{name => 'source', label => gettext('Source')},],
    ];
}

__PACKAGE__->multistates_graph(
    empty_name    => 'New',
    multistates   => [[deleted => d_pgettext('Genre status', 'Deleted')],],
    actions       => {},
    right_actions => {
        delete  => d_pgettext('Genre action', 'Delete'),
        restore => d_pgettext('Genre action', 'Restore'),
        edit    => d_pgettext('Genre action', 'Edit'),
    },
    right_group        => [partner_genre => d_gettext('Right to manage partner genre')],
    right_name_prefix  => 'partner_genre_',
    multistate_actions => [
        {
            action    => 'delete',
            from      => 'not deleted',
            set_flags => ['deleted']
        },
        {
            action      => 'restore',
            from        => 'deleted',
            reset_flags => ['deleted']
        },
        {
            action => 'edit',
            from   => 'not deleted'
        },
    ]
);

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    my $ids = array_uniq(map {$_->{'id'} // ()} @$result);

    if ($fields->need('internal_genres')) {
        $fields->{'__INTERNAL_GENRES__'} = {};

        foreach (
            @{
                $self->internal_genre->query(fields => $self->internal_genre->_get_fields_obj([qw(id public_id name)]))
                  ->join(
                    table   => $self->partner_db->partner_and_internal_genre,
                    fields  => [qw(partner_id)],
                    filter  => {partner_id => $ids},
                    join_on => [internal_id => '=' => {id => $self->partner_db->internal_genre}],
                  )->get_all()
            }
          )
        {
            my $partner_id = delete($_->{'partner_id'});
            $fields->{'__INTERNAL_GENRES__'}{$partner_id} //= [];
            push(@{$fields->{'__INTERNAL_GENRES__'}{$partner_id}}, $_);
        }
    }

    if ($fields->need('video_an_site')) {
        $fields->{'__VIDEO_AN_SITE__'} = {
            map {$_->{'id'} => $_} @{
                $self->video_an_site->get_all(
                    fields => [qw(id domain)],
                    filter => {id => array_uniq(map {$_->{'site_id'} // ()} @$result)}
                )
              }
        };
    }
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied unless $self->check_short_rights('add');

    my @require_fields = qw(site_id public_id name source_id);
    throw Exception::Validation::BadArguments gettext('Expected fields: %s',
        join(', ', grep {!$opts{$_}} @require_fields))
      if grep {!defined($opts{$_})} @require_fields;

    $opts{'date'} //= curdate(oformat => 'db');

    $self->_trim_opts(\%opts);
    $self->_check_opts(\%opts);

    my $id;
    try {
        $id = $self->partner_db_table()->add(
            {
                modification_date => curdate(oformat => 'db_time'),
                hash_transform(\%opts, [qw(site_id public_id name date hits source_id)])
            }
        );
    }
    catch Exception::DB::DuplicateEntry with {
        throw Exception::Validation::BadArguments gettext('The genre already exists');
    };

    return $id;
}

sub on_action_edit {
    my ($self, $obj, %opts) = @_;

    my $fields = $self->get($obj->{'id'}, fields => ['editable_fields'])->{'editable_fields'};

    my @bad_fields = sort grep {!$fields->{$_}} keys(%opts);
    throw Exception::Validation::BadArguments gettext('You can not edit the following fields: %s',
        join(', ', @bad_fields))
      if @bad_fields;

    $self->_trim_opts(\%opts);
    $self->_check_opts(\%opts);

    if (exists($opts{'genres'})) {
        my $internal_genres = $self->internal_genre->get_all(filter => {id => $opts{'genres'}});

        throw Exception::Validation::BadArguments gettext('Unknown genres with id: %s',
            join(', ', arrays_difference($opts{'genres'}, [map {$_->{'id'}} @$internal_genres])))
          unless @{$opts{'genres'}} == @$internal_genres;

        $self->partner_db->partner_and_internal_genre->delete($self->partner_db->filter({partner_id => $obj->{'id'}}));

        $self->partner_db->partner_and_internal_genre->add_multi(
            [map {{partner_id => $obj->{'id'}, internal_id => $_}} @{$opts{'genres'}}]);
    }

    if (keys(%opts)) {
        delete($opts{'genres'});
        $opts{'modification_date'} //= curdate(oformat => 'db_time');

        $self->partner_db_table()->edit($obj->{'id'}, \%opts);
    }
}

sub query_filter {
    my ($self, $filter) = @_;

    throw Exception::Denied unless $self->check_short_rights('view_all');

    return $filter;
}

sub add_into_unallocated_genres {
    my ($self, $id) = @_;

    my $internal_id = $self->internal_genre->get_all(fields => ['id'], filter => {public_id => NO_CATEGORY})->[0]{'id'};

    $self->do_action($id, 'edit', genres => [$internal_id]);
}

sub check_action {
    my ($self, $object, $action) = @_;

    return FALSE
      unless $self->check_short_rights('edit');

    $self->SUPER::check_action($object, $action);
}

sub get_free_partner_genres {
    my ($self, %opts) = @_;

    my @ids = map {$_->{'id'}} @{
        $self->query(fields => $self->_get_fields_obj(['id']),)->left_join(
            table   => $self->partner_db->partner_and_internal_genre,
            fields  => [],
            join_on => [partner_id => '=' => {id => $self->partner_db_table()}],
            filter => ['AND', [{isnull => ['partner_id']}]]
          )->get_all()
      };

    return \@ids if $opts{'only_ids'};

    my $filter = ['id' => 'IN' => \@ids];
    if (exists($opts{'filter'}) && $opts{'filter'}) {
        $filter = ['AND', [$filter, $opts{'filter'}]];
    }

    return $self->get_all_with_meta(%opts, filter => $filter);
}

sub _check_opts {
    my ($self, $opts) = @_;

    if (exists($opts->{'source_id'})) {
        throw Exception::Validation::BadArguments gettext('Unknown source with "%s"', $opts->{'source_id'})
          unless exists($SOURCE{$opts->{'source_id'}});
    }
}

sub _trim_opts {
    my ($self, $opts) = @_;

    my @fields = keys(%{$self->get_model_fields()});

    push(@fields, qw(genres));

    foreach (@fields) {
        $_ =~ s/^\s*|\s*$//g if exists($opts->{$_}) && !ref($opts->{$_});
    }
}

TRUE;
