package Application::Model::Product::VideoAN::Site::Video::StatFiles;

use qbit;
use Compress::Zlib;
use Utils::XLS;
use Math::Round;
use HTML::Entities;
use Text::CSV_XS;
use File::Temp qw(tempfile);
use Archive::Zip qw(AZ_OK);
use Utils::ClickHouse::Local;
use Utils::Logger qw(INFO WARN);
use File::stat;
use PiConstants qw(
  $DSP_OWN_ADV_ID
  $DSP_UNSOLD_ID
  $VAT_COEFFICIENT
  );

use Application::Model::DAC;
use base qw(
  Application::Model::DBManager::Base
  Application::Model::Statistics::_Utils::Money
  Application::Model::DAC
  );

use Application::Model::Statistics::Fields::Metrics;

use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::StatFiles;
use Exception::API::MediaStorage::S3;

sub accessor      {'video_stat_files'}
sub db_table_name {'video_stat_files'}

__PACKAGE__->model_accessors(
    partner_db    => 'Application::Model::PartnerDB::VideoAN',
    video_an_site => 'Application::Model::Page::Video',
    statistics    => 'Application::Model::Statistics',
    kv_store      => 'QBit::Application::Model::KvStore',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'video_stat_files',
            description => d_gettext('Rights to use video stat files'),
            rights      => {
                video_stat_files_add          => d_gettext('Right to add video stat file'),
                video_stat_files_edit         => d_gettext('Right to edit video stat file'),
                video_stat_files_view_all     => d_gettext('Right to view all video stat files'),
                video_stat_files_edit_all     => d_gettext('Right to edit all video stat files'),
                video_stat_files_view_filters => d_gettext('Right to view filters'),
            }
        }
    ]
);

my $TYPES = [
    {
        id    => 'yesterday',
        label => d_gettext('Yesterday'),
    },
    {
        id    => 'this_month',
        label => d_gettext('This month'),
    },
    {
        id    => 'archive',
        label => d_gettext('Archive'),
    }
];

my @input_columns = qw(
  page_id
  video_content_id
  video_content_name
  video_publisher_id
  video_publisher_name
  video_bk_all_hits
  video_bk_shows
  bk_partner_price_w_nds
  bk_partner_price_wo_nds
  video_clid
  );

my @output_columns = @input_columns[1 .. $#input_columns];

__PACKAGE__->model_fields(
    id                => {default => TRUE, db => TRUE, pk   => TRUE,     type => 'number', api => 1},
    page_id           => {default => TRUE, db => TRUE, type => 'number', api  => 1},
    from_date         => {default => TRUE, db => TRUE,},
    to_date           => {default => TRUE, db => TRUE,},
    modification_date => {default => TRUE, db => TRUE,},
    s3_key            => {default => TRUE, db => TRUE,},
    s3_size           => {default => TRUE, db => TRUE,},
    data              => {default => TRUE, db => TRUE,},
    uncompress_data   => {
        label      => d_gettext('Uncompress data'),
        depends_on => [qw(data)],
        get        => sub {
            return '' if length($_[1]->{'data'}) < 4;
            uncompress(substr($_[1]->{'data'}, 4, length($_[1]->{'data'}) - 1));
        },
    },
    date => {
        depends_on => [qw(from_date)],
        get        => sub {
            return $_[1]->{from_date};
        },
    },
    type => {
        depends_on => [qw(from_date to_date)],
        get        => sub {
            return _type_from_period($_[1]->{from_date}, $_[1]->{to_date});
        },
    },
    name => {
        label      => d_gettext('Name'),
        depends_on => [qw(page_id from_date to_date)],
        get        => sub {
            return _build_report_name(@{$_[1]}{qw(page_id from_date to_date)});
        },
    },
    size => {
        depends_on => ['uncompress_data', 's3_size'],
        default    => TRUE,
        get        => sub {
            my $bytes = $_[1]->{'s3_size'} // length($_[1]->{'uncompress_data'});
            my $kb = sprintf('%.02f', $bytes / 1024);
            return $bytes . ' B' if $kb =~ /^0\.00/;
            my $mb = sprintf('%.02f', $bytes / (1024 * 1024));
            return $kb . ' KB' if $mb =~ /^0\.00/;
            return $mb . ' MB';
        },
    },
    editable_fields => {
        label => d_gettext('Editable fields'),
        get   => sub {
            $_[0]->model->get_editable_fields();
          }
    },
    available_fields => {
        label => d_gettext('Available fields'),
        get   => sub {
            return $_[0]->model->get_available_fields();
          }
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id      => {type => 'number', label => d_gettext('File ID')},
        page_id => {type => 'number', label => d_gettext('Page ID')},
        type    => {
            type      => 'dictionary',
            db_filter => sub {

                $_[1]->[2] = [$_[1]->[2]] unless ref($_[1]->[2]) eq 'ARRAY';

                return [OR => [map {$_[0]->_type_filter($_)} @{$_[1]->[2]}]];
            },
            label  => d_gettext('Type'),
            values => sub {
                [
                    map {
                        {%$_, label => $_->{'label'}()}
                      } @$TYPES
                ];
              }
        },
        date => {
            label     => d_gettext('Date'),
            type      => 'date',
            db_filter => sub {
                return ['from_date' => $_[1]->[1] => \$_[1]->[2]];
            },
        },
        to_date           => {type => 'date', label => d_gettext('Date')},
        from_date         => {type => 'date', label => d_gettext('Date')},
        modification_date => {type => 'date', label => d_gettext('modification date')},
    }
);

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        $self->check_short_rights('view_filters')
        ? (
            [{name => 'page_id', label => gettext('Page ID')}, {name => 'type', label => gettext('Type')},],
            [{name => 'date',    label => gettext('Date')},],
          )
        : ()
    ];
}

sub get_available_fields {
    my ($self) = @_;

    my $model_fields = $self->get_model_fields;

    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub get_editable_fields {{}}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied unless $self->check_short_rights('add');

    my @require_fields = qw(page_id modification_date from_date to_date s3_key s3_size);

    my @bad_fields = grep {!$opts{$_} || !length($opts{$_})} @require_fields;
    throw Exception::Validation::BadArguments gettext('Expected following fields: %s', join(', ', @bad_fields))
      if @bad_fields;

    $self->partner_db_table()->add({hash_transform(\%opts, \@require_fields)});
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter->and([OR => [map {$self->_type_filter($_)} ('yesterday', 'this_month', 'archive')]]);
    $filter = $self->limit_filter_by_special($filter, 'page_id', 'video_an_site', 'id');

    return $filter;
}

sub query_alias {'all_reports'}

sub query_join {
    my ($self, $query) = @_;

    $query->join(
        table => $self->partner_db->query->select(
            table  => $self->partner_db_table(),
            fields => {
                page_id   => 'page_id',
                from_date => 'from_date',
                to_date   => 'to_date',
                max_md    => {MAX => ['modification_date']},
            },
          )->group_by('page_id', 'from_date', 'to_date'),
        fields  => [qw()],
        alias   => 'most_recent_reports',
        join_on => [
            AND => [
                [{'page_id'   => 'all_reports'}         => '=' => 'page_id'],
                [{'from_date' => 'all_reports'}         => '=' => 'from_date'],
                [{'to_date'   => 'all_reports'}         => '=' => 'to_date'],
                [{'max_md'    => 'most_recent_reports'} => '=' => {'modification_date' => 'all_reports'}]
            ]
        ],
    );

    return $query;
}

sub generate_publisher_reports {
    my ($self, $from_date, $to_date) = @_;

    my $stat_fh = $self->_get_preprocessed_data_fh($from_date, $to_date);

    my $raw_data_stream = $self->_get_report_data_stream($stat_fh);

    my $first_line = $raw_data_stream->();
    my $filename;
    my %reports;

    my %video_page_ids = map {$_->{id} => 1} @{$self->app->video_an_site->get_all(fields => [qw(id)])};

    INFO 'start splitting';

    while (defined $first_line) {
        my $page_id = $first_line->{page_id};
        ($first_line, $filename) = $self->_shift_report_from_preprocessed_data($first_line, $raw_data_stream);

        if (exists($video_page_ids{$page_id})) {
            $reports{$page_id}{report} = $filename;
        } else {
            INFO "Unknown page_id = $page_id. Skipping";
        }
    }

    close($stat_fh) or throw Exception::StatFiles "Failed to close preprocessed data file: $!";

    INFO 'start archiving';

    for my $page_id (keys %reports) {
        try {
            my $archive_filename =
              $self->_create_archive($reports{$page_id}{report} => "${page_id}_${from_date}_${to_date}.tsv");
            $reports{$page_id}{archive} = $archive_filename;
        }
        catch Exception::StatFiles with {
            my ($exception) = @_;
            WARN $exception;
        };
    }

    INFO 'start sending to S3';

    for my $page_id (keys %reports) {
        try {
            my $curdate         = curdate(oformat => 'db_time');
            my $rand            = int(rand(1000000));
            my $curdate_for_key = _datetime_for_key($curdate);
            my $key             = "${page_id}_${from_date}_${to_date}_${curdate_for_key}_${rand}.zip";

            $self->app->api_media_storage_s3->put_file($key, 'application/zip', $reports{$page_id}{archive});

            my $size = stat($reports{$page_id}{archive})->size;

            $self->add(
                page_id           => $page_id,
                from_date         => $from_date,
                to_date           => $to_date,
                modification_date => $curdate,
                s3_key            => $key,
                s3_size           => $size,
            );
        }
        catch Exception::API::MediaStorage::S3 with {
            my ($exception) = @_;
            WARN $exception;
        };
    }

    return 1;
}

sub _build_report_name {
    my ($page_id, $from_date, $to_date) = @_;

    my $name;
    if ($from_date eq $to_date) {
        $name = d_gettext('Page ID %s (Statistics for %s)',
            $page_id, format_date($from_date, gettext('%d.%m.%Y'), iformat => 'db'));
    } else {
        $name = d_gettext(
            'Page ID %s (Statistics for %s - %s)',
            $page_id,
            format_date($from_date, gettext('%d.%m.%Y'), iformat => 'db'),
            format_date($to_date,   gettext('%d.%m.%Y'), iformat => 'db')
        );
    }

    return $name->();
}

sub _aggregate_and_sort_data_files {
    my ($self, $input_files, $output_file) = @_;

    Utils::ClickHouse::Local->new(
        input_files => $input_files,
        output_file => $output_file,
        structure   => [
            UpdateTime    => 'DateTime',
            PageID        => 'UInt64',
            DSPID         => 'UInt64',
            PublisherID   => 'String',
            ContentID     => 'String',
            PublisherName => 'String',
            ContentName   => 'String',
            Shows         => 'UInt64',
            Wins          => 'UInt64',
            Hits          => 'UInt64',
            Price         => 'UInt64',
            PartnerPrice  => 'UInt64',
            CLID          => 'UInt64',
        ],
      )->query(
        "SELECT PageID as page_id,
                ContentID as video_content_id,
                ContentName as video_content_name,
                PublisherID as video_publisher_id,
                PublisherName as video_publisher_name,
                sum(Wins) as video_bk_all_hits,
                sum(if(DSPID == $DSP_OWN_ADV_ID OR DSPID == $DSP_UNSOLD_ID, 0, Shows)) as video_bk_shows,
                round(divide(multiply(sum(PartnerPrice), $VAT_COEFFICIENT), pow(10, 6)), 2) as bk_partner_price_w_nds,
                round(divide(sum(PartnerPrice), pow(10, 6)), 2) as bk_partner_price_wo_nds,
                CLID as video_clid
         FROM table
         GROUP BY PageID, PublisherID, PublisherName, ContentID, ContentName, CLID
         ORDER BY PageID, video_bk_all_hits, PublisherID, PublisherName, ContentID, ContentName;"
      );
}

sub _get_preprocessed_data_fh {
    my ($self, $from_date, $to_date) = @_;

    my $cache_files = $self->app->stat_download_data->get_all_most_recent(
        fields => [qw(method stat_date filepath)],
        filter => [
            AND => [
                ['method'    => '='  => 'get_publisher_stat'],
                ['stat_date' => '>=' => $from_date],
                ['stat_date' => '<=' => $to_date]
            ]
        ],
    );

    my $need_files = dates_delta_days($from_date, $to_date, iformat => 'db') + 1;
    my $have_files = @$cache_files;
    throw Exception::StatFiles "Wrong number of files for period $from_date - $to_date: $have_files/$need_files"
      if $need_files != $have_files;

    my @cache_paths =
      map {$_->{filepath}} @$cache_files;

    my $preprocessed_data_file_name = (tempfile(UNLINK => 1))[1];

    $self->_aggregate_and_sort_data_files(\@cache_paths, $preprocessed_data_file_name);
    open(my $fh, '<', $preprocessed_data_file_name)
      or throw Exception::StatFiles "Failed to open preprocessed data file $preprocessed_data_file_name: $!";

    return $fh;
}

sub _shift_report_from_preprocessed_data {
    my ($self, $first_line, $raw_data_stream) = @_;

    my $page_id = $first_line->{page_id};

    my ($report_fh, $report_filename) = tempfile(UNLINK => 1);
    binmode($report_fh, ':encoding(UTF-8)');
    my $report_data_sink = $self->_get_report_data_sink($report_fh);

    my $row = $first_line;
    do {
        my $data = $row;
        $report_data_sink->($data);
    } while (($row = $raw_data_stream->()) && $row->{page_id} == $page_id);

    close($report_fh);

    return ($row, $report_filename);
}

sub _create_archive {
    my ($self, $source_filename, $dest_filename) = @_;

    my ($archive_fh, $archive_filename) = tempfile(UNLINK => 1);

    my $zip = Archive::Zip->new();
    $zip->addFile($source_filename => $dest_filename);
    unless ($zip->writeToFileHandle($archive_fh) == AZ_OK) {
        throw Exception::StatFiles "Failed to create zip archive $archive_filename with $dest_filename";
    }

    close($archive_fh);

    return $archive_filename;
}

sub _get_report_data_stream {
    my ($self, $stat_fh) = @_;

    my $tsv =
      Text::CSV_XS->new({binary => 1, sep_char => "\t", eol => "\n", escape_char => undef, quote_char => undef});
    $tsv->column_names(@input_columns);

    return sub {
        $tsv->getline_hr($stat_fh);
    };
}

sub _get_report_data_sink {
    my ($self, $report_fh) = @_;

    my $tsv =
      Text::CSV_XS->new({binary => 1, sep_char => "\t", eol => "\n", escape_char => undef, quote_char => undef});

    $tsv->column_names(@output_columns);

    my @titles = ();
    foreach my $field_id (@output_columns) {
        my $field;
        if ($field_id =~ /hits/) {
            $field = get_field_hits($field_id);
        } elsif ($field_id =~ /shows/) {
            $field = get_field_shows($field_id);
        } elsif ($field_id =~ /wo?_nds/) {
            $field = get_field_money($field_id);
        } elsif (exists($Application::Model::Statistics::Product::DIMENSION_TYPES{$field_id})) {
            $field = $Application::Model::Statistics::Product::DIMENSION_TYPES{$field_id};
        } else {
            throw Exception gettext('Unknown field with id: %s', $field_id);
        }

        my $title = $field->{'title'}->();
        $title =~ s/&nbsp;/ /g;

        push(@titles, $title);
    }

    $tsv->print($report_fh, \@titles);

    return sub {
        $tsv->print($report_fh, [@{$_[0]}{$tsv->column_names}]);
    };
}

sub _type_from_period {
    my ($from_date, $to_date) = @_;

    my $yesterday = name2date('yesterday', oformat => 'db');
    my $yesterday_norm = trdate(db => norm => $yesterday);

    my $from_date_norm = trdate(db => norm => $from_date);

    my $this_month_first_day = clone($yesterday_norm);
    $this_month_first_day->[2] = 1;
    $this_month_first_day = trdate(norm => db => $this_month_first_day);

    my $this_month_last_day = clone($yesterday_norm);
    $this_month_last_day->[2] = trdate(norm => days_in_month => $this_month_last_day);
    $this_month_last_day = trdate(norm => db => $this_month_last_day);

    my $from_date_first_day = clone($from_date_norm);
    $from_date_first_day->[2] = 1;
    $from_date_first_day = trdate(norm => db => $from_date_first_day);

    my $from_date_last_day = clone($from_date_norm);
    $from_date_last_day->[2] = trdate(norm => days_in_month => $from_date_last_day);
    $from_date_last_day = trdate(norm => db => $from_date_last_day);

    if ($from_date eq $yesterday && $to_date eq $yesterday) {
        return 'yesterday';
    } elsif ($from_date eq $this_month_first_day && $to_date eq $yesterday) {
        return 'this_month';
    } elsif ($from_date eq $from_date_first_day
        && $to_date eq $from_date_last_day
        && ($from_date_norm->[0] < $yesterday_norm->[0] || $from_date_norm->[1] < $yesterday_norm->[1]))
    {
        return 'archive';
    } else {
        return undef;
    }
}

sub _type_filter {
    my ($self, $type) = @_;

    my $yesterday = name2date('yesterday', oformat => 'db');
    my $first_day = trdate(db => norm => $yesterday);
    $first_day->[2] = 1;
    $first_day = trdate(norm => db => $first_day);

    my $types = {
        yesterday  => [AND => [['from_date' => '=' => \$yesterday], ['to_date' => '=' => \$yesterday],]],
        this_month => [AND => [['from_date' => '=' => \$first_day], ['to_date' => '=' => \$yesterday],]],
        archive    => [
            AND => [
                ['from_date' => '<' => \$first_day],
                ['from_date' => '=' => {'DATE_FORMAT' => ['from_date', \'%Y-%m-01']}],
                ['to_date' => '=' => {'LAST_DAY' => ['from_date']}],
            ]
        ],
    };

    return $types->{$type};
}

sub _datetime_for_key {
    my ($dt) = @_;

    $dt =~ s/:/-/g;
    $dt =~ s/\s/_/g;

    return $dt;
}

sub delete_publisher_reports {
    my ($self, $filter) = @_;

    my $reports = $self->partner_db_table()->get_all(
        fields => [qw(id s3_key)],
        filter => $filter,
    );

    for my $report (@$reports) {
        try {
            $self->app->api_media_storage_s3->delete($report->{s3_key});
            $self->partner_db_table->delete({id => $report->{id}});
            INFO "Deleted report id = $report->{id}, s3_key = $report->{s3_key}";
        }
        catch Exception::API::MediaStorage::S3 with {
            my ($exception) = @_;
            WARN $exception;
        };
    }
}

TRUE;
