package Application::Model::ProductManager;

use Coro;
use Coro::Specific;

use qbit;

use base qw(QBit::Application::Model);

use Exception::DB::NoFieldsAvailable;
use Exception::Validation::BadArguments;

sub accessor {'product_manager'}

my %URL_BY_MODEL = (
    mobile_app                        => '/v2/mobile/applications/',
    mobile_app_settings               => '/v2/mobile/campaigns/',
    mobile_app_rtb                    => '/v2/mobile/rtb/',
    context_on_site_campaign          => '/v2/context/campaigns/',
    context_on_site_direct            => '/v2/context/direct/',
    context_on_site_rtb               => '/v2/context/rtb/',
    context_on_site_stripe            => '/v2/context/stripe/',
    distribution_campaign             => '/v2/distribution/campaigns/',
    search_on_site_campaign           => '/v2/search/campaigns/',
    search_on_site_direct             => '/v2/search/direct/',
    search_on_site_premium            => '/v2/search/premium/',
    video_scenaries                   => '/v2/video/scenaries/',
    video_an_site                     => '/v2/video/sites/',
    video_an_site_instream            => '/v2/video/instream/',
    video_an_site_inpage              => '/v2/video/inpage/',
    dsp                               => '/v2/dsp/connections/',
    internal_context_on_site_campaign => '/v2/internal/context/campaigns/',
    internal_search_on_site_campaign  => '/internal_search_on_site_campaign/list/',
    internal_context_on_site_direct   => '/v2/internal/context/direct/',
    internal_context_on_site_content  => '/v2/internal/context/content/',
    internal_context_on_site_rtb      => '/v2/internal/context/rtb/',
    internal_context_on_site_stripe   => '/internal_context_on_site_stripe/list/',
    internal_search_on_site_direct    => '/internal_search_on_site_direct/list/',
    internal_search_on_site_premium   => '/internal_search_on_site_premium/list/',
    internal_mobile_app               => '/v2/internal/mobile/applications/',
    internal_mobile_app_rtb           => '/v2/internal/mobile/rtb/',
    ssp_mobile_app_settings           => '/v2/ssp/moderation/',
    ssp_context_on_site_campaign      => '/v2/ssp/moderation/',
    ssp_video_an_site                 => '/v2/ssp/moderation/',
    context_on_site_adblock           => '/v2/context/adblock/',
    moderation                        => '/v2/moderation/',
);

sub get_accessors_by_tag {
    my ($self, $tag) = @_;

    my $app_stash = package_stash('QBit::Application');
    my $tags = $app_stash->{'__TAGS_ACCESSORS__'} || {};

    throw gettext('Accessors tag "%s" not found in "%s"', $tag, join(', ', keys %$tags)) unless $tags->{$tag};

    return $tags->{$tag};
}

sub get_block_accessors_by_page_n_tag {
    my ($self, $page_accessor, $tag) = @_;

    my $app_stash = package_stash('QBit::Application');
    my $tags = $app_stash->{'__TAGS_ACCESSORS__'} || {};

    throw gettext('Accessors tag "%s" not found in "%s"', $tag, join(', ', keys %$tags)) unless $tags->{$tag};

    my %tag_hash = map {$_ => 1} @{$tags->{$tag}};

    return [
        map {$app_stash->{'__NEED_ACCESSORS__'}->{$_}->{accessor}}
          grep {
                 $app_stash->{'__NEED_ACCESSORS__'}->{$_}->{models}->{page}
              && $page_accessor eq $app_stash->{'__NEED_ACCESSORS__'}->{$_}->{models}->{page}
              && exists $tag_hash{$app_stash->{'__NEED_ACCESSORS__'}->{$_}->{accessor}}
          }
          keys %{$app_stash->{'__NEED_ACCESSORS__'}}
    ];
}

sub get_site_model_accessors {
    my ($self) = @_;

    return $self->get_accessors_by_tag('site_model');
}

sub get_video_model_accessors {
    my ($self) = @_;

    return $self->get_accessors_by_tag('video_model');
}

sub get_source_model_accessors {$_[0]->get_accessors_by_tag('source_model')}

sub get_application_model_accessors {
    my ($self) = @_;

    return $self->get_accessors_by_tag('application_model');
}

sub get_rtb_model_accessors {
    my ($self) = @_;

    return $self->get_accessors_by_tag('rtb_model');
}

sub get_rtb_models {
    my ($self) = @_;

    return [map {$self->app->$_} @{$self->get_rtb_model_accessors()},];
}

sub get_block_model_accessors {
    my ($self) = @_;

    return $self->get_accessors_by_tag('block_model');
}

sub get_block_model_names {
    my ($self) = @_;

    my @result = ();
    foreach my $accessor (@{$self->get_page_model_accessors()}) {
        push(@result, @{$self->app->$accessor->get_block_model_names()});
    }

    return [sort @result];
}

sub get_external_product_accessors {
    my ($self) = @_;

    return $self->get_accessors_by_tag('external');
}

sub get_internal_product_accessors {
    my ($self) = @_;

    return $self->get_accessors_by_tag('internal');
}

=head1 get_product_structure_for_page_id

    ->get_product_structure_for_page_id( $page_id );

Возвращает структуру данных с информацией о продукте с указанным Page ID.

Метод считает что в системе может быть только одни продукт с определенным
Page ID.

Метод действует с учетом прав пользователя. Если нет продукта с указанным
Page ID, то в возвращаемом хэше ключ status равен "not_found". Если есть продукт с указанным
Page ID, но у пользователя нет к ним доступа, то status = "no_access".
Если, же есть продукт с указанным Page ID и у пользователя
есть к нему доступ, то возвращается структура данных с информацией о
продукте и status = "ok".

Пример:

    ->get_product_structure_for_page_id( 13150 );

вернет:

    {
        status => "ok",
        data   => {
            internal_context_on_site_campaign => {
                # модель данных
                model => 'internal_context_on_site_campaign',

                # название продукта на языке пользователя
                name  => 'Внутренние контекстные площадки',

                # урл, по которому показывается конкретно этот продукт
                url   => '/internal_context_on_site_campaign/list?search_json=["AND",[["id","IN",["13150"]]]]',
        },
    }

Ключи хеша сейчас это названием модели, но их логика может поменятся, поэтому
закладыватся на них не нужно.

=cut

sub get_product_structure_for_page_id {
    my ($self, $page_id) = @_;

    throw Exception::Validation::BadArguments gettext("No Page ID") unless defined($page_id);

    my $avaliable_to_user = {};
    my $all               = {};

    my $all_pages = $self->app->all_pages->get_all(fields => [qw(model)], filter => {page_id => $page_id});

    throw gettext('Page not found for page_id "%s"', $page_id) unless @$all_pages;
    throw gettext('Two identical pages with page_id "%s"', $page_id) if @$all_pages > 1;

    my $product = $all_pages->[0]{'model'};

    throw gettext('Model not found for page_id "%s"', $page_id) unless $product;
    throw gettext('Unknown accessor "%s"',            $product) unless $self->app->can($product);

    my $page_id_field = $self->app->$product->get_page_id_field_name();

    my $result = $self->app->$product->get_all(
        filter => {$page_id_field => $page_id},
        fields => [$page_id_field, 'multistate'],
    );

    if (@$result == 1) {
        my $is_deleted = $self->app->$product->check_multistate_flag($result->[0]->{'multistate'}, 'deleted');
        $avaliable_to_user->{$product} = {
            model => $product,
            name  => $self->app->$product->get_product_name(),
            url   => $self->_get_page_id_search_url(
                model      => $product,
                page_id    => $page_id,
                is_deleted => $is_deleted,
            ),
        };
    }

    {
        my $tmp_rights = $self->app->add_tmp_rights($self->app->$product->get_right('view_all'));

        my $result = $self->app->$product->get_all(fields => [$page_id_field], filter => {$page_id_field => $page_id},);

        if (@$result == 1) {
            $all->{$product} = TRUE;
        }
    }

    my $answer;

    if (%$avaliable_to_user) {
        $answer = {status => 'ok', data => $avaliable_to_user};
    } elsif (%$all) {
        $answer = {status => 'no_access'};
    } else {
        $answer = {status => 'not_found'};
    }

    return $answer;
}

sub get_page_model_accessors {
    my ($self) = @_;

    return $self->get_accessors_by_tag('page_model');
}

# Возвращает список аксессоров пейджовых моделей которые не нужно светить наружу
sub get_special_page_model_accessors {
    my ($self) = @_;

    return ['distribution_campaign'];
}

sub get_statistics_accessors {
    my ($self) = @_;

    return $self->get_accessors_by_tag('statistics');
}

sub get_statistics_products {
    my ($self) = @_;

    return $self->get_accessors_by_tag('statistics_products');
}

sub _get_login_search_url {
    my ($self, %opts) = @_;

    my $login = delete($opts{'login'});
    throw Exception::Validation::BadArguments gettext("No login") unless defined($login);

    my $model = delete($opts{'model'});
    throw Exception::Validation::BadArguments gettext("No model") unless defined($model);

    my $is_deleted = delete($opts{'is_deleted'});
    throw Exception::Validation::BadArguments gettext("No is_deleted") unless defined($is_deleted);

    my $url = $self->_get_url_by_model($model, for_deleted => $is_deleted);

    return $self->_get_url_with_search(
        $url,
        {
            format     => 'search={"owner.login":["IN",["%s"]]}',
            old_format => 'search_json=["AND",[["owner","MATCH",["login","=","%s"]]]]'
        },
        $login
    );
}

sub _get_page_id_search_url {
    my ($self, %opts) = @_;

    my $page_id = delete($opts{'page_id'});
    throw Exception::Validation::BadArguments gettext("No Page ID") unless defined($page_id);

    my $model = delete($opts{'model'});
    throw Exception::Validation::BadArguments gettext("No model") unless defined($model);

    my $is_deleted = delete($opts{'is_deleted'});
    throw Exception::Validation::BadArguments gettext("No is_deleted") unless defined($is_deleted);

    my $url = $self->_get_url_by_model($model, for_deleted => $is_deleted);

    return $self->_get_url_with_search(
        $url,
        {
            format     => 'search={"%s":["IN",["%s"]]}',
            old_format => 'search_json=["AND",[["%s","IN",["%s"]]]]'
        },
        $self->app->$model->get_page_id_field_name(),
        $page_id
    );
}

sub _get_url_by_model {
    my ($self, $model, %opts) = @_;

    throw Exception::Validation::BadArguments gettext('Expected param "model"') unless defined($model);

    my $url = $URL_BY_MODEL{$model} // throw gettext('Url for model "%s" not found', $model);

    my $old_format_url = $url =~ /list\/$/;

    if ($opts{'for_deleted'}) {
        if ($old_format_url) {
            $url .= '?deleted=1&';
        } else {
            $url .= 'archive/?';
        }
    } else {
        $url .= '?';
    }

    return {url => $url, old_format_url => $old_format_url};
}

sub _get_url_with_search {
    my ($self, $url, $formats, @args) = @_;

    my $main_url = $url->{'url'};
    if ($url->{'old_format_url'}) {
        $main_url .= $formats->{'old_format'};
    } else {
        $main_url .= $formats->{'format'};
    }

    return sprintf($main_url, @args);
}

#создаём ссылку, которая будет управлять ключём в __DBH__, где хранится коннект
my $coro_specific = Coro::Specific->new();

sub get_product_structure {
    my ($self, $field, $value) = @_;

    my $filter = {$field => $value};
    if ($field eq 'bundle_id') {
        $filter = {'domain' => $value};
    } elsif ($field eq 'pi_id') {
        $filter = [
            'page_id',
            '= ANY',
            $self->app->context_on_site_campaign->query(
                fields => $self->app->context_on_site_campaign->_get_fields_obj(['page_id']),
                filter => $self->app->context_on_site_campaign->get_db_filter({pi_id => $value})
              )->union(
                $self->app->search_on_site_campaign->query(
                    fields => $self->app->search_on_site_campaign->_get_fields_obj(['page_id']),
                    filter => $self->app->search_on_site_campaign->get_db_filter({pi_id => $value})
                )
              )
        ];
    } elsif ($field eq 'adfox_login') {
        $filter = [
            'owner_id',
            '= ANY',
            $self->app->partner_db->query->select(
                table  => $self->app->partner_db->user_adfox,
                fields => [qw(user_id)],
                filter => {adfox_login => $value}
            )
        ];
    } elsif ($field eq 'domain') {
        my $domain = get_domain($value);
        if (defined $domain) {
            my $www_domain = 'www.' . $domain;
            $filter = [OR => [[$field => '=' => \$domain], [$field => '=' => \$www_domain]],];
        }
    }

    my @ssp_pages = grep {$self->app->$_->isa('Application::Model::Page::SSP')} @{$self->get_page_model_accessors()};
    my $all_pages = $self->app->all_pages->query(
        fields => $self->app->all_pages->_get_fields_obj([qw(id model login)]),
        filter => $self->app->partner_db->filter($filter)->and([model => 'NOT IN' => \\@ssp_pages])
    )->get_all();
    my $structure = {'__DICT__' => {}, '__DATA__' => {}};

    foreach my $row (@$all_pages) {
        $structure->{'__DICT__'}{$row->{'model'}}{$row->{'id'}} = $row->{'login'};
    }

    #задаём значение переменной $DBH_KEY и обработчик сигнала
    local $QBit::Application::Model::DB::DBH_KEY = $coro_specific;
    local $Coro::State::DIEHOOK                  = \&QBit::Exceptions::die_handler;

    my @threads = ();
    foreach my $product (keys(%{$structure->{'__DICT__'}})) {
        push(
            @threads,
            async(
                \&get_structure_product_with_blocks,
                $self, $product, [keys(%{$structure->{'__DICT__'}{$product}})], $structure
            )
        );
    }

    # @$results : array of arrays of results from coro routines
    my $results = $self->join_all_coros(\@threads);

    my @result = ();

    foreach my $login (sort(keys(%{$structure->{'__DATA__'}}))) {
        my @products = ();
        foreach my $product (sort(keys(%{$structure->{'__DATA__'}{$login}}))) {
            push(
                @products,
                {
                    model    => $product,
                    name     => $self->app->$product->get_product_name(),
                    products => $structure->{'__DATA__'}{$login}{$product}
                }
            );
        }

        push(@result, {login => $login, products => \@products});
    }

    return \@result;
}

sub get_structure_product_with_blocks {
    my ($self, $product, $ids, $structure, %opts) = @_;

    my $model_fields = $self->app->$product->get_model_fields();

    my ($domain_name, $page_id_name);
    if (exists($model_fields->{'domain'})) {
        $domain_name = 'domain';
    } elsif (exists($model_fields->{'app_bundle_id'})) {
        $domain_name = 'app_bundle_id';
    } elsif (exists($model_fields->{'store_id'})) {
        $domain_name = 'store_id';
    } elsif (exists($model_fields->{'caption'})) {
        $domain_name = 'caption';
    } else {
        throw gettext('Field for domain not found');
    }

    $page_id_name = $self->app->$product->get_page_id_field_name();

    my @additional_fields = ();
    if ($self->app->$product->can('get_block_model_names')) {
        push(@additional_fields,
            grep {exists($model_fields->{$_})}
            map  {$_ . "_count"} @{$self->app->$product->get_block_model_names()});
    }

    #для каждой рутины задаём уникальное значение для DBH_KEY
    #создаём нвоый коннект и патчим dbh
    $$coro_specific = "_$product";
    $self->app->partner_db->_connect(use_coro_mysql => TRUE);

    my $result = [];

    try {
        $result = $self->app->$product->get_all(
            fields => [$domain_name, 'id', $page_id_name, @additional_fields, 'is_deleted'],
            filter => {id => $ids},
        );
    }
    catch Exception::DB::NoFieldsAvailable with {};

    foreach my $el (@$result) {
        my $sub_product;

        if (@additional_fields) {
            push(@$sub_product,
                map {{model => $_, count => delete($el->{$_ . '_count'}),}}
                  @{$self->app->$product->get_block_model_names()});
        }

        $el->{'products'} = $sub_product if $sub_product;

        my $id = $el->{'id'};

        $el->{'page_id'} = delete($el->{$page_id_name});
        $el->{'domain'}  = delete($el->{$domain_name});
        delete($el->{'id'});

        my $login = $structure->{'__DICT__'}{$product}{$id};
        push(@{$structure->{'__DATA__'}{$login}{$product}}, $el);
    }

    return $result;
}

=head2 get_model_data_from_public_id

    my $model_name = $app->product_manager->get_model_data_from_public_id( 'R-I-49688-1' );

Это вернет хеш] {
    accessor  => 'internal_context_on_site_rtb',
    public_id => 'R-I-49688-1'
}

В случае ошибки может выдать:

    {
        error => 'no_data_for_zero_block',
        error_message => 'Для нулевого блока не существует данных',
    }

=cut

# TODO: унести в Utils::PublicID
sub get_model_data_from_public_id {
    my ($self, $public_id, $is_relaxed) = @_;

    if ($public_id =~ /-0\z/) {
        return {
            error         => 'no_data_for_zero_block',
            error_message => gettext('No info for zero block'),
        };
    }

    # TODO: заменить на check_public_ids
    foreach (@{$self->get_page_model_accessors()}) {
        foreach my $block_model ($self->app->$_->get_block_models()) {

            # PI-10318 - при $is_relaxed в случае Директа, также ищем в RTB
            my @public_ids = ($public_id);
            if ($is_relaxed && $public_id =~ m/^D-/) {
                # Direct -> RTB
                (my $rtb_public_id = $public_id) =~ s/^D-/R-/;
                push @public_ids, $rtb_public_id;
            }

            foreach my $public_id (@public_ids) {

                my $id = $block_model->_split_id($public_id);
                next if ref($id) ne 'HASH';

                my $obj = $block_model->get($id, fields => ['id']);
                if ($obj) {
                    return {
                        accessor  => $block_model->accessor(),
                        public_id => $public_id
                    };
                }

                my $right = $block_model->get_right('view_all');

                unless ($self->check_rights($right)) {
                    $obj = $block_model->partner_db_table()->get_all(
                        fields => ['id'],
                        filter => [
                            'AND',
                            [
                                $id,
                                (
                                    $block_model->is_block_table_with_multiple_models()
                                    ? ['model', '=', \$block_model->accessor()]
                                    : ()
                                ),
                            ]
                        ],

                    );

                    throw Exception::Denied gettext('Access denied') if $obj;
                }
            }
        }
    }

    throw Exception::Validation::BadArguments gettext('Incorrect public_id %s', $public_id);
}

sub get_old_block_model_accessors {
    my ($self) = @_;

    return [
        qw(
          context_on_site_content
          context_on_site_market
          context_on_site_market_api
          context_on_site_mcb
          indoor_block
          internal_context_on_site_rtb
          internal_mobile_app_rtb
          mobile_app_rtb
          outdoor_block
          search_on_site_market
          search_on_site_market_api
          search_on_site_mcb
          )
    ];
}

sub get_adfox_blocks_model_accessors {
    my ($self) = @_;

    my $block_accessors = $self->get_block_model_accessors();

    my @blocks_with_adfox = ();
    foreach (sort @$block_accessors) {
        if ($self->app->$_->DOES('Application::Model::Role::Block::Has::AdfoxBlock')) {
            push(@blocks_with_adfox, $_);
        }
    }

    return \@blocks_with_adfox;
}

TRUE;
