package Application::Model::QBitValidatorChecker;

=encoding UTF-8

=cut

use qbit;

use base qw(QBit::Application::Model);

use Exception::Validator::Fields;
use PiConstants qw($TECHNICAL_RTB_BLOCK_ID $NOTIFICATION_FIELD_LIMIT);
use Utils::Logger qw/ INFOF WARNF ERROR INFO WARN/;
use Utils::MonitoringUtils qw(get_pjapi);

sub accessor {'qbit_validator_checker'}

=head2 get_model_names_that_use_qbit_validator

    my @names = $app->qbit_validator_checker->get_model_names_that_use_qbit_validator();

=cut

sub get_model_names_that_use_qbit_validator {
    my ($self) = @_;

    my $app = $self->app;
    my %ignore = map {$_ => TRUE} qw(
      mobile_mediation_block
      context_on_site_rtb
      internal_context_on_site_rtb
      mobile_app_rtb
      internal_mobile_app_rtb
      widgets
      );
    my @names =
      sort
      grep {!$ignore{$_} && $app->$_->can('get_template')}
      keys %{$app->get_models()};

    return @names;
}

=head2 check_all_elements_in_model

    my ($is_ok, $details) = $app->qbit_validator_checker->check_all_elements_in_model(accessor => $name);

=cut

sub check_all_elements_in_model {
    my ($self, %opts) = @_;
    my (
        $name,      $page_ids,        $step,               $random_start_position, $verbose,
        $instances, $instance_number, $is_send_to_juggler, $heartbeat
       )
      = @opts{
        qw(
          accessor page_ids step start verbose
          instances instance_number is_send_to_juggler heartbeat
          )
      };

    $page_ids //= [];
    $is_send_to_juggler //= 0;
    my $model   = $self->app->$name;
    my $is_ok   = 1;
    my $details = {
        model_name         => $name,
        package            => ref($model),
        all_elements_count => 0,
    };

    my $page_field_name =
        $model->can('get_page_id_field_name')
      ? $model->get_page_id_field_name()
      : undef;

    if (@$page_ids && !$page_field_name) {
        INFOF(q[SKIP model=%s because it hasn't page id field], $name);
        return (1, $details);
    }

    # некоторые поля проверяются на уникальность
    my %ignore_fields = (cookie_match => [qw(tag)], statistics_reports => [qw(_OWNER_ID _LEVEL)]);
    my $opts_and_filters_and_every_nth = {
        context_on_site_direct => {
            db_filter => ['id' => '<>' => \0],
            opts => {fix => {page_id => {type => 'int_un'}}},
        },
        internal_context_on_site_content => {
            db_filter => ['id' => '<>' => \0],
            opts => {fix => {page_id => {type => 'int_un'}}},
        },
        internal_context_on_site_direct => {
            db_filter => ['id' => '<>' => \0],
            opts => {fix => {page_id => {type => 'int_un'}}},

        },
        internal_context_on_site_natural => {
            db_filter => ['id' => '<>' => \0],
            opts => {fix => {page_id => {type => 'int_un'}}},

        },
        internal_search_on_site_direct => {
            db_filter => ['id' => '<>' => \0],
            opts => {fix => {page_id => {type => 'int_un'}}},
        },
        internal_search_on_site_premium => {
            db_filter => ['id' => '<>' => \0],
            opts => {fix => {page_id => {type => 'int_un'}}},
        },
        context_on_site_rtb => {
            db_filter => ['id' => '<>' => \$TECHNICAL_RTB_BLOCK_ID],
            opts => {fix => {page_id => {type => 'int_un'}}},
        },
        internal_context_on_site_rtb => {
            db_filter => ['id' => '<>' => \$TECHNICAL_RTB_BLOCK_ID],
            opts => {fix => {page_id => {type => 'int_un'}}},
        },
        search_on_site_direct => {
            db_filter => ['id' => '<>' => \0],
            opts => {fix => {page_id => {type => 'int_un'}}},
        },
        # Хак на переходный период
        # Когда все площадки с нужными facility_type выставят business_oid, нужно это выпилить
        indoor       => {opts => {fix => {business_oid => {type => 'int_un', optional => TRUE}}},},
        indoor_block => {opts => {fix => {page_id      => {type => 'int_un'}}},},
        internal_mobile_app_rtb =>
          {opts => {fix => {campaign_id => {type => 'int_un'}, page_id => {type => 'int_un'}}},},
        context_on_site_adblock         => {opts => {fix => {campaign_id => {type => 'int_un'}}},},
        context_on_site_stripe          => {opts => {fix => {page_id     => {type => 'int_un'}}},},
        context_on_site_content         => {opts => {fix => {page_id     => {type => 'int_un'}}},},
        context_on_site_natural         => {opts => {fix => {page_id     => {type => 'int_un'}}},},
        internal_context_on_site_stripe => {opts => {fix => {page_id     => {type => 'int_un'}}},},
        # с фичой simple_inapp можно создать приложение без стора
        # но проверить в зависимости от фичи мы не можем, потому что на mobile_app нет owner
        mobile_app => {
            opts => {
                fix =>
                  {store_id => {type => 'scalar', optional => TRUE}, store_url => {type => 'scalar', optional => TRUE}}
            },
        },
        mobile_app_rtb => {opts => {fix => {campaign_id => {type => 'int_un'}, page_id => {type => 'int_un'}}},},
        mobile_app_settings => {opts => {fix => {application_id => {type => 'int_un'}}},},
        moderation_reason   => {
            opts => {
                fix => {
                    manager_txt => {optional => TRUE, len_max => 512, len_min => 1},
                    partner_txt => {optional => TRUE, len_max => 512, len_min => 1},
                },
            },
        },
        simple_notification => {
            opts => {
                fix => {
                    message     => {len_max  => 1024, len_min => 1},
                    button_text => {optional => TRUE, len_max => 1024},
                    title       => {optional => TRUE, len_max => 1024},
                },
            },
        },
        notification => {
            opts => {
                fix => {
                    message => {
                        len_min => 1,
                        check   => sub {
                            my ($qv, $val) = @_;

                            my $value = length($val);
                            my $limit = $NOTIFICATION_FIELD_LIMIT->{message}{$qv->data->{type}};

                            throw Exception::Validator::Fields gettext('%s must between %s and %s', 'message', 1,
                                $limit)
                              if $value < 1 || $value > $limit;
                        },
                    },
                    caption => {
                        len_min => 1,
                        check   => sub {
                            my ($qv, $val) = @_;

                            my $value = length($val);
                            my $limit = $NOTIFICATION_FIELD_LIMIT->{caption}{$qv->data->{type}};

                            throw Exception::Validator::Fields gettext('%s must between %s and %s', 'caption', 1,
                                $limit)
                              if $value < 1 || $value > $limit;
                        },
                    },
                    button_caption => {
                        optional => TRUE,
                        len_min  => 1,
                        check    => sub {
                            my ($qv, $val) = @_;

                            my $value = length($val);
                            my $limit = $NOTIFICATION_FIELD_LIMIT->{button_caption}{$qv->data->{type}};

                            throw Exception::Validator::Fields gettext('%s must between %s and %s', 'button_caption', 1,
                                $limit)
                              if $value < 1 || $value > $limit;
                        },
                    },
                    expectation_caption => {optional => TRUE, len_max => 30, len_min => 1},
                }
            }
        },
        site => {
            opts => {
                fix => {
                    domain => {
                        len_max => 255,
                        len_min => 4,
                        check   => sub {
                            my ($qv, $domain) = @_;

                            throw Exception::Validator::Fields gettext('Domain doesn\'t exist')
                              unless get_domain($domain);
                          }
                    }
                },
            },
        },
        # for dooh blocks check only that <=6 mds_avatars.id are in the photo_id_list
        (
            map {
                $_ => {
                    opts => {
                        fix => {
                            photo_id_list => {
                                type     => 'array',
                                all      => {type => 'scalar'},
                                optional => TRUE,
                                check    => sub {
                                    my ($qv, $photo_id_list) = @_;

                                    my $DOOH_BLOCK_PHOTOS_MAX = $qv->app->get_dooh_block_photos_max();
                                    if ($DOOH_BLOCK_PHOTOS_MAX < @$photo_id_list) {
                                        throw Exception::Validator::Fields gettext(
                                            "No more than %d photoes allowed for one block",
                                            $DOOH_BLOCK_PHOTOS_MAX);
                                    }

                                    my $user_photos = $qv->app->mds_avatars->get_all(
                                        fields => ['id'],
                                        filter => {'id' => $photo_id_list},
                                    );

                                    unless (@$photo_id_list == @$user_photos) {
                                        my %hh = map {$_->{id} => 1} @$user_photos;
                                        throw Exception::Validator::Fields gettext("Photo link [%s] is unacceptable",
                                            join(',', grep {!exists $hh{$_}} @$photo_id_list));
                                    }
                                  }
                            }
                        },
                    },
                  }
              } qw(indoor_block outdoor_block)
        ),
    };
    if (!defined($step) || int($step) < 1) {
        $step = 1;
    }
    my @ignore_fields = $self->_get_field_names_to_ignore($name);
    push(@ignore_fields, @{$ignore_fields{$name}}) if exists($ignore_fields{$name});

    my $model_fields = $model->get_model_fields;
    my %pk           = map {$_ => TRUE} grep {$model_fields->{$_}{'pk'}} keys(%$model_fields);
    my ($pk_field)   = keys(%pk);

    my $fields = arrays_difference([keys(%$model_fields)], \@ignore_fields);

    my $block_accessors = $self->app->product_manager->get_block_model_accessors();
    my $is_block = grep {$name eq $_} @$block_accessors;

    INFOF("--- $name ---");
    INFOF($is_block ? 'is BLOCK accessor' : 'is NOT BLOCK accessor');

    my @filters    = ();
    my @db_filters = ();

    if (my $db_filter = $opts_and_filters_and_every_nth->{$name}{db_filter}) {
        push(@db_filters, $db_filter);
    }

    if (my $model_filter = $opts_and_filters_and_every_nth->{$name}{model_filter}) {
        push(@filters, $model_filter);
    }

    if ($instances && $instance_number) {
        push(@db_filters,
            [{'MOD' => [($is_block ? 'page_id' : $page_field_name), \$instances]}, '=', \($instance_number - 1)]);
    }

    if (@$page_ids && $page_field_name) {
        push(@db_filters, [($is_block ? 'page_id' : $page_field_name) => 'IN' => \$page_ids]);
    }
    my $fields_to_request = array_uniq(@$fields, keys(%pk), $page_field_name // ());

    INFOF('Requesting all elements count...');

    my $all_elements_count;
    my $all_id;
    if ($is_block) {
        throw Exception 'Table "all_blocks" do not support filters of a model' if @filters;

        $all_elements_count = $self->app->partner_db->all_blocks->get_all(
            fields => {cnt => {COUNT => ['id']}},
            filter => ['AND' => [['model', '=', \$name], @db_filters]]
        )->[0]{'cnt'};
    } else {
        my $db_filter;
        if (@filters || @db_filters) {
            $db_filter = $model->get_db_filter(['AND' => \@filters]);

            $db_filter->and(['AND', \@db_filters]) if @db_filters;
        }

        $all_id = $model->get_all(
            fields => [sort keys %pk],
            (
                $db_filter
                ? (filter => $db_filter)
                : ()
            )
        );
        $all_elements_count = @$all_id;
    }

    $details->{'all_elements_count'} = $all_elements_count;
    INFOF('    ... %d elements to check', $details->{'all_elements_count'});

    INFOF('Requesting every Nth id...') if $step > 1;

    my $id_every_nth = [];

    if (defined($random_start_position)) {
        WARNF('random start posision should be more than 0 and lower every_nth %d', $step)
          if ($random_start_position < 1 || $random_start_position > $step);

        $random_start_position--;
    } else {
        $random_start_position = int(rand($step));
    }

    if ($step > 1) {
        INFOF('%d is random start posision', $random_start_position + 1);
    }
    if ($is_block) {
        $id_every_nth = $self->app->partner_db->all_blocks->get_all(
            fields => ['id', 'page_id'],
            filter => [
                'AND' => [
                    [['%' => ['id_autoincrement', \$step]], '=', \$random_start_position],
                    ['model', '=', \$name], @db_filters
                ]
            ]
        );
    } else {
        if ($step > 1) {
            for (my $i = $random_start_position; $i < @$all_id; $i += $step) {
                push @$id_every_nth, $all_id->[$i];
            }
        } else {
            $id_every_nth = $all_id;
        }
    }

    INFOF('    ... %d every Nth id returned', scalar @$id_every_nth) if $step > 1;

    my $count = 0;
    my $limit = 1000;

    my $pja = get_pjapi($self);

    my $service = "qbit_validator.${name}" . (defined($instance_number) ? "_$instance_number" : '');
    while (my @block_of_id = splice @$id_every_nth, 0, $limit) {
        $heartbeat->($self->app) if $heartbeat;

        INFOF('Requesting ' . @block_of_id . ' elements...') if $verbose;

        my @pairs = ();
        for my $keys (@block_of_id) {
            push @pairs,
              [
                'AND' => [
                    $is_block
                    ? (['id', '=', $keys->{'id'}], ['page_id', '=', $keys->{'page_id'}])
                    : map {[$_, '=', $keys->{$_}]} keys %pk
                ]
              ];
        }

        my $elements = $model->get_all(
            fields => $fields_to_request,
            filter => ['AND' => [['OR' => [@pairs]], @filters]]
        );

        INFOF('    ... %d elements returned', scalar @$elements) if $verbose;

        foreach my $element (@$elements) {
            INFOF(
                '%d/%d processed (%s)',
                $count + $random_start_position + 1,
                $details->{'all_elements_count'},
                (
                    join '; ',
                    map {join '=', $_, $element->{$_} // 'undef'} @{array_uniq(keys(%pk), $page_field_name // ())}
                )
            ) if $verbose;

            $count += $step;

            my $pk_value;
            foreach (keys(%pk)) {
                $pk_value->{$_} = $element->{$_};
                delete($element->{$_}) unless in_array($_, $fields);
            }

            map {delete($element->{$_})} @{$ignore_fields{$name}} if exists($ignore_fields{$name});

            my $qv = QBit::Validator->new(
                data => $element,
                app  => $model,
                $model->get_template(
                    $opts_and_filters_and_every_nth->{$name}{opts}
                    ? %{$opts_and_filters_and_every_nth->{$name}{opts}}
                    : (fields => $fields, values => $pk_value)
                ),
            );
            if ($qv->has_errors) {
                my @fields_with_errors = $qv->get_fields_with_error();

                my %error_fields;

                foreach my $f (@fields_with_errors) {
                    #pk поля как правило проверяются при добавлении get_all_campaigns_for_adding
                    #затем площадка может быть удалена или находится в другом статусе.
                    my $field_name = $f->{'path'}[0] // 'ERROR_IN_ROOT';

                    next if $pk{$field_name};

                    $error_fields{$field_name} = {
                        msg   => join('; ', @{$f->{'msgs'}}),
                        value => $element->{$field_name},
                        path  => $f->{'path'},
                    };
                }

                if (%error_fields) {
                    my $data = {
                        pk           => $pk_value,
                        error_fields => \%error_fields,
                    };
                    push @{$details->{error_elements}}, $data;

                    my $error_message = gettext(
                        "## %s (%s%s)\nCurrent problem elements: %s/%s\nLast element: %s - %s\n",
                        $details->{'package'},
                        $details->{'model_name'},
                        (defined($instance_number) ? "#$instance_number" : ''),
                        scalar(@{$details->{'error_elements'}}),
                        $details->{'all_elements_count'},
                        to_json($pk_value),
                        ($verbose ? to_json(\%error_fields) : join(', ', keys(%error_fields)))
                    );

                    if ($is_send_to_juggler) {
                        $pja->send(
                            events => [
                                {
                                    service     => $service,
                                    status      => 'CRIT',
                                    description => $error_message,
                                }
                            ]
                        );

                    }

                    $is_ok = 0;
                    INFO $error_message;
                }
            }
        }
    }

    if ($is_send_to_juggler) {
        if ($is_ok) {
            $pja->send(
                events => [
                    {
                        service => $service,
                        status  => 'OK',
                    }
                ]
            );
        } else {
            my $error_message = gettext(
                "## %s (%s%s)\nTotal problem elements: %s/%s\n",
                $details->{'package'},
                $details->{'model_name'},
                (defined($instance_number) ? "#$instance_number" : ''),
                scalar(@{$details->{'error_elements'}}),
                $details->{'all_elements_count'},
            );
            ERROR {
                message     => $error_message,
                extra       => $details->{'error_elements'},
                fingerprint => ['Cron', 'qbit_validator_checker', $service, 'total'],
            };
            $pja->send(
                events => [
                    {
                        service     => $service,
                        status      => 'CRIT',
                        description => $error_message,
                    }
                ]
            );
        }
    }

    return ($is_ok, $details);
}

sub _get_field_names_to_ignore {
    my ($self, $model_name) = @_;

    my $model        = $self->app->$model_name;
    my $model_fields = $model->get_model_fields();

    my @fields = keys(%{$model->get_model_fields});

    if ($model->isa('Application::Model::ValidatableMixin')) {
        return grep {!$model_fields->{$_}{'need_check'}} @fields;
    }

    my @names_to_ignore;

    try {
        $model->get_template(fields => \@fields);
    }
    catch Exception::Validation::BadArguments with {
        my $error_message = $_[0]->{'text'};

        if (($error_message =~ /Неизвестные поля: (.*)/) || ($error_message =~ /Unknown fields: (.*)/)) {
            @names_to_ignore = split(/, /, $1);
        }

    };

    return @names_to_ignore;
}

sub generate_juggler {
    my ($self, $pja) = @_;

    return FALSE unless in_array($self->get_option('stage', 'unknown'), [qw(production preprod)]);

    my @children = ();

    my $juggler_host = $self->get_option('api_juggler')->{'host'};

    my @models = $self->get_model_names_that_use_qbit_validator();

    my $instances_by_model = {
        context_on_site_campaign => 5,
        video_an_site_instream   => 2,
    };

    my %skip_models = map {$_ => TRUE} qw(
      context_on_site_rtb
      internal_context_on_site_rtb
      mobile_app_rtb
      design_templates
      );

    foreach my $model (@models) {
        next if $skip_models{$model};
        my @services = qw();
        if (my $instances = $instances_by_model->{$model}) {
            push(@services, "qbit_validator.${model}_$_") for (1 .. $instances);
        } else {
            push(@services, "qbit_validator.$model");
        }

        foreach my $service (@services) {
            $pja->add_or_update(
                service => $service,
                tags    => ($self->get_option('api_juggler')->{tags} // []),
                ttl     => "28h",
                meta    => {
                    urls => [
                        {
                            title => 'Что делать, если горит проверка',
                            url   => 'https://wiki.yandex-team.ru/partner/w/juggler-checks/qbit-validator/',
                            type  => 'wiki'
                        }
                    ]
                }
            );

            push(@children, {host => $juggler_host, service => $service});
        }
    }

    # мета проверка со звонком
    $pja->add_or_update(
        service    => 'qbit_validator.aggregator',
        ttl        => '28h',
        aggregator => 'logic_or',
        children   => \@children,
        pronounce =>
          "Кубит валидатор нашел некорректные сущности в базе данных",
        tags => ['show-on-dashboard', 'standard-notification',],
        meta => {
            urls => [
                {
                    title => 'Что делать, если горит проверка',
                    url   => 'https://wiki.yandex-team.ru/partner/w/juggler-checks/qbit-validator/',
                    type  => 'wiki'
                }
            ]
        }
    );
}

1;
