
=head1 Name

Application::Model::QualityCoef - Obtaining of the ad platform quality coefficients.

=head1 Description

L<http://wiki.yandex-team.ru/partner/w/quality-coefficient>

Disabled L<https://st.yandex-team.ru/PI-7970>

=cut

package Application::Model::QualityCoef;

use qbit;

use base qw(QBit::Application::Model);

sub accessor {'quality_coef'}

__PACKAGE__->model_accessors(
    api_conv_db     => 'QBit::Application::Model::API::Yandex::ConvDB',
    partner_db      => 'Application::Model::PartnerDB::QualityCoef',
    api_bk          => 'Application::Model::API::Yandex::BK',
    product_manager => 'Application::Model::ProductManager',
);

=head1 Methods

=head2 get_coef

Returns quality coefficient for the given Page ID.

If there is no data for that Page ID the default value for the quality
coefficient will be returned.

    my $page_id = 118826;
    my $coef = $app->quality_coef->get_coef($page_id); # variable is now `100`

=cut

sub get_coef {
    my ($self, $page_id) = @_;

    my $coef = $self->get_pages_coef(pages => [$page_id])->{$page_id};

    return $coef;
}

=head2 get_coef_history

Returns coefficient history for the given Page ID for the last N days (N is
hardcorde in the sub).

    my $data = $app->quality_coef->get_coef_history($page_id);

Returns

    {
        labels => [
            ...
            "2014-02-10",
            "2014-02-11",
            "2014-02-12",
            "2014-02-13",
        ],
        values => [
            ...
            83,
            83,
            undef,
            undef,
        ],
    }


=cut

sub get_coef_history {
    my ($self, $page_id) = @_;

    my $DAYS_NUMBER = 365;

    my $end_date = curdate(oformat => "db");
    my $start_date = date_sub($end_date, day => $DAYS_NUMBER, iformat => 'db', oformat => 'db');

    my $def_data = $self->partner_db->quality_def_coef->get_all(
        fields => [qw(dt k)],
        filter => [AND => [[dt => '>=' => \$start_date], [dt => '<=' => \$end_date],]],
        order_by => [['dt', TRUE]],
    );
    my %date2def_coef = map {$_->{dt} => $_->{k}} @$def_data;

    my $page_data = $self->partner_db->quality_pages_coef->get_all(
        fields => [qw(dt k)],
        filter => [AND => [[dt => '>=' => \$start_date], [dt => '<=' => \$end_date], [page_id => '=' => \$page_id],]],
        order_by => [['dt', TRUE]],
    );
    my %date2page_coef = map {$_->{dt} => $_->{k}} @$page_data;

    my @dates = dates2array($start_date, $end_date, iformat => 'db', oformat => 'db');

    my $labels;
    my $values;

    foreach my $date (@dates) {
        push @$labels, $date;

        if (exists($date2page_coef{$date})) {
            push @$values, $date2page_coef{$date};
        } elsif (exists($date2def_coef{$date})) {
            push @$values, $date2def_coef{$date};
        } else {
            push @$values, undef;
        }
    }

    my $history = {
        labels => $labels,
        values => $values,
    };

    return $history;
}

=head2 get_def_coef

Get default quality coefficient from the db.

B<Arguments:>

=over

=item

B<%opts> - additional arguments:

=over

=item

B<date> - (date => value), date to get data for.
    Current date is used by default.
    E.g. (date => '2013-08-08')

=back

=back

B<Return value:> scalar, integer number, default quality coefficient on the date.
    E.g. 80

=cut

sub get_def_coef {
    my ($self, %opts) = @_;

    $opts{'date'} ||= curdate(oformat => "db");

    return $self->partner_db->quality_def_coef->get_all(
        fields => [qw(k)],
        filter => [AND => [[dt => '<=' => \$opts{'date'}]]],
        order_by => [['dt', TRUE]],
        limit    => 1,
    )->[0]->{'k'};
}

=head2 get_pages_coef

Get quality coefficient by pages from the db.

B<Arguments:>

=over

=item

B<%opts> - additional arguments:

=over

=item

B<date> - (date => value), date to get data for.
    Current date is used by default.
    E.g. (date => '2013-08-08')

=item

B<pages> - (pages => [page_id_1, page_id_2, ...]), pages to get data for.
    All pages on the date are used by default.
    E.g. (pages => [11111, 22222])

=back

=back

B<Return value:> hash reference, {page_id => k}, quality coefficients for the pages on the date.
    E.g. {11111 => 71, 22222 => 85}

=cut

sub get_pages_coef {
    my ($self, %opts) = @_;

    $opts{'date'} ||= curdate(oformat => "db");

    my $nearest_date = $self->partner_db->quality_pages_coef->get_all(
        fields => [qw(dt)],
        filter =>
          [AND => [[dt => '<=' => \$opts{'date'}], ($opts{'pages'} ? [page_id => IN => \$opts{'pages'}] : ()),]],
        order_by => [['dt', TRUE]],
        limit    => 1,
    )->[0]->{'dt'};

    my $pages_coef = {
        $nearest_date
        ? (
            map {$_->{'page_id'} => $_->{'k'}} @{
                $self->partner_db->quality_pages_coef->get_all(
                    fields => [qw(page_id k)],
                    filter => {
                        dt => $nearest_date,
                        ($opts{'pages'} ? (page_id => $opts{'pages'}) : ())
                    }
                )
              }
          )
        : ()
    };

    if ($opts{'pages'} && (@{$opts{'pages'}} > keys(%$pages_coef))) {
        my $def_coef = $self->get_def_coef($nearest_date ? (date => $nearest_date) : ());
        $pages_coef->{$_} ||= $def_coef foreach @{$opts{'pages'}};
    }

    return $pages_coef;
}

TRUE;
