package Application::Model::Queue;

use qbit;

use base qw(Application::Model::Common RestApi::MultistateModel);

use PiConstants qw(
  $QUEUE_IS_METHOD_DISABLED_KEY
  );

use Time::ETA;
use Utils::Logger qw(ERROR INFOF);
use Utils::MonitoringUtils qw(send_to_graphite);

use Utils::JSON qw(fix_type_for_complex);
use Utils::PublicID qw(split_block_public_id);
use Utils::UniqueID qw(make_unique_id);

use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Queue::NeedRestart;

my @QUEUE_ARRAY = (
    {
        method_name                  => 'statistics_intake',
        method_type                  => 1,
        method_name_view             => d_gettext('Statistics update'),
        view_right                   => 'queue_statistics_intake',
        add_right                    => 'queue_add_statistics_intake',
        tries_max                    => 3,
        estimated                    => 1800,
        allowed_concurrent_execution => 0,
        error_description            => sub {
            my ($error) = @_;

            return gettext('Failed on date: %s', $error->{'date'});
        },
        milestone_number => sub {
            my ($opts) = @_;

            my @dates = dates2array(
                $opts->{'start'}, $opts->{'end'},
                iformat => 'db',
                oformat => 'db',
            );

            return scalar(@dates) * scalar(@{$opts->{'products'}});
        },
        need_notify   => 0,
        has_interface => TRUE,
    },
    {
        method_name      => 'update_in_bk',
        method_type      => 2,
        method_name_view => d_gettext('Update in BK'),
        view_right       => 'queue_update_in_bk',
        add_right        => 'queue_add_update_in_bk',
        tries_max        => 1,
        estimated        => sub {
            @{$_[0]->{page_ids}} * 10;
        },
        allowed_concurrent_execution => 1,
        error_description            => sub {
            my ($error) = @_;

            return gettext('Failed on: %s', join(', ', @$error));
        },
        milestone_number => sub {
            my ($opts) = @_;

            return scalar @{$opts->{'page_ids'}};
        },
        need_notify   => 0,
        has_interface => TRUE,
    },
    {
        method_name                  => 'db_dump',
        method_type                  => 3,
        method_name_view             => d_gettext('DB dump'),
        view_right                   => 'queue_db_dump',
        add_right                    => 'queue_add_db_dump',
        tries_max                    => 1,
        estimated                    => 1800,
        allowed_concurrent_execution => 1,
        milestone_number             => sub {
            my ($opts) = @_;

            return scalar @{$opts->{'page_ids'}} + scalar @{$opts->{'logins'}};
        },
        need_notify => 0,
    },
    {
        method_name                  => 'update_mobile_app_status',
        method_type                  => 4,
        method_name_view             => d_gettext('Update MobileApp status'),
        view_right                   => 'queue_update_mobile_app_status',
        add_right                    => 'queue_add_update_mobile_app_status',
        try_after                    => 60,
        tries_max                    => 100,
        estimated                    => 60,
        allowed_concurrent_execution => 0,
        error_description            => sub {
            my ($error) = @_;

            return to_json($error);
        },
        milestone_number => 1,
        need_notify      => 1,
    },
    {
        method_name                  => 'mark_read_fns_notification',
        method_type                  => 5,
        method_name_view             => d_gettext('Mark read FNS notification'),
        view_right                   => 'queue_mark_read_fns_notification',
        add_right                    => 'queue_add_mark_read_fns_notification',
        tries_max                    => 100,
        try_after                    => 60,
        estimated                    => 60,
        allowed_concurrent_execution => 1,
        error_description            => sub {
            my ($error) = @_;

            return to_json($error);
        },
        milestone_number => 1,
        need_notify      => 1,
    },
    {
        method_name                  => 'send_email_to_processing',
        method_type                  => 6,
        method_name_view             => d_gettext('Send email to processing'),
        view_right                   => 'queue_send_email_to_processing',
        add_right                    => 'queue_add_send_email_to_processing',
        try_after                    => 60,
        tries_max                    => 100,
        estimated                    => 60,
        allowed_concurrent_execution => 1,
        error_description            => sub {
            my ($error) = @_;

            return to_json($error);
        },
        milestone_number => 1,
        need_notify      => 1,
        has_interface    => TRUE,
    },
    {
        method_name       => 'change_blocked_brands',
        method_type       => 7,
        method_name_view  => d_gettext('Change list blocked brands'),
        view_right        => 'queue_change_blocked_brands',
        add_right         => 'queue_add_change_blocked_brands',
        tries_max         => 1,
        estimated         => 3600,
        error_description => sub {
            my ($error) = @_;

            return to_json($error);
        },
        milestone_number => 1,
        has_interface    => TRUE,
    },
    {
        method_name       => 'turn_on_unmoderate_dsp',
        method_type       => 8,
        method_name_view  => d_gettext('Turn on unmoderated DSP'),
        view_right        => 'queue_turn_on_unmoderate_dsp',
        add_right         => 'queue_add_turn_on_unmoderate_dsp',
        tries_max         => 1,
        estimated         => 3600,
        error_description => sub {
            my ($error) = @_;

            return to_json($error);
        },
        milestone_number => 1,
        has_interface    => TRUE,
    },
    {
        method_name       => 'on_revoke_user_roles',
        method_type       => 9,
        method_name_view  => d_gettext('Revoke of the user roles'),
        view_right        => 'queue_revoke_user_roles',
        add_right         => 'queue_add_queue_revoke_user_roles',
        tries_max         => 3,
        estimated         => 3600,
        error_description => sub {
            my ($error) = @_;

            return to_json($error);
        },
        milestone_number             => 1,
        allowed_concurrent_execution => 1,
        need_notify                  => 1,
        get_equal_in_queue           => sub {
            my ($self, $params) = @_;
            my $tasks = $self->get_all(fields => [qw(params_ref can_be_run)], filter => ['AND', [{method_type => 9}]]);
            my @task_equal = grep {
                     $_->{can_be_run}
                  && ($params->{user}{id} eq $_->{params_ref}{user}{id})
                  && (0 == scalar @{arrays_difference($_->{params_ref}{roles_id}, $params->{roles_id})})
            } @$tasks;
            return @task_equal ? $task_equal[0] : undef;
        },
    },
    {
        method_name      => 'convert_cpm_in_user_blocks',
        method_type      => 10,
        method_name_view => d_gettext('Convert CPM values in user blocks'),
        view_right       => 'queue_convert_cpm_in_user_blocks',
        add_right        => 'queue_add_convert_cpm_in_user_blocks',
        tries_max        => 3,
        estimated        => sub {
            3600;
        },
        allowed_concurrent_execution => 1,
        error_description            => sub {
            my ($error) = @_;

            return gettext('Failed on: %s', join(', ', @$error));
        },
        milestone_number => sub {
            my ($opts) = @_;

            return 1;
        },
        need_notify => 0,
    },
    {
        method_name           => 'do_action',
        method_type           => 11,
        method_name_view      => d_gettext("Do action"),
        view_right            => 'queue_do_action',
        add_right             => 'queue_add_do_action',
        canonical_params_func => sub {
            my ($params) = @_;

            return $params unless defined($params->{'modelIds'});

            my $modelIds = $params->{'modelIds'};
            $params->{'modelIds'} = [];

            foreach my $id (@$modelIds) {
                if ($id !~ /^\d+$/) {
                    my (undef, $page_id, $block_id) = split_block_public_id($id);
                    $id = make_unique_id($params->{'modelName'}, $page_id, $block_id);
                }

                push(@{$params->{'modelIds'}}, $id);
            }

            return $params;
        },
        try_after => 60,
        tries_max => 3,
        estimated => sub {
            return 22 unless exists($_[0]->{'modelIds'});

            return int(2 + @{$_[0]->{'modelIds'}} * 0.02);
        },

        allowed_concurrent_execution => 0,
        error_description            => sub {
            my ($error) = @_;

            return gettext('Failed on: %s', join(', ', @$error));
        },
        milestone_number => sub {
            my ($opts) = @_;

            return 1 unless exists($opts->{'modelIds'});

            return scalar @{$opts->{'modelIds'}};
        },
        need_notify   => 1,
        has_interface => TRUE,
    },
);

my %QUEUE;
my %QUEUE_BY_TYPE;
my %QUEUE_TYPE_TO_NAME;

for my $task_meta (@QUEUE_ARRAY) {
    $QUEUE{$task_meta->{method_name}}              = $task_meta;
    $QUEUE_BY_TYPE{$task_meta->{method_type}}      = $task_meta;
    $QUEUE_TYPE_TO_NAME{$task_meta->{method_type}} = $task_meta->{method_name};
}

sub accessor      {'queue'}
sub db_table_name {'queue'}

sub get_product_name {gettext('queue')}

__PACKAGE__->register_rights(
    [
        {
            name        => 'queue',
            description => d_gettext('Rights to use queue'),
            rights      => {
                queue_view     => d_gettext('Right to view queue in menu'),
                queue_view_all => d_gettext('Right to view all queue'),
                map {
                    (
                        "queue_add_$_" => d_gettext('Right to add queue with type "%s"',  $_),
                        "queue_$_"     => d_gettext('Right to view queue with type "%s"', $_)
                    )
                  } qw(
                  db_dump
                  mark_read_fns_notification
                  send_email_to_processing
                  statistics_intake
                  update_in_bk
                  update_mobile_app_status
                  queue_revoke_user_roles
                  turn_on_unmoderate_dsp
                  do_action
                  )
            }
        }
    ]
);

__PACKAGE__->model_accessors(
    partner_db    => 'Application::Model::PartnerDB',
    queue_methods => 'Application::Model::Queue::Methods',
    users         => 'Application::Model::Users',
    statistics    => 'Application::Model::Statistics',
);

__PACKAGE__->multistates_graph(
    empty_name  => 'New',
    multistates => [
        [working               => d_gettext('Working')],
        [canceled              => d_gettext('Canceled')],
        [finished_with_success => d_gettext('Successfully finished')],
        [finished_with_error   => d_gettext('Finished with error')],
        [need_restart          => d_gettext('Need restart')],
    ],
    actions => {
        start               => d_gettext('Start'),
        pass_milestone      => d_gettext('Pass milestone'),
        cancel              => d_gettext('Cancel'),
        finish_with_success => d_gettext('Finish with success'),
        finish_with_error   => d_gettext('Finish with error'),
        restart             => d_gettext('Restart'),
        need_restart        => d_gettext('Need restart'),
    },
    multistate_actions => [
        {
            action      => 'start',
            from        => '__EMPTY__',
            set_flags   => ['working'],
            reset_flags => ['finished_with_error'],
        },
        {
            action => 'pass_milestone',
            from   => 'working',
        },
        {
            action    => 'cancel',
            from      => '__EMPTY__',
            set_flags => ['canceled'],
        },
        {
            action      => 'finish_with_success',
            from        => 'working',
            set_flags   => ['finished_with_success'],
            reset_flags => ['working'],
        },
        {
            action      => 'finish_with_error',
            from        => 'working',
            set_flags   => ['finished_with_error'],
            reset_flags => ['working'],
        },
        {
            action      => 'restart',
            from        => 'finished_with_error or need_restart',
            reset_flags => ['finished_with_error', 'need_restart'],
        },
        {
            action      => 'need_restart',
            from        => 'working',
            set_flags   => ['need_restart'],
            reset_flags => ['working'],
        },
    ],
);

__PACKAGE__->model_fields(
    id        => {db => 'u', pk => 1, default => TRUE},
    public_id => {
        db      => 'u',
        db_expr => 'id',
        api     => TRUE,
        type    => 'string',
    },
    group_id => {db => 'u'},
    add_dt   => {
        db   => 'u',
        api  => TRUE,
        type => 'string',
    },
    time_eta   => {db => 'u'},
    multistate => {
        db   => 'u',
        api  => TRUE,
        type => 'number',
    },
    params => {
        db   => 'u',
        api  => TRUE,
        type => 'string',
    },
    user_id => {
        db   => 'u',
        api  => TRUE,
        type => 'number',
    },
    error_data => {
        db   => 'u',
        api  => TRUE,
        type => 'string',
    },
    start_dt => {db => 'u'},
    end_dt   => {db => 'u'},
    log      => {
        db   => 'u',
        api  => TRUE,
        type => 'string',
    },
    result => {
        db   => 'u',
        api  => TRUE,
        type => 'string',
    },
    tries         => {db => 'u'},
    grabbed_by    => {db => 'u'},
    grabbed_at    => {db => 'u'},
    grabbed_until => {db => 'u'},
    data_result   => {
        depends_on => ['result'],
        get        => sub {
            return defined($_[1]->{'result'}) ? from_json($_[1]->{'result'}) : [];
        },
        api  => TRUE,
        type => 'complex',
    },
    params_ref => {
        depends_on => ['params'],
        get        => sub {
            from_json($_[1]->{'params'});
        },
        api  => TRUE,
        type => 'complex',
    },
    method_type => {
        db   => 'u',
        api  => TRUE,
        type => 'string',
    },
    method_name => {
        depends_on => ['method_type'],
        get        => sub {
            $_[0]->{'__METHOD_NAMES__'}{$_[1]->{'method_type'}}{'name'};
        },
        api  => TRUE,
        type => 'string',
    },
    method_name_view => {
        depends_on => ['method_type'],
        get        => sub {
            $_[0]->{'__METHOD_NAMES__'}{$_[1]->{'method_type'}}{'method_name_view'};
        },
        api  => TRUE,
        type => 'string',
    },
    time_eta_ref => {
        depends_on => ['time_eta'],
        get        => sub {
            if (Time::ETA->can_spawn($_[1]->{'time_eta'})) {
                my $eta = Time::ETA->spawn($_[1]->{'time_eta'});
                return $eta;
            } else {
                return undef;
            }
        },
    },
    elapsed_seconds => {
        depends_on => ['time_eta_ref'],
        get        => sub {
            my $eta = $_[1]->{'time_eta_ref'};
            if (defined($eta)) {
                return $eta->get_elapsed_seconds();
            } else {
                return undef;
            }
        },
    },
    remaining_time => {
        depends_on => [qw(time_eta_ref multistate)],
        get        => sub {
            my $eta = $_[1]->{'time_eta_ref'};
            if ($_[1]->{'multistate'} == 1 && defined($eta) && $eta->can_calculate_eta()) {
                return $eta->get_remaining_time();
            } else {
                return undef;
            }
        },
        api  => TRUE,
        type => 'string'
    },
    completed_percent => {
        depends_on => ['time_eta_ref'],
        get        => sub {
            my $eta = $_[1]->{'time_eta_ref'};
            if (defined($eta)) {
                return int($eta->get_completed_percent());
            } else {
                return undef;
            }
        },
        api  => TRUE,
        type => 'number',
    },
    multistate_name => {
        depends_on => ['multistate'],
        get        => sub {
            $_[0]->model->get_multistate_name($_[1]->{'multistate'});
        },
        api  => TRUE,
        type => 'string',
    },
    error_data_ref => {
        depends_on => ['error_data'],
        get        => sub {
            defined($_[1]->{'error_data'}) ? from_json($_[1]->{'error_data'}) : undef;
        },
        api  => TRUE,
        type => 'complex',
    },
    error_description => {
        depends_on => [qw(error_data_ref method_name)],
        get        => sub {
            my ($self, $obj) = @_;

            my $error                    = $obj->{error_data_ref};
            my $error_description_getter = $QUEUE{$obj->{method_name}}{error_description};

            return (
                $error
                ? ($error_description_getter ? $error_description_getter->($error) : $error)
                : ''
            );
        },
        api  => TRUE,
        type => 'string',
    },
    milestone_number => {
        depends_on => [qw(params_ref method_name)],
        get        => sub {
            my ($self, $obj) = @_;

            my $method_name      = $obj->{method_name};
            my $milestone_number = $QUEUE{$method_name}{milestone_number};

            throw gettext("No way to define milestone number for method '%s'.", $method_name)
              unless $milestone_number;

            return (ref($milestone_number) eq 'CODE' ? $milestone_number->($obj->{params_ref}) : $milestone_number);
        },
        api  => TRUE,
        type => 'number',
    },
    need_notify => {
        depends_on => ['method_type'],
        get        => sub {
            my ($self, $obj) = @_;

            my $method_name = $obj->{method_name};
            return $QUEUE{$method_name}{need_notify};
        },
    },
    can_be_run => {
        depends_on => ['multistate'],
        get        => sub {
            in_array(
                $_[1]->{'multistate'},
                $_[0]->model->get_multistates_by_filter(
                    'not (finished_with_success or finished_with_error or need_restart or canceled)')
            );
        },
        api  => TRUE,
        type => 'boolean',
    },
    login => {
        depends_on => ['user_id'],
        get        => sub {
            if (not(defined($_[1]->{'user_id'}))) {
                throw gettext('No information about user');
            }
            $_[0]->{'users'}->{$_[1]->{'user_id'}}->{'login'};
        },
        api  => TRUE,
        type => 'string',
    },
    available_fields => {
        depends_on => [qw(id)],
        get        => sub {
            return $_[0]->model->get_available_fields($_[1]);
        },
        type     => 'complex',
        fix_type => \&fix_type_for_complex,
        api      => TRUE,
    },
    fields_depends => {
        depends_on => [qw(id)],
        get        => sub {
            return {};
        },
        type => 'complex',
        api  => TRUE,
    },
    actions => {
        depends_on => [qw(id multistate)],
        get        => sub {
            $_[0]->model->get_actions($_[1]);
        },
        type => 'complex',
        api  => TRUE
    },
    additional_info => {
        depends_on => [
            qw(
              id
              method_name
              params_ref
              tries
              grabbed_at
              grabbed_by
              grabbed_until
              group_id
              )
        ],
        get => sub {
            $_[0]->model->get_additional_info($_[1]);
        },
        api      => TRUE,
        type     => 'array',
        sub_type => 'complex'
    }
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id          => {type => 'number',     label => d_gettext('UID')},
        multistate  => {type => 'multistate', label => d_gettext('Status')},
        method_type => {
            type   => 'dictionary',
            label  => d_gettext('Method type'),
            values => sub {
                return [
                    map {+{id => $_->{'method_type'}, label => $_->{'method_name_view'},}}
                    sort {$a->{'method_name_view'} cmp $b->{'method_name_view'}} @{$_[0]->get_available_queue()}
                ];
            },
        },
        user => {
            type           => 'subfilter',
            model_accessor => 'users',
            field          => 'user_id',
            fk_field       => 'id',
            label          => d_gettext('User'),
        },
        login => {
            type => 'alias',
            path => [qw(user login)]
        }
    },
);

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        {name => 'id',          label => gettext('ID')},
        {name => 'multistate',  label => gettext('Status')},
        {name => 'method_type', label => gettext('Method type')},
        {name => 'login',       label => gettext('Login')}
    ];
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('login')) {
        my $list = $self->users->get_all(
            fields => [qw(id login)],
            filter => {id => array_uniq([map {$_->{'user_id'}} @$result])},
        );
        $fields->{'users'}->{$_->{id}} = $_ foreach (@$list);
    }

    if ($fields->need('method_name') || $fields->need('method_name_view')) {
        $fields->{'__METHOD_NAMES__'} = {
            map {$QUEUE{$_}->{'method_type'} => {name => $_, method_name_view => $QUEUE{$_}->{'method_name_view'}()}}
              keys(%QUEUE)
        };
    }
}

sub get_available_fields {
    my ($self, $obj) = @_;

    return {map {$_ => TRUE} keys(%{$self->get_model_fields()})};
}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Validation::BadArguments gettext("Expected 'method_name'") unless defined($opts{'method_name'});
    throw Exception::Validation::BadArguments gettext('Unknown method "%s"', $opts{'method_name'})
      unless $QUEUE{$opts{'method_name'}};

    throw Exception::Denied 'Has not right ' . $QUEUE{$opts{'method_name'}}->{'add_right'}
      unless $self->check_rights($QUEUE{$opts{'method_name'}}->{'add_right'});

    my $canonical_params_func =
      exists($QUEUE{$opts{'method_name'}}->{'canonical_params_func'})
      ? $QUEUE{$opts{'method_name'}}->{'canonical_params_func'}
      : sub {return $_[0]};

    my $serialized_params = to_json($canonical_params_func->($opts{'params'}), canonical => TRUE);

    throw Exception::Validation::BadArguments gettext('Params is too long') if length($serialized_params) > 65535;

    my $cur_user_uid = $self->get_option('cur_user', {})->{'id'};

    my $equal_task;
    if (my $sub_get_equal_in_queue = $QUEUE{$opts{'method_name'}}->{'get_equal_in_queue'}) {
        $equal_task = $sub_get_equal_in_queue->($self, $opts{'params'});
    }

    return $equal_task ? $equal_task : $self->partner_db->queue->add(
        {
            add_dt      => curdate(oformat => 'db_time'),
            method_type => $QUEUE{$opts{'method_name'}}->{'method_type'},
            params      => $serialized_params,
            user_id     => $cur_user_uid,
            multistate  => 0,
            tries       => 0,
            group_id    => $opts{group_id},
        }
    );
}

sub on_action_finish_with_error {
    my ($self, $task, %opts) = @_;

    $task = $self->_get_object_fields($task, [qw(id time_eta_ref)]);

    my $eta = $task->{'time_eta_ref'};
    if ($eta->is_paused()) {
        INFOF 'Task with id %s is already paused', $task->{'id'};
    } else {
        $eta->pause();
    }
    $self->partner_db->queue->edit(
        $task->{'id'},
        {
            end_dt     => curdate(oformat => 'db_time'),
            log        => $opts{'exception'}->message(),
            error_data => $opts{'exception'}->can('error_data')
              && $opts{'exception'}->error_data() ? to_json($opts{'exception'}->error_data()) : undef,
            time_eta => $task->{'time_eta_ref'}->serialize(),
            $self->get_grabbed_args(type => 'finish',),
        }
    );
}

sub on_action_finish_with_success {
    my ($self, $task, %opts) = @_;

    my $data = {
        end_dt => curdate(oformat => 'db_time'),
        $self->get_grabbed_args(type => 'finish',),
    };

    if (defined($opts{'result'})) {
        my $result = $self->get($task, fields => [qw(data_result)])->{'data_result'};

        push(@$result, $opts{'result'});

        $data->{'result'} = to_json($result, canonical => TRUE);
    }

    $self->partner_db->queue->edit($task->{'id'}, $data);
}

sub on_action_need_restart {
    my ($self, $task, %opts) = @_;

    $task = $self->_get_object_fields($task, [qw(id method_name milestone_number time_eta_ref)]);

    my $eta = $task->{'time_eta_ref'} // Time::ETA->new(milestones => ($task->{milestone_number}));
    if ($eta->is_paused()) {
        INFOF 'Task with id %s is already paused', $task->{'id'};
    } else {
        $eta->pause();
    }

    $self->partner_db->queue->edit(
        $task->{'id'},
        {
            log        => $opts{'exception'}->message(),
            error_data => $opts{'exception'}->can('error_data')
              && $opts{'exception'}->error_data() ? to_json($opts{'exception'}->error_data()) : undef,
            time_eta => $eta->serialize(),
            $self->get_grabbed_args(
                type        => 'need_restart',
                method_name => $task->{method_name},
            ),
        }
    );
}

sub on_action_pass_milestone {
    my ($self, $task) = @_;

    $task = $self->get($task->{'id'}, fields => [qw(id time_eta_ref)]);

    $task->{'time_eta_ref'}->pass_milestone();
    $self->partner_db->queue->edit($task->{'id'}, {time_eta => $task->{'time_eta_ref'}->serialize()});
}

sub on_action_restart {
    my ($self, $task) = @_;

    $task = $self->_get_object_fields($task, [qw(id method_name)]);

    $self->partner_db->queue->edit(
        $task->{'id'},
        {
            log => undef,
            $self->get_grabbed_args(type => 'finish'),
        }
    );
}

sub on_action_start {
    my ($self, $task, %opts) = @_;

    $task = $self->_get_object_fields($task, [qw(id method_name milestone_number params_ref time_eta_ref tries)]);

    my $eta = $task->{'time_eta_ref'}
      // Time::ETA->new(milestones => ($task->{milestone_number}));    # milestones can't be zero
    $eta->resume() if $eta->is_paused();

    $self->partner_db->queue->edit(
        $task->{id},
        {
            tries      => $opts{tries},
            start_dt   => curdate(oformat => 'db_time'),
            error_data => undef,
            end_dt     => undef,
            log        => undef,
            time_eta   => $eta->serialize(),
            $self->get_grabbed_args(
                type        => 'start',
                method_name => $task->{method_name},
                params      => $task->{params_ref},
            ),
        },
    );
}

sub query {
    my ($self, %opts) = @_;

    my $filter = $self->partner_db->filter($opts{'filter'});
    $filter->and({method_type => $self->get_type()});
    $filter->and({user_id => $self->get_option('cur_user', {})->{id}}) unless $self->check_short_rights('view_all');

    my $query = $self->partner_db->query->select(
        table  => $self->partner_db_table(),
        fields => $opts{'fields'}->get_db_fields('u'),
        filter => $filter
    );

    return $query;
}

sub api_available_fields {
    my ($self) = @_;

    my @fields = $self->SUPER::api_available_fields();

    return sort @fields;
}

sub api_available_actions {qw(cancel restart)}

sub check_stuck_tasks {
    my ($self) = @_;

    my $need_notify_task_types = [
        map  {$_->{method_type}}
        grep {$_->{need_notify}} values(%QUEUE)
    ];

    my $filter = $self->partner_db->filter(
        [
            AND => [
                ['method_type', 'IN', \$need_notify_task_types],
                ['multistate',  'IN', \$self->get_multistates_by_filter('finished_with_error')],
            ],
        ]
    );

    my $count = $self->partner_db->queue->get_all(
        fields => {count => {COUNT => ['id']},},
        filter => $filter,
    )->[0]{'count'};

    send_to_graphite(
        interval => 'one_min',
        path     => 'Queue.Tasks.stuck_count',
        value    => $count // 0,
        solomon  => {
            metric => 'stuck_count',
            sensor => 'Queue.Tasks',
        }
    );

    if (defined($count) && $count > 0) {
        throw Exception::Queue('There are stucked tasks in queue');
    }
}

sub clean_up_finished_tasks {
    my ($self) = @_;

    my $result = $self->partner_db->queue->delete(
        $self->partner_db->queue->filter(
            [AND => [['end_dt', '<=', \date_sub(curdate(), day => 14, oformat => "db_time")],]]
        ),
    );

    return $result;
}

sub get_additional_info {
    my ($self, $task) = @_;

    my $addition_info = [];

    if ($task->{'method_name'} eq 'statistics_intake') {
        push(@$addition_info, {tag => 'b', content => gettext('Products')}, {tag => 'br'});

        my @products = ();
        foreach my $product_name (@{$task->{'params_ref'}{'products'}}) {

            try {
                my $title = $self->statistics->get_level_full_title($product_name);
                push @products, $title;
            }
            catch {
                # Уровни статистики поменялись после решения задачи
                # PI-694 - Статистика дистрибуции в одну строку.
                # В codebase уже нет данных о старых уровнях, поэтому
                # показываем не title (потому что его нет), а текстовое имя
                # продукта
                push @products, $product_name;
            }

        }
        push(@$addition_info, {tag => 'pre', content => join('; ', @products)}, {tag => 'br'});

        push(@$addition_info, {tag => 'b', content => gettext('Period')}, {tag => 'br'});
        push(
            @$addition_info,
            {
                tag     => 'pre',
                content => format_date($task->{'params_ref'}{'start'}, gettext('%d.%m.%Y'), iformat => 'db_time')
                  . ' - '
                  . format_date($task->{'params_ref'}{'end'}, gettext('%d.%m.%Y'), iformat => 'db_time')
            }
        );

        if (defined($task->{'params_ref'}->{'logins'}) && ref($task->{'params_ref'}->{'logins'}) eq 'ARRAY') {
            push(@$addition_info, {tag => 'br'}, {tag => 'b', content => gettext('Logins')}, {tag => 'br'});
            push(@$addition_info, {tag => 'pre', content => join(', ', @{$task->{'params_ref'}->{'logins'}})});
        } elsif (!defined($task->{'params_ref'}->{'logins'})) {
            # there can be no logins in update statistics task
        } else {
            throw gettext("Field 'login' is not in the expected format.");
        }
    } elsif ($task->{'method_name'} eq 'update_in_bk') {
        push(@$addition_info, {tag => 'b', content => gettext('Campaigns')}, {tag => 'br'});
        my @campaigns = ();
        foreach (@{$task->{'params_ref'}{'page_ids'}}) {
            push(@campaigns, $_);
            if (@campaigns == 6) {
                push(@$addition_info, {tag => 'pre', content => join(', ', @campaigns)}, {tag => 'br'});
                @campaigns = ();
            }
        }
        push(@$addition_info, {tag => 'pre', content => join(', ', @campaigns)}) if @campaigns;
    } elsif ($task->{'method_name'} eq 'do_action') {
        push(@$addition_info, {tag => 'pre', content => $task->{'params'}});
    } elsif (
        in_array(
            $task->{'method_name'},
            [
                qw(
                  send_email_to_processing
                  update_mobile_app_status
                  change_blocked_brands
                  turn_on_unmoderate_dsp
                  )
            ]
        )
      )
    {
        my $data = to_json(
            {
                hash_transform(
                    $task,
                    [
                        qw(
                          tries
                          grabbed_at
                          grabbed_by
                          grabbed_until
                          group_id
                          )
                    ],
                    {params_ref => 'task',}
                )
            },
            pretty => TRUE,
        );
        push(@$addition_info, {tag => 'pre', content => $data});
    }

    return $addition_info;
}

sub get_available_queue {
    my ($self) = @_;

    return [
        map {
            {
                method_type      => $QUEUE{$_}->{'method_type'},
                method_name      => $_,
                method_name_view => $QUEUE{$_}->{'method_name_view'}(),
                has_interface    => $QUEUE{$_}->{'has_interface'},
            }
          }
          sort grep {
            $self->check_rights($QUEUE{$_}->{'add_right'})
          } keys(%QUEUE)
    ];
}

sub get_fields_defaults {
    my ($self, $opts) = @_;

    my $fields_depends = $self->get_fields_depends();

    my %need_fields = map {$_ => TRUE} @{$opts->{fields}};
    foreach (map {@{$fields_depends->{'depends'}{$_} // []}} @{$opts->{changed_fields}}) {
        $need_fields{$_} = TRUE;
    }

    my $result = {};
    if ($need_fields{'available_queues'}) {
        $result->{'available_queues'} = [grep {$_->{'has_interface'}} @{$self->get_available_queue()}];
    }

    return $result;
}

sub get_fields_depends {{}}

sub get_grabbed_args {
    my ($self, %opts) = @_;

    my ($method_name, $params, $type) = @opts{
        qw(
          method_name
          params
          type
          )
      };

    if ($type eq 'start') {
        my $estimated =
          ref($QUEUE{$method_name}{estimated}) eq 'CODE'
          ? $QUEUE{$method_name}{estimated}->($params)
          : $QUEUE{$method_name}{estimated};
        return (
            grabbed_by    => $self->get_worker_id(),
            grabbed_at    => curdate(oformat => 'db_time'),
            grabbed_until => date_add(curdate(), second => $estimated, oformat => 'db_time'),
        );
    } elsif ($type eq 'need_restart') {
        my @wait_before_restart =
          defined($QUEUE{$method_name}{try_after})
          ? (grabbed_until => date_add(curdate(), second => $QUEUE{$method_name}{try_after}, oformat => 'db_time'))
          : (grabbed_until => undef);
        return (
            grabbed_by => undef,
            grabbed_at => undef,
            @wait_before_restart,
        );
    } elsif ($type eq 'finish') {
        return (
            grabbed_by    => undef,
            grabbed_at    => undef,
            grabbed_until => undef,
        );
    }
}

sub get_type {
    my ($self) = @_;

    return [
        sort {$a <=> $b}
        map {$QUEUE{$_}->{'method_type'}} grep {$self->check_rights($QUEUE{$_}->{'view_right'})} keys(%QUEUE)
    ];
}

sub get_worker_id {
    my ($self) = @_;
    return sprintf("%s:%d", $self->get_option('hostname', 'localhost'), $$);
}

sub get_disabled_methods {
    my ($self) = @_;
    # Храним именно факт выключения, чтобы по дефолту (нет записи в kv_store)
    # метод очереди был _включен_
    return from_json($self->app->kv_store->get($QUEUE_IS_METHOD_DISABLED_KEY) // '{}');
}

sub grab_next_task {
    my ($self) = @_;

    my $disabled_methods = $self->get_disabled_methods();
    my @queues =
      grep {$self->queue_methods->can($_->{'method_name'}) && !$disabled_methods->{$_->{'method_name'}}} values(%QUEUE);

    my $allowed_concurrent = [
        map  {$_->{method_type}}
        grep {$_->{allowed_concurrent_execution}} @queues
    ];

    my $disallowed_concurrent = [
        map  {$_->{method_type}}
        grep {!$_->{allowed_concurrent_execution}} @queues
    ];

    my $task;
    $self->partner_db->transaction(
        sub {
            my $filter = $self->partner_db->filter(
                [
                    'id', 'IN',
                    # Этот оберточный запрос нужен, чтобы оставить в результате только одно поле id. Иначе не работает IN.
                    $self->partner_db->query->select(
                        fields => [qw(id)],
                        table  => $self->partner_db->query->select(
                            alias  => 'q1',
                            table  => $self->partner_db->queue,
                            fields => [qw(id)],
                            filter => [
                                AND => [
                                    ['grabbed_by', 'IS', \undef],
                                    # Этим фильтром мы отсеиваем allowed_concurrent = 0 задачки, которые уже начаты
                                    # Мы не можем положить этот фильтр в подзапрос,
                                    # потому что тогда неправильно сработает группировка
                                    ['multistate', 'IN', \$self->get_multistates_by_filter('__EMPTY__')],
                                ]
                            ],
                          )->join(
                            alias => 'q2',
                            table => $self->partner_db->query->select(
                                table  => $self->partner_db->queue,
                                fields => {
                                    method_type => '',
                                    group_id    => '',
                                    # Ищем задачу, вставшую в очередь первой для каждой группы method_type-group_id
                                    min_id => {MIN => ['id']}
                                },
                                filter => $self->partner_db->queue->filter(
                                    [
                                        AND => [
                                            [
                                                OR => [
                                                    [
                                                        AND => [
                                                            # Для конкурентных методов просто отсеиваем все начатые задачи
                                                            # Нас не интересует, есть ли другие работающие задачи того же типа
                                                            # Их можно выполнять параллельно
                                                            ['method_type', 'IN', \$allowed_concurrent],
                                                            ['grabbed_by',  'IS', \undef],
                                                            [
                                                                'multistate', 'IN',
                                                                \$self->get_multistates_by_filter('__EMPTY__')
                                                            ],
                                                        ]
                                                    ],
                                                    [
                                                        AND => [
                                                            # Для неконкурентных оставляем задачи, которые еще могут дорботать до конца
                                                            # Среди них: неначатые, работающие, ожидающие рестарта, завершенные с ошибкой
                                                            ['method_type', 'IN', \$disallowed_concurrent],
                                                            [
                                                                'multistate',
                                                                'IN',
                                                                \$self->get_multistates_by_filter(
'__EMPTY__ or working or finished_with_error or need_restart'
                                                                )
                                                            ],
                                                        ]
                                                    ],
                                                ]
                                            ],
                                        ]
                                    ]
                                ),
                              )->group_by(qw(method_type group_id)),
                            join_on => [{id => 'q1'} => '=' => {min_id => 'q2'}],
                          )
                    )
                ]
            );

            $filter->and(['multistate', 'IN', \$self->get_multistates_by_filter('__EMPTY__')]);
            my $tasks = $self->get_all(
                fields     => [qw(id user_id method_name params_ref error_data_ref tries)],
                filter     => $filter,
                limit      => 1,
                for_update => 1,
            );

            $task = $tasks->[0];

            if ($task) {
                $task->{tries} += 1;
                $self->do_action($task->{id}, 'start', tries => $task->{tries});
            }
        }
    );

    return $task;
}

sub restart_lost_tasks {
    my ($self) = @_;

    my $filter = $self->partner_db->filter(
        [
            'id', 'IN',
            $self->partner_db->query->select(
                table  => $self->partner_db->queue,
                fields => [qw(id)],
                filter => [
                    AND => [
                        [
                            OR => [
                                ['grabbed_until', '<=', \curdate(oformat => 'db_time')],
                                ['grabbed_until', 'IS', \undef],
                            ]
                        ],
                        ['multistate', 'IN', \$self->get_multistates_by_filter('working or need_restart')],
                    ],
                ],
            )
        ]
    );

    my $lost_tasks = $self->get_all(
        fields => [qw(id method_name time_eta_ref tries)],
        filter => $filter,
    );

    for my $task (@$lost_tasks) {
        if ($task->{tries} >= $QUEUE{$task->{method_name}}{tries_max}) {
            $self->do_action($task->{'id'}, 'finish_with_error', exception => Exception->new('Max tries exceeded'));
        } else {
            $self->maybe_do_action($task, 'need_restart', exception => Exception->new('Need restart'));
            $self->do_action($task, 'restart');
        }
    }
}

sub start {
    my ($self, $task) = @_;

    my $error_data_ref = $task->{'error_data_ref'};
    $task = $self->_get_object_fields($task, [qw(id user_id method_name params_ref time_eta_ref tries)]);

    my $method = $task->{'method_name'};

    try {
        # То, ради чего все затевалось — запускаем задачу
        my $result = $self->queue_methods->$method(
            hash_transform($task, ['user_id', 'tries'], {id => 'task_id', time_eta_ref => 'eta'}),
            error_data_ref => $error_data_ref,
            %{$task->{'params_ref'}},
        );

        $self->do_action($task->{'id'}, 'finish_with_success', result => $result);
    }
    catch {
        my $exception = shift;

        if ($task->{tries} >= $QUEUE{$task->{method_name}}{tries_max}) {
            $self->do_action($task->{'id'}, 'finish_with_error', exception => $exception);
        } else {
            $self->do_action($task, 'need_restart', exception => $exception);
        }

        ERROR $exception;
    };

    return undef;
}

TRUE;
