package Application::Model::Queue::Methods;

=encoding UTF-8

=head1 Описание

=cut

use qbit;

use base qw(QBit::Application::Model);

use Time::ETA;

use Utils::Logger;

use Exception::Queue;
use Exception::Queue::NeedRestart;
use Exception;
use Exception::Validation::BadArguments;
use Exception::SelfemployedOeBS::Unbound;
use Exception::SelfemployedOeBS::Unregistered;

sub accessor {'queue_methods'}

__PACKAGE__->model_accessors(
    all_pages             => 'Application::Model::AllPages',
    api_adfox_graphql     => 'Application::Model::API::Yandex::AdFoxGraphQL',
    api_selfemployed_oebs => 'QBit::Application::Model::API::Yandex::SelfemployedOeBS',
    db_dumper             => 'Application::Model::DBDumper',
    exception_dumper      => 'Application::Model::ExceptionDumper',
    mail_notification     => 'Application::Model::MailNotification',
    partner_db            => 'Application::Model::PartnerDB',
    queue                 => 'Application::Model::Queue',
    rbac                  => 'Application::Model::RBAC',
    statistics            => 'Application::Model::Statistics',
    users                 => 'Application::Model::Users',
    support               => 'Application::Model::Queue::Support',
    context_on_site_rtb   => 'Application::Model::Product::AN::ContextOnSite::RTB',
    mobile_app_rtb        => 'Application::Model::Product::AN::MobileApp::RTB',
);

my $BLOCK_MODELS_WITH_CPM_CURRENCY = {
    context_on_site_rtb => {model_specific_fields => [qw(campaign_id video_cpm)]},
    mobile_app_rtb      => {model_specific_fields => [qw(campaign_id)]}
};

sub update_in_bk {
    my ($self, %opts) = @_;

    throw Exception::Validation::BadArguments gettext("Expected 'page_ids'") unless defined($opts{'page_ids'});
    throw Exception::Validation::BadArguments gettext("Expected 'task_id'")  unless defined($opts{'task_id'});
    throw Exception::Validation::BadArguments gettext("Expected 'user_id'")  unless defined($opts{'user_id'});

    my $page_ids = defined($opts{'error_data_ref'}) ? $opts{'error_data_ref'} : $opts{'page_ids'};

    my $body        = gettext("Some campaigns have not been updated in BK:\n");
    my @not_updated = ();

    if (@$page_ids) {
        foreach my $page_id (@$page_ids) {
            try {
                my $object = $self->all_pages->get_all(fields => ['id', 'model'], filter => {page_id => $page_id})->[0];

                throw Exception gettext('Page with Page ID %s not found', $page_id) unless defined($object);

                my ($id, $model) = @$object{qw( id model)};

                my $page_id_field = $self->app->$model->get_page_id_field_name();

                $self->app->$model->do_action($id, 'set_need_update');

                $self->app->$model->update_in_bk({$page_id_field => $page_id});
            }
            catch {
                $body .= "$page_id: " . shift->message() . "\n";

                push(@not_updated, $page_id);
            };
            $self->queue->do_action($opts{'task_id'}, 'pass_milestone');
        }
    } else {
        # Corner case - no pages, just 1 milestone set
        # in queue->on_action_start to satisfy Time::ETA
        $self->queue->do_action($opts{'task_id'}, 'pass_milestone');
    }

    if (@not_updated) {
        my @login_roles = keys(%{$self->rbac->get_roles_by_user_id($opts{'user_id'})});
        my @login_rights = map {$_->{right}} @{$self->rbac->get_roles_rights(role_id => \@login_roles)};

        my $to =
          (grep {$_ eq 'yndx_login'} @login_rights)
          ? $opts{'user_id'}
          : 0;

        $self->mail_notification->add(
            type    => 0,
            user_id => $to,
            opts    => {
                (
                    $to
                    ? ()
                    : (
                        check_alive => FALSE,
                        to          => $self->get_option('error_notify')->{'email_to'},
                      )
                ),
                subject => gettext('Task %s on update in bk', $opts{'task_id'}),
                values  => {
                    message_body       => $body,
                    plain_text_wrapper => TRUE,
                },
            },
        );

        throw Exception::Queue($body, \@not_updated);
    }

    return undef;
}

sub db_dump {
    my ($self, %opts) = @_;

    throw Exception::Validation::BadArguments gettext("Expected 'task_id'") unless defined($opts{'task_id'});

    my $dump_path = $self->db_dumper->dump(
        page_ids      => $opts{'page_ids'},
        logins        => $opts{'logins'},
        date          => $opts{'date'} // date_sub(curdate(), week => 2, oformat => 'db'),
        filename_part => $opts{'task_id'},
    );

    return {hostname => $self->get_option('hostname'), file => $dump_path};
}

sub send_email_to_processing {
    my ($self, %opts) = @_;

    my %args = hash_transform(\%opts, [qw(uid tries)]);

    $self->users->send_email_to_processing(user_id => $opts{uid});

    return \%args;
}

sub statistics_intake {
    my ($self, %opts) = @_;

    throw Exception::Validation::BadArguments gettext("Expected 'start'")    unless defined($opts{'start'});
    throw Exception::Validation::BadArguments gettext("Expected 'end'")      unless defined($opts{'end'});
    throw Exception::Validation::BadArguments gettext("Expected 'products'") unless defined($opts{'products'});
    throw Exception::Validation::BadArguments gettext("Expected 'task_id'")  unless defined($opts{'task_id'});
    throw Exception::Validation::BadArguments gettext("Expected 'eta'")      unless defined($opts{'eta'});

    $opts{'start'} = $opts{'error_data_ref'}->{date} if $opts{'error_data_ref'};

    my @dates = dates2array(
        $opts{'start'}, $opts{'end'},
        iformat => 'db',
        oformat => 'db',
    );

    my $eta = $opts{'eta'};

    my $update_not_started_after_resume = defined($opts{'error_data_ref'});

    foreach my $date (@dates) {
        foreach my $level_name (@{$opts{'products'}}) {
            next
              if $update_not_started_after_resume
                  && $level_name ne $opts{error_data_ref}->{level_name};

            $update_not_started_after_resume = FALSE;

            try {
                my $accessor = 'statistics_' . $level_name;

                my $storage = (grep {$_ eq $level_name} ('dsp', 'dsp_response')) ? 'mysql' : 'clickhouse';

                $self->app->$accessor->update_statistics(
                    from    => $date,
                    to      => $date,
                    storage => $storage,
                    ($opts{'logins'} ? (logins => $opts{'logins'}) : ())
                );

                $self->queue->do_action($opts{'task_id'}, 'pass_milestone');
            }
            catch {
                my $exception = shift;

                $self->exception_dumper->dump_as_html_file($exception);
                throw Exception::Queue $exception, {date => $date, level_name => $level_name};
            };
        }
    }

    return undef;
}

sub update_mobile_app_status {
    my ($self, %opts) = @_;

    my %args = hash_transform(\%opts, [qw(app_id active tries)]);

    $self->api_adfox_graphql->update_mobile_app_status(app_id => $opts{app_id}, active => $opts{active});

    return \%args;
}

sub mark_read_fns_notification {
    my ($self, %opts) = @_;

    my %args = hash_transform(\%opts, [qw(inn notification_id tries)]);

    try {
        $self->api_selfemployed_oebs->post_notif_read([{inn => $opts{inn}, messageId => $opts{notification_id}}]);
    }
    catch Exception::SelfemployedOeBS::Unbound with {
        INFOF('TAXPAYER_UNBOUND for INN=[%s]', $opts{inn});
    }
    catch Exception::SelfemployedOeBS::Unregistered with {
        INFOF('TAXPAYER_UNREGISTERED for INN=[%s]', $opts{inn});
    };

    return \%args;
}

sub change_blocked_brands {
    my ($self, %opts) = @_;

    my $result = $self->support->get_object('change_blocked_brands')->apply($self->app, \%opts);

    return $result;
}

sub turn_on_unmoderate_dsp {
    my ($self, %opts) = @_;

    my $result = $self->support->get_object('turn_on_unmoderate_dsp')->apply($self->app, \%opts);

    return $result;
}

sub on_revoke_user_roles {
    my ($self, %opts) = @_;

    return $self->users->_stop_and_delete_user_pages(%opts);
}

TRUE;
