package Application::Model::Queue::Support::Base;

use qbit;
use Exception::Denied;
use Exception::Form;
use Utils::PublicID;
use Utils::Logger qw(INFOF);

use Application::Model::Queue::Support::Target::Logins;
use Application::Model::Queue::Support::Target::Pages;
use Application::Model::Queue::Support::Target::Blocks;

use Application::Model::Queue::Support::Source::Brands;
use Application::Model::Queue::Support::Source::SampleBrands;
use Application::Model::Queue::Support::Source::Sample;
use Application::Model::Queue::Support::Source::UnmoderatedDSP;
use Application::Model::Queue::Support::Source::UnmoderatedRtbAuction;

use Application::Model::Queue::Support::Author::Ticket;

use Application::Model::Queue::Support::Apply::ApplyGM;
use Application::Model::Queue::Support::Apply::ApplyDeleted;
use Application::Model::Queue::Support::Apply::Method;

use Application::Model::Queue::Support::Filter::SpecialBlock;
use Application::Model::Queue::Support::Filter::UnmoderatedRtbAuction;
use Application::Model::Queue::Support::Filter::MobileBlockTypeDSP;

my @parts  = qw(target apply source author);
my @single = qw(target source);
my @any    = qw(apply);
my @all    = qw(author);

sub method {die "method 'method' need override"}

sub get_structure {
    my ($self) = @_;

    unless ($self->{structure}) {
        my $data = $self->get_structure_data;
        push @{$data->{author}}, 'Application::Model::Queue::Support::Author::Ticket';
        push @{$data->{apply}},  'Application::Model::Queue::Support::Apply::ApplyGM';
        push @{$data->{apply}},  'Application::Model::Queue::Support::Apply::ApplyDeleted';
        $self->{structure} = $data;
    }

    return $self->{structure};
}

sub get_owner_data {
    my ($self, $data) = @_;

    return $data->{Application::Model::Queue::Support::Author::Ticket->data_name};
}

sub get_data_fields {
    my ($self) = @_;

    my $structure = $self->get_structure();
    my @fields;
    for my $type (@parts) {
        for my $e (@{$structure->{$type}}) {
            push @fields, $e->data_name;
        }
    }
    return \@fields;
}

sub get_form_fields {
    my ($self) = @_;

    my $structure = $self->get_structure();
    my @data;
    for my $type (@parts) {
        for my $e (@{$structure->{$type}}) {
            push @data, $e->field;
        }
    }
    push @data, {label => gettext('Send'), type => 'button', subtype => 'submit'};
    return \@data;
}

sub get_form_data {
    my ($self, $form) = @_;

    my $structure = $self->get_structure();
    my %params;
    for my $type (@parts) {
        for my $e (@{$structure->{$type}}) {
            if (my $form_name = $e->form_name) {
                $params{$form_name} = $form->get_value($form_name);
            }
        }
    }

    return \%params;
}

sub check_form_data {
    my ($self, $app, $data) = @_;

    $self->set_filters($app, $data);

    my $structure = $self->get_structure();
    for my $type (@single) {
        unless (1 == (my @field = grep {$_->form_name && $data->{$_->form_name}} @{$structure->{$type}})) {
            throw Exception::Form gettext('You must enter one of %s: %s',
                $type, join(', ', map {$_->field->{label}} grep {$_->form_name} @{$structure->{$type}}));
        } else {
            my ($error, $count) = $field[0]->check($app, $data);
            if ($error) {
                throw Exception::Form gettext('Has invalid entry in %s: %s', $field[0]->field->{label}, $error);
            }
            unless ($count) {
                throw Exception::Form gettext('No valid %s', $field[0]->field->{label});
            }
        }
    }

    for my $type (@all) {
        for my $e (@{$structure->{$type}}) {
            my ($error, $count) = $e->check($app, $data);
            if ($error) {
                throw Exception::Form gettext('Has invalid entry in %s: %s', $e->field->{label}, $error);
            }
            unless ($count) {
                throw Exception::Form gettext('No valid %s', $e->field->{label});
            }
        }
    }

    for my $type (@any) {
        for my $e (@{$structure->{$type}}) {
            my ($error) = $e->check($app, $data);
            if ($error) {
                throw Exception::Form gettext('Has invalid entry in %s: %s', $e->field->{label}, $error);
            }
        }
    }

    $self->clear_filters($app, $data);
}

sub make_process {
    my ($self, $app) = @_;

    throw Exception::Denied
      unless $app->check_rights($self->right_add);

    return (
        title  => $self->title,
        fields => $self->get_form_fields,
        save   => sub {
            my ($form) = @_;

            my $params = $self->get_form_data($form);

            try {
                $self->check_input_params($app, $params);
                $app->queue->add(
                    method_name => $self->method,
                    params      => $params,
                );
            }
            catch {
                throw Exception::Form $_[0]->message;
            };

            $form->{'complete_message'} = $self->get_complete_message;
        }
    );
}

sub check_input_params {
    my ($self, $app, $params) = @_;

    my $orig = clone($params);
    $self->check_form_data($app, $params);

    try {
        $self->post_ticket_message(
            app        => $app,
            data       => $params,
            input      => $orig,
            message    => $self->get_start_message,
            attachment => {full_parsed_data => $params},
        );
    }
    catch {
        throw Exception::Form $_[0]->message;
    };
}

sub get_blocks {
    my ($self, $app, $data) = @_;

    my $structure = $self->get_structure();

    for my $e (@{$structure->{target}}) {
        if ($data->{$e->data_name}) {
            return $e->get_blocks($app, $data);
        }
    }
    return [];
}

sub get_pages {
    my ($self, $app, $data) = @_;

    my $structure = $self->get_structure();

    for my $e (@{$structure->{target}}) {
        if ($data->{$e->data_name}) {
            return $e->get_pages($app, $data);
        }
    }
    return [];
}

sub apply_to_list {
    my ($self, $app, $data, $list) = @_;

    my $structure = $self->get_structure();

    for my $e (@{$structure->{source}}) {
        if ($e->can_apply($data)) {
            return $e->apply_to_list($app, $data, $list);
        }
    }
}

sub set_filters {
    my ($self, $app, $data) = @_;

    my $structure = $self->get_structure();
    my %filter;
    for my $e (@{$structure->{filter}}) {
        $e->get_filter($app, $data, \%filter);
    }

    my @filter_block;
    if ($filter{blocks}) {
        push @filter_block, @{$filter{blocks}};
    }
    if (my $skip_model = $filter{skip_block_model}) {
        push @filter_block, [model => 'NOT IN' => \$skip_model];
    }

    if (my $check_fields = $filter{check_fields}) {
        my $models = $app->product_manager->get_block_model_names;
        for my $accessor (@$models) {
            my @filter_parts;
            for my $check (@$check_fields) {
                next if $check->{model} && $check->{model} ne $accessor;
                if ($app->$accessor->get_model_fields->{$check->{need_field}}) {
                    push @filter_parts, [OR => [$check->{filter}, [$check->{need_field} => '=' => \undef]]];
                }
            }
            if (@filter_parts) {
                my $db_table = $app->$accessor->partner_db_table();

                if ($db_table->have_fields('model')) {
                    push(@filter_parts, ['model', '=', \$accessor]);
                }

                my $query = $app->partner_db->query->select(
                    table  => $db_table,
                    fields => {
                        'page_id' => $app->$accessor->get_page_id_field_name(),
                        'id'      => 'id'
                    },
                    filter => [AND => \@filter_parts],
                );

                $query->fields_order('page_id', 'id');

                push(@filter_block, [{'' => [qw(page_id id)]}, 'NOT IN', $query]);
            }
        }
    }

    my @filter_page;
    if (my $skip_model = $filter{skip_page_model}) {
        push @filter_page, [model => 'NOT IN' => \$skip_model];
    }

    $data->{filter} = {
        filter_all_pages  => \@filter_page,
        filter_all_blocks => \@filter_block,
        join_all_blocks   => [],
    };
}

sub clear_filters {
    my ($self, $app, $data) = @_;

    delete $data->{filter};
}

sub apply_real_process {
    my ($self, $app, $data) = @_;
    return {};
}

sub apply_pre_process {
    my ($self, $app, $data) = @_;
}

sub apply_post_process {
    my ($self, $app, $data, $input, $result) = @_;
}

sub apply {
    my ($self, $app, $data) = @_;

    my %args = hash_transform($data, $self->get_data_fields);
    INFOF("APPLY_START %s ticket: %s", $self->method, $self->get_owner_data($data)->{ticket});

    $self->set_filters($app, $data);

    my $result;
    my $error;
    try {
        $self->apply_pre_process($app, $data);
        $result = $self->apply_real_process($app, $data);
    }
    catch {
        my ($e) = @_;
        $error = $args{error} = $e->message;
        $args{error_full} = $e->as_string;
        INFOF("APPLY_ERROR %s '%s'", $self->method, $error);
        $self->post_ticket_message(
            app     => $app,
            data    => $data,
            message => $self->get_error_message,
            error   => $args{error}
        );
        $self->mail_message($app, $data, $self->get_error_message, {error => $args{error}});
    };

    if ($result) {
        INFOF("APPLY_DONE %s", $self->method);
        $self->apply_post_process($app, $data, \%args, $result);

        delete $result->{skip};
        my $errors = delete $result->{errors};
        my $resend = delete $result->{resend};
        my $blocks = delete $result->{blocks};

        $app->all_pages->mark_pages_for_async_update(page_ids => $resend) unless $data->{noresend};

        my $attachment = {
            rollback_data => $result,
            errors        => [map {ref $_ eq 'ARRAY' ? $_ : $_->message} @$errors],
        };

        if ($blocks) {
            $args{changed_block} = $attachment->{changed_block} = [sort @{array_uniq($blocks)}];
        }

        if ($resend) {
            $args{changed_pages} = $attachment->{changed_pages} = [sort {$a <=> $b} @$resend];
        }

        try {
            $self->post_ticket_message(
                app        => $app,
                data       => $data,
                message    => $self->get_done_message,
                input      => \%args,
                attachment => $attachment,
            );
        }
        catch {
            my ($e) = @_;
            $args{fail_on_post} = $e->message;
        };

        $self->mail_message($app, $data, $self->get_done_message, \%args, $result);
        INFOF("APPLY_END %s", $self->method);
    } else {
        throw $error;
    }

    return \%args;
}

sub get_complete_message {gettext('Task has added to queue and result will be send to ticket and author')}
sub get_error_message    {gettext('Errors occurred during execution task')}
sub get_start_message    {gettext("Add task")}
sub get_done_message     {gettext("Finish task")}

sub get_milestone_number {1}

sub create_comment {
    my ($self, %params) = @_;

    my $str = sprintf("%s\n", $params{message});

    if ($params{input}) {
        $str .= sprintf("<{input params\n%%%%(json)\n%s\n%%%%\n}>\n", to_json($params{input}, pretty => TRUE));
    }

    if ($params{error}) {
        $str .= sprintf("<{error\n%%%%(json)\n%s\n%%%%\n}>\n", to_json($params{error}, pretty => TRUE));
    }

    if (my $result = $params{result}) {
        if ($result->{pages}) {
            $str .= sprintf("<{pages\n%%%%\n%s\n%%%%\n}>\n", join("\n", @{$result->{pages}}));
        }
        if ($result->{blocks}) {
            $str .= sprintf("<{blocks\n%%%%\n%s\n%%%%\n}>\n", join("\n", @{$result->{blocks}}));
        }
    }

    return $str;
}

sub post_ticket_message {
    my ($self, %params) = @_;

    my $owner_data = $self->get_owner_data($params{data});

    return $params{app}->startrek->add_comment(
        $owner_data->{ticket},
        $self->create_comment(%params),
        $owner_data->{ticket_user},
        $params{attachment},
    );
}

sub mail_message {
    my ($self, $app, $data, $message, $input, $result) = @_;

    my $owner_data = $self->get_owner_data($data);

    $app->mail_notification->add(
        type    => 11,
        user_id => $owner_data->{current_user},
        opts    => {
            subject => $message,
            values  => {
                message_body       => to_json({input => $input, result => $result}, pretty => TRUE),
                plain_text_wrapper => TRUE,
            },
        },
    );
}

sub apply_to_list_gm {
    my ($self, %params) = @_;
    my ($list, $app, $fields, $sub, $comment_orig) = @params{qw(list app fields sub comment)};

    my $comment = {%$comment_orig};
    $comment->{comment} .= ' - bk_data';

    my %by_model;
    my %resend;
    my @errors;
    my @blocks;
    for my $block (@$list) {
        my $accessor = $block->{block_model};
        my $pn       = $app->$accessor->get_page_id_field_name;
        my $row =
          $app->$accessor->get($block->{public_id},
            fields => [$pn, 'id', 'bk_data', 'multistate', 'public_id', @$fields],);

        try {
            my $bk_data = from_json($row->{bk_data});
            my $new_data = $sub->(row => $row, bk_data => $bk_data);
            $app->partner_db->transaction(
                sub {
                    my $table = $app->$accessor->partner_db_table();
                    $table->edit($row, {bk_data => to_json($new_data, pretty => TRUE)});
                    $self->store_action_log(
                        app     => $app,
                        comment => $comment,
                        list    => {$accessor => [$row]},
                        as_pk   => TRUE,
                    );
                }
            );
            push @{$by_model{$accessor}},
              {
                public_id     => $block->{public_id},
                orig_bk_data  => $row->{bk_data},
                canonical_new => to_json($new_data, canonical => TRUE),
                canonical_old => to_json(from_json($row->{bk_data}), canonical => TRUE),
              };
            $resend{$row->{$pn}} = TRUE;
            push @blocks, $block->{public_id};
        }
        catch {
            my ($e) = @_;
            push @errors, $e;
        };
    }
    return {
        blocks           => \@blocks,
        bk_data_by_model => \%by_model,
        resend           => [sort keys %resend],
        errors           => \@errors,
    };
}

sub store_action_log {
    my ($self, %params) = @_;
    my ($app, $comment, $list_by_model, $as_pk) = @params{qw(app comment list as_pk)};

    for my $accessor (keys %$list_by_model) {
        $self->store_model_action_log($app->$accessor, $comment, $list_by_model->{$accessor}, $as_pk);
    }
}

sub store_model_action_log {
    my ($self, $model, $comment, $list, $as_pk) = @_;

    return unless $model->_action_log_db_table();

    my $pk  = $model->_multistate_db_table->primary_key;
    my @pk  = map {"elem_$_"} @$pk;
    my @npk = $as_pk ? @$pk : ('page_id', @pk == 2 ? 'block_id' : ());
    my $dt  = curdate(oformat => 'db_time');

    my @add;
    for my $row (@$list) {
        my %row = (
            old_multistate => $row->{multistate},
            new_multistate => $row->{multistate},
            action         => 'edit',
            dt             => $dt,
            %$comment,
        );
        @row{@pk} = @{$row}{@npk};
        push @add, \%row;
    }

    $model->_action_log_db_table()->add_multi(\@add);
}

sub split_blocks_by_state {
    my ($self, $app, $data, $list) = @_;

    my %by_model;
    for my $row (@$list) {
        push @{$by_model{$row->{block_model}}}, $row;
    }

    my %result = (
        godmode => {
            deleted => [],
            working => [],
        },
        normal => {
            deleted => [],
            working => [],
        },
    );
    for my $accessor (keys %by_model) {
        my $state_deleted = $app->$accessor->get_multistate_by_name('deleted');
        my %pid = map {$_->{public_id} => $_} @{$by_model{$accessor}};
        for my $row (
            @{
                $app->$accessor->get_all(
                    fields => ['public_id', 'is_custom_bk_data', 'multistate'],
                    filter => {public_id => [sort keys %pid]}
                )
            }
          )
        {
            my $is_deleted = $row->{multistate} & $state_deleted ? 'deleted' : 'working';
            my $is_gm      = $row->{is_custom_bk_data}           ? 'godmode' : 'normal';
            push @{$result{$is_gm}{$is_deleted}}, $pid{$row->{public_id}};
        }
    }

    my $need_deleted = Application::Model::Queue::Support::Apply::ApplyDeleted->get_value($data);
    my $need_godmode = Application::Model::Queue::Support::Apply::ApplyGM->get_value($data);
    if ($need_deleted && $need_godmode) {
        return {
            godmode  => [@{$result{godmode}{deleted}}, @{$result{godmode}{working}}],
            database => [@{$result{godmode}{working}}, @{$result{godmode}{deleted}}, @{$result{normal}{deleted}}],
            model => [@{$result{normal}{working}}],
        };
    } elsif ($need_godmode) {
        return {
            godmode  => [@{$result{godmode}{working}}],
            database => [],
            model    => [@{$result{normal}{working}}],
        };
    } elsif ($need_deleted) {
        return {
            godmode  => [],
            database => [@{$result{normal}{deleted}}],
            model    => [@{$result{normal}{working}}],
        };
    } else {
        return {
            godmode  => [],
            database => [],
            model    => [@{$result{normal}{working}}],
        };
    }
}

sub apply_block_list_by_state {
    my ($self, %opts) = @_;

    my ($app, $data, $list, $sub_godmode, $sub_database, $sub_model, $fields, $comment) =
      @opts{qw(app data list sub_godmode sub_database sub_model fields comment)};

    my $parts = $self->split_blocks_by_state($app, $data, $list);

    my $result = $self->apply_to_list_gm(
        comment => $comment,
        list    => $parts->{godmode},
        app     => $app,
        fields  => $fields,
        sub     => $sub_godmode
    );

    my @errors;
    $app->support->apply_by_parts(
        comment => $comment,
        list    => $parts->{database},
        max     => 100,
        main    => $sub_database,
        error   => sub {
            push @errors, $_[0];
        }
    );

    my %resend;
    for my $block (@{$parts->{model}}) {
        try {
            $sub_model->(
                app   => $app,
                block => $block,
                data  => $data
            );
            $resend{$block->{page_id}} = TRUE;
            push @{$result->{blocks}}, $block->{public_id};
        }
        catch {
            my ($e) = @_;
            push @errors, [$block->{public_id}, $e->message];
        };
    }

    push @{$result->{errors}}, @errors;
    push @{$result->{resend}}, keys %resend;
    return $result;
}

1;
