package Application::Model::RBAC;

use qbit;
use List::Util qw(any);

use base qw(QBit::Application::Model::RBAC::DB);

use Exception::Validation::BadArguments;

use PiConstants qw($YAN_PARTNER_ASSISTANT_ROLE_ID);

sub accessor {'rbac'}

__PACKAGE__->model_accessors(
    mail_notification => 'Application::Model::MailNotification',
    partner_db        => 'Application::Model::PartnerDB',
    user_features     => 'Application::Model::Users::Features',
    users             => 'Application::Model::Users',
);

use PiConstants qw($PARTNERS_INTERNAL_ACCESS_REQUESTS_MAIL :ROLES :ROLES_GROUPS);

# атрибут related_page_accessor обозначает модели, которая данная партнерская роль "открывает"
# значения related_page_accessor должны быть уникальными среди всех ролей и принадлежать только
# ролям - владельцам (owner)
my $ROLES = [
    {
        id          => $DEVELOPER_ROLE_ID,
        name        => d_gettext('Developer'),
        is_internal => TRUE,
        group       => $GROUP_DEV,
        idm         => 'DEVELOPER',
    },
    {
        id          => $INTERNAL_YAN_MANAGER_ROLE_ID,
        name        => d_gettext('Yandex services: Manager'),
        is_internal => TRUE,
        group       => $GROUP_YNDX_SERVICES,
        idm         => 'INTERNAL_YAN_MANAGER',
        idm_fields  => [
            {
                slug     => "page_id",
                name     => "Page ID",
                type     => "charfield",
                required => \1
            }
        ]
    },
    {
        id          => $INTERNAL_YAN_ADMINISTRATOR_ROLE_ID,
        name        => d_gettext('Yandex services: Administrator'),
        is_internal => TRUE,
        group       => $GROUP_YNDX_SERVICES,
        idm         => 'INTERNAL_YAN_ADMINISTRATOR',
    },
    {id => $DSP_PARTNER_ROLE_ID, name => d_gettext('DSP: Partner'), group => $GROUP_DSP},
    {
        id          => $DSP_MANAGER_ROLE_ID,
        name        => d_gettext('DSP: Manager'),
        is_internal => TRUE,
        group       => $GROUP_DSP,
        idm         => 'DSP_MANAGER',
    },
    {
        id                    => $SITE_PARTNER_ROLE_ID,
        name                  => d_gettext('YAN Sites: Partner'),
        related_page_accessor => [qw(context_on_site_campaign search_on_site_campaign)],
        group                 => $GROUP_YAN,
        assessor_applicable   => TRUE,
    },
    {
        id          => $YAN_MANAGER_ROLE_ID,
        name        => d_gettext('YAN: Manager'),
        is_internal => TRUE,
        group       => $GROUP_YAN,
        idm         => 'YAN_MANAGER',
    },
    {
        id          => $YAN_MODERATOR_ROLE_ID,
        name        => d_gettext('YAN: Moderator'),
        is_internal => TRUE,
        group       => $GROUP_YAN,
        idm         => 'YAN_MODERATOR',
    },
    {
        id                    => $YAN_VIEWER_ROLE_ID,
        name                  => d_gettext('YAN: Viewer'),
        is_internal           => TRUE,
        group                 => $GROUP_YAN,
        viewer                => TRUE,
        conflict_role_for_set => [$YAN_MANAGER_ROLE_ID, $YAN_MODERATOR_ROLE_ID],
        idm                   => 'YAN_VIEWER',
    },
    {
        id                    => $DSP_VIEWER_ROLE_ID,
        name                  => d_gettext('DSP: Viewer'),
        is_internal           => TRUE,
        group                 => $GROUP_DSP,
        viewer                => TRUE,
        conflict_role_for_set => [$DSP_MANAGER_ROLE_ID],
        idm                   => 'DSP_VIEWER',
    },
    {
        id                    => $INTERNAL_YAN_VIEWER_ROLE_ID,
        name                  => d_gettext('Yandex services: Viewer'),
        is_internal           => TRUE,
        group                 => $GROUP_YNDX_SERVICES,
        viewer                => TRUE,
        conflict_role_for_set => [$INTERNAL_YAN_MANAGER_ROLE_ID, $INTERNAL_YAN_ADMINISTRATOR_ROLE_ID],
        idm                   => 'INTERNAL_YAN_VIEWER',
    },
    {
        id                    => $VIDEO_PARTNER_ROLE_ID,
        name                  => d_gettext('YAN Video: Partner'),
        related_page_accessor => ['video_an_site'],
        group                 => $GROUP_YAN,
        assessor_applicable   => TRUE,
    },
    {
        id                    => $MOBILE_PARTNER_ROLE_ID,
        name                  => d_gettext('YAN Applications: Partner'),
        related_page_accessor => ['mobile_app_settings'],
        group                 => $GROUP_YAN,
        assessor_applicable   => TRUE,
    },
    {
        id                    => $BUSINESS_UNIT_ROLE_ID,
        name                  => d_gettext("Business unit"),
        group                 => $GROUP_YAN,
        required_role_for_set => [$SITE_PARTNER_ROLE_ID],
    },
    {
        id                    => $ADBLOCK_PARTNER_ROLE_ID,
        name                  => d_gettext('YAN Sites: AdBlock partner'),
        group                 => $GROUP_YAN,
        required_role_for_set => [$SITE_PARTNER_ROLE_ID],
    },
    {
        id                    => $YAN_PARTNER_ASSISTANT_ROLE_ID,
        name                  => d_gettext("YAN Sites: Partner's assistant"),
        group                 => $GROUP_YAN,
        conflict_role_for_set => [$YAN_MANAGER_ROLE_ID, $YAN_MODERATOR_ROLE_ID, $YAN_VIEWER_ROLE_ID],
        assessor_applicable   => TRUE,
    },
    {
        id    => $TUTBY_ROLE_ID,
        name  => d_gettext("Tutby aggregator"),
        group => $GROUP_TUTBY,
    },
    {
        id                    => $INDOOR_PARTNER_ROLE_ID,
        name                  => d_gettext('Indoor: Partner'),
        related_page_accessor => ['indoor'],
        group                 => $GROUP_YAN,
        assessor_applicable   => TRUE,
    },
    {
        id    => $ADFOX_ROLE_ID,
        name  => d_gettext('AdFox'),
        group => $GROUP_YAN
    },
    {
        id                    => $OUTDOOR_PARTNER_ROLE_ID,
        name                  => d_gettext('Outdoor: Partner'),
        related_page_accessor => ['outdoor'],
        group                 => $GROUP_YAN,
        assessor_applicable   => TRUE,
    },
    {
        id                    => $ROBOT_ASSISTANT_ROLE_ID,
        name                  => d_gettext('Robot assistant'),
        is_internal           => TRUE,
        group                 => $GROUP_ROBOTS,
        required_role_for_set => [$YAN_MANAGER_ROLE_ID, $INTERNAL_YAN_ADMINISTRATOR_ROLE_ID],
        idm                   => 'ROBOT_ASSISTANT',
    },
    {
        id           => $ASSESSOR_ROLE_ID,
        name         => d_gettext('Assessor'),
        all_external => TRUE,
        group        => $GROUP_YAN
    },
    {
        id          => $PROTECTED_PAGES_EDITOR_ROLE_ID,
        name        => d_gettext('Protected pages editor'),
        is_internal => TRUE,
        group       => $GROUP_DEV,
        idm         => 'PROTECTED_PAGES_EDITOR',
    },
    {
        id          => $DISTRIBUTION_MANAGER_ROLE_ID,
        name        => d_gettext('Distribution: manager'),
        is_internal => TRUE,
        group       => $GROUP_YAN,
        idm         => 'DISTRIBUTION_MANAGER',
    },
];

my %ROLES = map {$_->{'id'} => $_} @$ROLES;

foreach my $role (@$ROLES) {
    if (exists($role->{'required_role_for_set'})) {
        my @unknown_roles = grep {!$ROLES{$_}} @{$role->{'required_role_for_set'}};

        throw ngettext(
            'Unknown role with id %s',
            'Unknown roles with id %s',
            scalar(@unknown_roles), join(', ', map {"\"$_\""} @unknown_roles)
        ) if @unknown_roles;
    }
}

=head1 check_role_id

Throws exeption Exception::Validation::BadArguments if role_id is incorrect.

Returns nothing interesting.

=cut

sub check_role_id {
    my ($self, $role_id) = @_;

    throw Exception::Validation::BadArguments(gettext('No role_id'))
      unless $role_id;

    my $roles = [map {$_->{'id'}} @{$self->get_roles}];

    throw Exception::Validation::BadArguments(gettext('Such a role with ID %s does not exist', $role_id))
      unless in_array($role_id, $roles);
}

=head2 cur_user_is_internal

=cut

sub cur_user_is_internal {
    my ($self) = @_;

    my $cur_user_roles = $self->get_cur_user_roles();

    my $cur_user_is_internal = !!grep {$cur_user_roles->{$_}{'is_internal'}} keys(%$cur_user_roles);

    return $cur_user_is_internal;
}

sub get_role {
    my ($self, $role_id) = @_;
    my $role = clone($ROLES{$role_id});
    $role->{name} = $role->{name}() if $role;
    return $role;
}

sub get_roles {
    my ($self, %opts) = @_;

    my $roles;

    if ($opts{'ids'}) {
        my %hash_ids = map {$_ => TRUE} @{$opts{'ids'}};
        $roles = clone([grep {$hash_ids{$_->{'id'}}} @$ROLES]);
    } else {
        $roles = clone($ROLES);
    }

    foreach my $filter (qw(assessor_applicable idm is_internal)) {
        if ($opts{$filter}) {
            $roles = [grep {$_->{$filter}} @$roles];
        }
    }

    if ($opts{id_only}) {
        $roles = [map {$_->{id}} @$roles];
    } else {
        $_->{name} = $_->{name}() foreach @$roles;    # gettext call is rather heavy
    }

    # PI-25944 Костыль, чтобы сохранить типизацию, на которую фронт завязан (можно убрать после выпиливания rosetta)
    if ($opts{id_only}) {
        $_ .= '' foreach (@$roles);
    } else {
        foreach my $role (@$roles) {
            foreach (qw(assessor_applicable group)) {
                $role->{$_} += 0 if (defined $role->{$_});
            }
            $role->{id} .= '';
        }
    }

    return $roles;
}

=head1 has_role_from_group_dev

=cut

sub has_role_from_group_dev {
    return $_[0]->_has_role_from_group($_[1], $GROUP_DEV);
}

=head1 has_role_from_group_dsp

=cut

sub has_role_from_group_dsp {
    return $_[0]->_has_role_from_group($_[1], $GROUP_DSP);
}

=head1 has_role_from_group_yan

=cut

sub has_role_from_group_yan {
    return $_[0]->_has_role_from_group($_[1], $GROUP_YAN);
}

=head1 has_role_from_group_yndx_services

=cut

sub has_role_from_group_yndx_services {
    return $_[0]->_has_role_from_group($_[1], $GROUP_YNDX_SERVICES);
}

sub revoke_roles {
    my ($self, $user_id, $roles, %opts) = @_;

    $roles = [$roles] unless ref($roles) eq 'ARRAY';

    my $cur_roles = $self->get_roles_by_user_id($user_id);

    my %del_roles = map {$_ => undef} @$roles;
    my %role_with_required;
    foreach (keys(%$cur_roles)) {
        if (exists($cur_roles->{$_}{'required_role_for_set'})) {
            $role_with_required{$_} = $cur_roles->{$_}{'required_role_for_set'};
        }
    }

    foreach my $role (keys(%role_with_required)) {
        my $required_role = $role_with_required{$role};

        my @not_exists_roles = grep {exists($del_roles{$_}) || !exists($cur_roles->{$_})} @$required_role;

        if (@not_exists_roles == @$required_role && !exists($del_roles{$role})) {
            my @cant_remove_roles = grep {exists($del_roles{$_})} @$required_role;

            throw Exception::Validation::BadArguments ngettext(
                'Do not remove role %s, because role "%s" depends for it',
                'Do not remove roles %s, because role "%s" depends for it',
                scalar(@cant_remove_roles),
                join(', ', map {"\"$cur_roles->{$_}{'name'}\""} @cant_remove_roles),
                $cur_roles->{$role}{'name'}
            );
        }
    }

    if (!$opts{allow_idm} and my @idm_roles = @{$self->get_roles(ids => $roles, idm => TRUE)}) {
        throw Exception::Validation::BadArguments(
            gettext('Roles: %s are under IDM', join(", ", map {$_->{name}} @idm_roles)),
        );
    }

    $self->SUPER::revoke_roles($user_id, $roles);
}

sub get_partner_owner_roles {
    my ($self) = @_;

    my @roles;
    foreach my $role (@{$self->get_roles()}) {
        push @roles, $role if exists $role->{'related_page_accessor'} or $role->{'all_external'};
    }

    return \@roles;
}

sub get_accessors_for_partner_owner_roles {
    my ($self) = @_;

    my %accessors;
    foreach my $role (@{$self->get_roles()}) {
        if (exists($role->{'related_page_accessor'})) {
            $accessors{$_} = $role foreach @{$role->{'related_page_accessor'}};
        }
    }

    return \%accessors;
}

sub set_user_role {
    my ($self, $user_id, $roles, %opts) = @_;

    $roles = [$roles] unless ref($roles) eq 'ARRAY';

    $self->check_role_id($_) foreach @$roles;

    my $cur_roles = $self->get_roles_by_user_id($user_id);
    my %all_roles = map {$_->{'id'} => $_} @{$self->get_roles()};

    # filter out all existing roles
    $roles = [grep {!exists($cur_roles->{$_})} @$roles];
    unless (@$roles) {
        throw Exception::Validation::BadArguments(
            gettext(
                'User "%s" already has roles',
                $self->users->get($user_id, fields => ['login'])->{'login'} // $user_id
            ),
            idm_warning => TRUE
        );
    }

    my @roles_with_conflict;
    my @roles_with_required;
    my @ordinary_roles;
    my @idm_roles;
    foreach (@$roles) {
        if (exists($all_roles{$_}->{'required_role_for_set'})) {
            push @roles_with_required, $_;
        } elsif (exists($all_roles{$_}->{'conflict_role_for_set'})) {
            push @roles_with_conflict, $_;
        } else {
            push @ordinary_roles, $_;
        }
        push @idm_roles, $_ if $all_roles{$_}{idm};
    }
    if (@idm_roles and !$opts{allow_idm}) {
        throw Exception::Validation::BadArguments(
            gettext('Roles: %s are under IDM', join(", ", map {$all_roles{$_}{name}} @idm_roles)),
        );
    }

    foreach my $role_id (@ordinary_roles, @roles_with_conflict, @roles_with_required) {
        my @incompatible_roles = ();

        foreach my $cur_role_id (keys(%$cur_roles)) {
            push(@incompatible_roles, $cur_role_id)
              if (($all_roles{$role_id}->{'viewer'} || $all_roles{$cur_role_id}->{'viewer'})
                && $all_roles{$role_id}->{'group'} == $all_roles{$cur_role_id}->{'group'})
              || ($all_roles{$role_id}->{'viewer'} && !$all_roles{$cur_role_id}->{'is_internal'})
              || ($all_roles{$cur_role_id}->{'viewer'} && !$all_roles{$role_id}->{'is_internal'});

            push(@incompatible_roles, $cur_role_id)
              if !!($all_roles{$role_id}->{'is_internal'}) != !!($all_roles{$cur_role_id}->{'is_internal'});

            if (
                (
                    exists($all_roles{$cur_role_id}->{'conflict_role_for_set'})
                    && in_array($role_id, $all_roles{$cur_role_id}->{'conflict_role_for_set'})
                )
                || (exists($all_roles{$role_id}->{'conflict_role_for_set'})
                    && in_array($cur_role_id, $all_roles{$role_id}->{'conflict_role_for_set'}))
               )
            {
                push(@incompatible_roles, $cur_role_id);
            }
        }

        throw Exception::Validation::BadArguments(
            gettext(
                'Role "%s" incompatible with roles: %s',
                $all_roles{$role_id}->{'name'},
                join(', ', map {"$all_roles{$_}->{'name'}"} grep {$cur_roles->{$_}} @incompatible_roles)
            )
        ) if @incompatible_roles;

        if (exists($all_roles{$role_id}->{'required_role_for_set'})) {
            my @not_exists_roles = grep {!exists($cur_roles->{$_})} @{$all_roles{$role_id}->{'required_role_for_set'}};

            throw Exception::Validation::BadArguments gettext(
                "Can't add role '%s', because the role depends on %s",
                $all_roles{$role_id}->{'name'},
                join(', ', map {"\"$all_roles{$_}->{'name'}\""} @not_exists_roles)
            ) if @not_exists_roles;
        }

        # add to cur_roles verified role
        $cur_roles->{$role_id} = $all_roles{$role_id};
    }

    $self->partner_db->transaction(
        sub {
            $self->SUPER::set_user_role($user_id, $roles);

            $self->user_features->auto_assign($user_id, $roles);
        }
    );

    # send only one letter
    if (any {$all_roles{$_}->{'is_internal'}} @$roles) {
        my $login = $self->users->get($user_id, fields => ['login'])->{'login'} // $user_id;

        $self->mail_notification->add_when_access_granted(
            {
                login   => $login,
                user_id => $user_id,
            }
        );
    }
}

=begin comment _get_role_ids_from_group

    my $role_ids = $self->_get_role_ids_from_group( YAN );

=end comment

=cut

sub _get_role_ids_from_group {
    my ($self, $group_id) = @_;

    my @role_ids = map {$_->{'id'}} grep {$_->{'group'} eq $group_id} @{$self->get_roles()};

    return \@role_ids;
}

=begin comment _has_at_least_one_role_id

    my $bool = $app->rbac->_has_at_least_one_role_id( 9, 15, 16, 17 );

=end comment

=cut

sub _has_at_least_one_role_id {
    my ($self, $role_ids, $group_role_ids) = @_;

    return FALSE if !defined($role_ids) || !@$role_ids;

    my %roles = map {$_ => TRUE} @$role_ids;

    foreach my $role_id (@$group_role_ids) {
        return TRUE if $roles{$role_id};
    }

    return FALSE;
}

=begin comment _has_role_from_group

=end comment

=cut

sub _has_role_from_group {
    my ($self, $role_ids, $group_id) = @_;

    return $self->_has_at_least_one_role_id($role_ids, $self->_get_role_ids_from_group($group_id));
}

TRUE;
